<?php

namespace ImageHopper\ImageHopper\Fields;

use ImageHopper\ImageHopper\API\ImageManager;
use ImageHopper\ImageHopper\Helpers\FileHelper;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * Class ImageHopperField
 *
 * @package ImageHopper\ImageHopper\Fields
 *
 * @since 1.0.0
 */
class ImageHopperField extends \GF_Field_FileUpload {

	/**
	 * @var string
	 *
	 * @since 1.0.0
	 */
	public $type = 'image_hopper';

	/**
	 * @var bool
	 *
	 * @since 1.0.0
	 */
	public $multipleFiles = true;

	/**
	 * @var string
	 *
	 * @since 1.0.0
	 */
	public $allowedExtensions;

	public function __construct( $data = [] ) {

		global $wp_version;

		parent::__construct( $data );

		/* ensure all fields have the same allowed extensions */
		$this->allowedExtensions = 'jpg, jpeg, gif, png, bmp';

		/* Add webp support */
		if ( version_compare( $wp_version, '5.8', '>=' ) ) {
			$this->allowedExtensions .= ', webp';
		}

		/* Add avif support */
		if ( version_compare( $wp_version, '6.5-RC1', '>=' ) ) {
			$this->allowedExtensions .= ', avif';
		}
	}

	/**
	 * Return the field title, for use in the form editor.
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_form_editor_field_title() {
		return esc_attr__( 'Image Hopper', 'image-hopper' );
	}

	/**
	 * Assign the field button to the Advanced Fields group.
	 *
	 * @return array
	 *
	 * since 1.0.0
	 */
	public function get_form_editor_button() {
		return [
			'group' => 'advanced_fields',
			'text'  => $this->get_form_editor_field_title(),
		];
	}

	/**
	 * Pretend the field is a `fileupload` under specific circumstances, otherwise default to `image_hopper`
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_input_type() {
		if (
			$this->is_fileupload_page() ||
			$this->is_form_submission() ||
			$this->is_entry_detail() ||
			$this->is_heartbeat_request()
		) {
			return apply_filters( 'image_hopper_field_input_type', 'fileupload', $this );
		}

		$type = empty( $this->inputType ) ? $this->type : $this->inputType;

		return apply_filters( 'image_hopper_field_input_type', $type, $this );
	}

	/**
	 * Returns the field's form editor icon.
	 *
	 * This could be an icon url or a gform-icon class.
	 *
	 * @since 1.4
	 *
	 * @return string
	 */
	public function get_form_editor_field_icon() {
		return 'gform-icon--post-image';
	}

	/**
	 * Check if the form is currently being submitted
	 *
	 * @return bool
	 *
	 * @since 1.0.0
	 */
	protected function is_form_submission() {
		/* phpcs:disable WordPress.Security.NonceVerification.Missing */
		return isset( $_POST['gform_submit'] ) ||
			   isset( $_POST['gforms_save_entry'] ) ||
			   doing_action( 'gform_entry_post_save' );
		/* phpcs:enable */
	}

	/**
	 * Check if user is currently uploading a file during this request
	 *
	 * @return bool
	 *
	 * @since 1.0.0
	 */
	protected function is_fileupload_page() {
		/* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
		if ( rgempty( 'gf_page', $_GET ) ) {
			return false;
		}

		$is_upload_page = $_SERVER['REQUEST_METHOD'] === 'POST' && rgget( 'gf_page' ) === \GFCommon::get_upload_page_slug();
		if ( ! $is_upload_page ) {
			return false;
		}

		return true;
	}

	/**
	 * Check if currently doing a heartbeat request (Partial Entry)
	 *
	 * @return bool
	 *
	 * @since 2.8.2
	 */
	protected function is_heartbeat_request() {
		return defined( 'DOING_AJAX' ) && DOING_AJAX && rgpost( 'action' ) === 'heartbeat';
	}

	/**
	 * The settings to include in the form editor.
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function get_form_editor_field_settings() {
		return apply_filters(
			'image_hopper_enabled_field_settings',
			[
				'conditional_logic_field_setting',
				'error_message_setting',
				'label_setting',
				'label_placement_setting',
				'admin_label_setting',
				'rules_setting',
				'file_size_setting',
				'visibility_setting',
				'description_setting',
				'css_class_setting',
				'image_hopper_max_number_files',
				'image_hopper_max_image_sizes',
				'image_hopper_output_quality',
				'image_hopper_minimum_image_size',
				'image_hopper_minimum_image_size_warning',
				'image_hopper_upscale_field',
				'image_hopper_rename_image_field',
			],
			$this
		);
	}

	/**
	 * The scripts to be included in the form editor.
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_form_editor_inline_script_on_page_render() {
		return apply_filters(
			'image_hopper_inline_javascript',
			sprintf(
					/* phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents */
				file_get_contents( __DIR__ . '/../../js/admin/formEditorInlineScript.js' ),
				$this->get_form_editor_field_title(),
				$this->allowedExtensions,
				esc_html__( 'Convert', 'image-hopper' ),
				esc_attr__( 'Convert to an Image Hopper field', 'image-hopper' ),
				esc_attr__( 'Convert to a File Upload field', 'image-hopper' )
			)
		);
	}

	/**
	 * Set the correct "for" ID for the label
	 *
	 * @param array $form
	 *
	 * @return string
	 *
	 * @since 2.10.3
	 */
	public function get_first_input_id( $form ) {
		return sprintf( 'input_%d_%d', $form['id'], $this->id );
	}

	/**
	 * Define the fields inner markup.
	 *
	 * @param array        $form  The Form Object currently being processed.
	 * @param string|array $value The field value. From default/dynamic population, $_POST, or a resumed incomplete submission.
	 * @param null|array   $entry Null or the Entry Object currently being edited.
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_field_input( $form, $value = '', $entry = null ) {
		$input = $this->build_field_input( $form, $entry, $value );

		return sprintf( "<div class='ginput_container ginput_container_%s gform-theme__no-reset--children'>%s</div>", $this->type, $input );
	}

	/**
	 * Gets the Image Hopper file input field
	 *
	 * @param array        $form  The Form Object currently being processed.
	 * @param null|array   $entry Null or the Entry Object currently being edited.
	 * @param string|array $value The field value. From default/dynamic population, $_POST, or a resumed incomplete submission.
	 *
	 * @return string
	 *
	 * @since
	 */
	protected function build_field_input( $form, $entry, $value ) {
		$id              = (int) $this->id;
		$form_id         = (int) $form['id'];
		$is_entry_detail = $this->is_entry_detail();
		/* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
		$is_gb_editor = \defined( 'REST_REQUEST' ) && REST_REQUEST && ! empty( $_REQUEST['context'] ) && 'edit' === $_REQUEST['context'];

		if ( $this->is_form_editor() || $is_gb_editor ) {
			return $this->editor_field_input();
		}

		// Prepare the value of the input ID attribute.
		$field_id = $is_entry_detail || $form_id === 0 ? "input_$id" : "input_{$form_id}_{$id}";

		// Get the value of the inputClass property for the current field.
		$inputClass      = $this->inputClass;
		$inputWidth      = ! empty( $this->inputWidth ) ? "data-image-resize-target-width='" . esc_attr( $this->inputWidth ) . "'" : '';
		$inputHeight     = ! empty( $this->inputHeight ) ? "data-image-resize-target-height='" . esc_attr( $this->inputHeight ) . "'" : '';
		$maxFiles        = ! empty( $this->maxFiles ) ? "data-max-files='" . esc_attr( $this->maxFiles ) . "'" : '';
		$multiples       = $this->maxFiles > 1 ? 'multiple' : '';
		$imageCrop       = $this->cropToSize ? 'true' : 'false';
		$cropAspectRatio = $this->cropToSize ? 'data-image-crop-aspect-ratio="' . esc_attr( $this->inputWidth . ':' . $this->inputHeight ) . '"' : '';
		$upscaleImage    = $this->cropToSize && $this->upscaleImageToCropSize ? 'true' : 'false';

		// Prepare the input classes.
		$class_suffix = $is_entry_detail ? '_admin' : '';
		$class        = esc_attr( $class_suffix . ' ' . $inputClass );

		// Prepare the other input attributes.
		$tabindex              = $this->get_tabindex();
		$placeholder_attribute = $this->get_field_placeholder_attribute();
		$required_attribute    = $this->isRequired ? 'aria-required="true"' : '';
		$invalid_attribute     = $this->failed_validation ? 'aria-invalid="true"' : 'aria-invalid="false"';
		$nonce                 = wp_create_nonce( 'gform_file_upload_' . $form_id );

		/* Ignore non-json encoded strings */
		$files = $value;
		if ( json_decode( $files, true ) === null ) {
			$files = ! empty( $files ) ? wp_json_encode( [ $files ] ) : '';
		}

		$is_admin        = $this->is_entry_detail() || $this->is_form_editor();
		$max_upload_size = ! $is_admin && $this->maxFileSize > 0 && is_numeric( $this->maxFileSize ) ? $this->maxFileSize : wp_max_upload_size() / 1048576;
		$max_upload_size = floor( $max_upload_size );

		$output_quality = ! empty( $this->outputQuality ) ? (int) $this->outputQuality : 90;
		/* Ensure the quality is in the range */
		if ( $output_quality > 100 ) {
			$output_quality = 100;
		} elseif ( $output_quality < 1 ) {
			$output_quality = 1;
		}

		/* If Output Quantity is empty, don't force unnecessary transformations */
		$image_transform_output_quantity = 'always';
		if ( empty( $this->outputQuality ) ) {
			$image_transform_output_quantity = 'optional';
		}

		$minimum_image_width  = ! empty( $this->inputMinImageSizeWidth ) ? (int) $this->inputMinImageSizeWidth : 1;
		$minimum_image_height = ! empty( $this->inputMinImageSizeHeight ) ? (int) $this->inputMinImageSizeHeight : 1;

		$minimum_image_warning        = (bool) $this->inputMinImageSizeWarning;
		$minimum_image_width_warning  = ! empty( $this->inputMinImageSizeWarningWidth ) ? (int) $this->inputMinImageSizeWarningWidth : '';
		$minimum_image_height_warning = ! empty( $this->inputMinImageSizeWarningHeight ) ? (int) $this->inputMinImageSizeWarningHeight : '';

		// Prepare the input tag for this field.
		$input = '<input
		name="' . esc_attr( 'input_' . $id ) . '"
		id="' . esc_attr( $field_id ) . '"
		type="file"
		class="' . esc_attr( $class ) . '"
		' . $tabindex . '
		' . $placeholder_attribute . '
		' . $required_attribute . '
		' . $invalid_attribute . '
		' . $inputHeight . '
		' . $inputWidth . '
		' . $maxFiles . '
		' . $multiples . '
		' . $cropAspectRatio . '	
		data-max-file-size="' . esc_attr( $max_upload_size . 'MB' ) . '"
		data-image-transform-output-quality="' . esc_attr( $output_quality ) . '"
		data-allow-image-crop="' . esc_attr( $imageCrop ) . '"
		data-image-resize-upscale="' . esc_attr( $upscaleImage ) . '"
		data-image-validate-size-min-width="' . esc_attr( $minimum_image_width ) . '"
		data-image-validate-size-min-height="' . esc_attr( $minimum_image_height ) . '"
		data-image-transform-output-quality-mode="' . esc_attr( $image_transform_output_quantity ) . '"
		data-form-id="' . esc_attr( $form_id ) . '"
		data-field-id="' . esc_attr( $id ) . '"
		data-files="' . esc_attr( $files ) . '"
		data-nonce="' . esc_attr( $nonce ) . '"
		data-min-image-warning="' . esc_attr( $minimum_image_warning ) . '"
		data-min-image-warning-width="' . esc_attr( $minimum_image_width_warning ) . '"
		data-min-image-warning-height="' . esc_attr( $minimum_image_height_warning ) . '"
		/>';

		return apply_filters( 'image_hopper_input_field', $input, $form, $entry, $value, $this );
	}

	/**
	 * Make the Form Editor preview look nice
	 *
	 * @return string
	 *
	 * @since 1.5
	 */
	public function editor_field_input() {
		$is_theme_framework = false;

		/* Check if rendering the form via the REST API (form block rendered in the post editor) */
		if ( defined( 'REST_REQUEST' ) && REST_REQUEST && rgget( 'context' ) === 'edit' ) {
			/* phpcs:ignore WordPress.Security.NonceVerification.Recommended */
			$is_theme_framework = ! empty( $_GET['attributes']['theme'] ) && ! in_array( $_GET['attributes']['theme'], [ 'gravity-theme', 'gravity' ], true );
		} elseif ( version_compare( \GFForms::$version, '2.9.0', '>=' ) ) {
			/* Always on in the form editor from GF2.9+ */
			$is_theme_framework = true;
		}

		ob_start();

		?>
		<div class="filepond--root filepond--hopper">
			<div class="filepond--drop-label <?php echo $is_theme_framework ? 'gform-theme-field-control' : ''; ?>">
				<label>
					<?php echo esc_html__( 'Drop files here or', 'image-hopper' ); ?>

					<?php echo $is_theme_framework ? '<br>' : ''; ?>

					<span class="filepond--label-action <?php echo $is_theme_framework ? 'button gform-theme-button gform-theme-button--control' : ''; ?>"><?php echo esc_html__( 'Select files', 'image-hopper' ); ?></span>
				</label>
			</div>

			<div class="filepond--panel filepond--panel-root"></div>
		</div>
		<?php

		return ob_get_clean();
	}

	/**
	 * Filter out the previously uploaded images when handling form submission
	 *
	 * @param string $value
	 * @param array  $form
	 * @param string $input_name
	 * @param int    $lead_id
	 * @param array  $lead
	 *
	 * @return array|string
	 *
	 * @since 1.0.0
	 */
	public function get_value_save_entry( $value, $form, $input_name, $lead_id, $lead ) {
		global $_gf_uploaded_files;

		/* Use the cached version if it already exists for this input type */
		if ( isset( $_gf_uploaded_files[ $input_name ] ) ) {
			return $_gf_uploaded_files[ $input_name ];
		}

		$lead  = \GFFormsModel::get_lead( $lead_id );
		$value = rgar( $lead, (string) $this->id, '' );
		$value = json_decode( $value, true ) !== null ? $value : '';

		$value = apply_filters( 'image_hopper_pre_get_value_save_entry', $value, $form, $input_name, $lead_id, $lead, $this );

		$existing_uploaded_files = [];
		if ( isset( \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] ) ) {
			list( $newly_uploaded_files, $existing_uploaded_files ) = $this->group_uploaded_files( $form, $lead, $input_name );

			\GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] = $newly_uploaded_files;

			/*
			 * If there is no value at this point and we have existing files available, store them in $value
			 * This can occur when prepopulating the Image Hopper field when the form is first loaded
			 */
			if ( empty( $value ) && count( $existing_uploaded_files ) > 0 ) {
				$value = wp_json_encode( $existing_uploaded_files, JSON_UNESCAPED_UNICODE );
			}
		}

		/* Ensure we only attempt to delete the removed images once */
		$value = $this->maybe_delete_stored_files( $value, $existing_uploaded_files, $lead_id );
		$value = $this->get_multifile_value( $form['id'], $input_name, $value );
		$value = $this->reorder_files( $value, $form['id'], $input_name );

		$value = apply_filters( 'image_hopper_post_get_value_save_entry', $value, $form, $input_name, $lead_id, $lead, $this );

		/* Cache the file data in the correct order */
		$_gf_uploaded_files[ $input_name ] = $value;

		return $value;
	}

	/**
	 * Check if the submission is empty or not
	 *
	 * @param int $form_id
	 *
	 * @return bool
	 *
	 * @since 2.7
	 */
	public function is_value_submission_empty( $form_id ) {
		$input_name = 'input_' . $this->id;
		$tmp_path   = \GFFormsModel::get_upload_path( $form_id ) . '/tmp/';

			$uploaded_files = \GFFormsModel::$uploaded_files[ $form_id ];
			$file_info      = rgar( $uploaded_files, $input_name );

		if ( empty( $file_info ) ) {
			return true;
		}

		foreach ( $file_info as $key => $file ) {
			$exists = false;
			if ( ! empty( $file['temp_filename'] ) ) {
				$tmp_file = $tmp_path . wp_basename( $file['temp_filename'] );
				if ( file_exists( $tmp_file ) ) {
					$exists = true;
				}
			} elseif ( ! empty( $file['uploaded_filename'] ) ) {
				/* validation on existing image URLs is done during saving */
				$exists = true;
			}

			if ( ! $exists ) {
				\GFCommon::log_debug( __METHOD__ . "(): Removing invalid file for {$input_name} key {$key}." );
				unset( \GFFormsModel::$uploaded_files[ $form_id ][ $input_name ][ $key ] );
			}
		}

			return empty( \GFFormsModel::$uploaded_files[ $form_id ][ $input_name ] );
	}

	/**
	 * Check if value is not stored as JSON data and encode it correctly.
	 * This handles cases where a single File Upload field is converted to Image Hopper
	 *
	 * @param string $value
	 * @param array $entry
	 * @param int $field_id
	 * @param array $columns
	 * @param array $form
	 *
	 * @return string
	 *
	 * @since 2.13
	 */
	public function get_value_entry_list( $value, $entry, $field_id, $columns, $form ) {
		if ( ! empty( $value ) && json_decode( $value ) === null ) {
			$value = wp_json_encode( [ $value ] );
		}

		return parent::get_value_entry_list( $value, $entry, $field_id, $columns, $form );
	}

	/**
	 * Handle the output on the Entry Details screen
	 *
	 * @param array|string $value
	 * @param string       $currency
	 * @param false        $use_text
	 * @param string       $format
	 * @param string       $media
	 *
	 * @return array|string
	 *
	 * @since 1.0.0
	 */
	public function get_value_entry_detail( $value, $currency = '', $use_text = false, $format = 'html', $media = 'screen' ) {
		if ( empty( $value ) ) {
			return '';
		}

		$file_urls = json_decode( $value, true );
		/* Handle cases where a single File Upload field is converted to Image Hopper */
		if ( $file_urls === null ) {
			$file_urls = [ $value ];
		}

		$output_arr      = [];
		$file_urls_count = count( $file_urls );
		$force_download  = in_array( 'download', $this->get_modifiers(), true );

		$output_arr[] = $format === 'html' ? '<div class="ih-field-container ' . esc_attr( 'ih-img-count-' . $file_urls_count ) . '">' : '';

		foreach ( $file_urls as $url ) {
			$secure_url = $this->get_download_url( $url, $force_download );

			/**
			 * Allow for override of SSL replacement
			 *
			 * By default Gravity Forms will attempt to determine if the schema of the URL should be overwritten for SSL.
			 * This is not ideal for all situations, particularly domain mapping. Setting $field_ssl to false will prevent
			 * the override.
			 *
			 * @param bool                $field_ssl True to allow override if needed or false if not.
			 * @param string              $secure_url The file path of the download file.
			 * @param GF_Field_FileUpload $field     The field object for further context.
			 */
			$field_ssl = apply_filters( 'gform_secure_file_download_is_https', true, $secure_url, $this );

			if ( \GFCommon::is_ssl() && strpos( $secure_url, 'http:' ) !== false && $field_ssl === true ) {
				$secure_url = str_replace( 'http:', 'https:', $secure_url );
			}

			/**
			 * Allows for the filtering of the file path before output.
			 *
			 * @param string              $secure_url The file path of the download file.
			 * @param GF_Field_FileUpload $field     The field object for further context.
			 *
			 * @since 2.1.1.23
			 *
			 */
			$secure_url = str_replace( '&amp;', '&', $secure_url );
			$secure_url = str_replace( ' ', '%20', apply_filters( 'gform_fileupload_entry_value_file_path', $secure_url, $this ) );

			$extension = pathinfo( wp_parse_url( $url, PHP_URL_PATH ), PATHINFO_EXTENSION );
			if ( in_array( strtolower( $extension ), [ 'jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'avif' ], true ) ) {
				if ( isset( $this->current_entry['id'] ) ) {
					list( $width, $height ) = $this->get_image_dimensions( $url, $this->current_entry['id'] );
				}

				$output_arr[] = $format === 'text' ?
					esc_url( $secure_url ) :
					sprintf(
						'<figure class="ih-field-el ih-field-el-img"><a href="%1$s" target="_blank" aria-label="%2$s"><img src="%1$s" style="max-width: 100%%; height: auto;" width="%3$s" height="%4$s" /></a></figure>',
						esc_url( $secure_url ),
						esc_attr__( 'Click to view', 'image-hopper' ),
						esc_attr( isset( $width ) ? $width : '' ),
						esc_attr( isset( $height ) ? $height : '' )
					);
			} else {
				/* Add support for non-images (eg. if a File Upload field is converted to an Image Hopper field) */
				$output_arr[] = $format === 'text' ?
					esc_url( $secure_url ) :
					sprintf(
						'<figure class="ih-field-el ih-field-el-other"><a href="%1$s" target="_blank" aria-label="%2$s">%3$s</a></figure>',
						esc_attr( $url ),
						esc_attr__( 'Click to view', 'image-hopper' ),
						wp_basename( $url )
					);
			}
		}

		$output_arr[] = $format === 'html' ? '</div>' : '';

		return implode( PHP_EOL, $output_arr );
	}

	/**
	 * Handle the merge tag output
	 *
	 * @param array|string $value
	 * @param string       $input_id
	 * @param array        $entry
	 * @param array        $form
	 * @param string       $modifier
	 * @param array|string $raw_value
	 * @param bool         $url_encode
	 * @param bool         $esc_html
	 * @param string       $format
	 * @param bool         $nl2br
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_value_merge_tag( $value, $input_id, $entry, $form, $modifier, $raw_value, $url_encode, $esc_html, $format, $nl2br ) {

		$show_urls    = in_array( 'url', $this->get_modifiers(), true );
		$use_raw_urls = in_array( 'raw', $this->get_modifiers(), true );

		$files = empty( $raw_value ) ? [] : json_decode( $raw_value, true );
		/* handle converted Single File Upload fields */
		if ( $files === null ) {
			$files = [ $raw_value ];
		}

		$get_image_count = in_array( 'count', $this->get_modifiers(), true );
		if ( $get_image_count ) {
			return is_array( $files ) ? count( $files ) : 0;
		}

		foreach ( $files as &$file ) {
			$original_url = $file;
			$extension    = pathinfo( wp_parse_url( $file, PHP_URL_PATH ), PATHINFO_EXTENSION );
			$is_image     = in_array( strtolower( $extension ), [ 'jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'avif' ], true );
			$filename     = wp_basename( $file );

			$file = ! $use_raw_urls ? $this->get_download_url( $file ) : $file;
			$file = str_replace( ' ', '%20', $file );
			if ( ! $show_urls && ! $use_raw_urls ) {
				/* files may not be images, so only output if appropriate extension is detected */
				if ( $is_image ) {
					if ( isset( $entry['id'] ) ) {
						list( $width, $height ) = $this->get_image_dimensions( $original_url, $entry['id'] );
					}

					$file = sprintf(
						'<a href="%1$s" target="_blank" aria-label="%2$s"><img style="max-width:100%%; padding: 5px; height: auto;" src="%1$s" width="%3$s" height="%4$s" /></a>',
						esc_url( $file ),
						esc_attr__( 'Click to view', 'image-hopper' ),
						esc_attr( isset( $width ) ? $width : '' ),
						esc_attr( isset( $height ) ? $height : '' )
					);
				} else {
					/* Add support for non-images (eg. if a File Upload field is converted to an Image Hopper field) */
					$file = sprintf(
						'<a href="%1$s" target="_blank" aria-label="%2$s">%3$s</a>',
						esc_attr( $file ),
						esc_attr__( 'Click to view', 'image-hopper' ),
						esc_html( $filename )
					);
				}
			} elseif ( $esc_html ) {
				$file = esc_html( $file );
			}
		}

		unset( $file );

		$value = $format === 'html' ? implode( '<br />', $files ) : implode( ', ', $files );
		if ( $url_encode ) {
			$value = rawurlencode( $value );
		}

		return $value;
	}

	/**
	 * @param $value
	 * @param $form_id
	 * @param $input_name
	 *
	 * @return string
	 *
	 * @since 1.5
	 */
	protected function reorder_files( $value, $form_id, $input_name ) {
		$posted_files = \GFCommon::json_decode( stripslashes( \GFForms::post( 'gform_uploaded_files' ) ) );

		/* Do nothing if no file input found */
		if ( ! isset( $posted_files[ $input_name ] ) ) {
			return $value;
		}

		/* Get the current files saved */
		$files = array_map( 'trim', json_decode( $value, true ) );

		/* Handle newly uploaded images included with existing images */
		$file_order = $this->get_new_file_order( $posted_files[ $input_name ], $files, $form_id, $input_name );

		/* Now order all the URLs correctly */
		$value             = [];
		$not_in_order_list = [];
		foreach ( $files as $file ) {
			/* If we cannot find the image we'll save it for later and append it to the end */
			$index = array_search( $this->strip_protocol_from_url( $file ), $file_order, true );
			if ( $index === false ) {
				$not_in_order_list[] = $file;
				continue;
			}

			$value[ $index ] = $file;
		}

		ksort( $value, SORT_NUMERIC );

		/* Merge in any images not found in the user order */
		$value = array_merge( $value, $not_in_order_list );

		return wp_json_encode( $value, JSON_UNESCAPED_UNICODE );
	}

	/**
	 * Get the user-defined order of all images in a format that can be processed
	 *
	 * Newly uploaded images are saved automatically to the end of the $files list so we will reverse the two lists
	 * to make it easy to match up the order with the correct file URL.
	 *
	 * We can't just check the filename from $uploaded_files_tmp because when Gravity Forms moves images from the
	 * tmp location to permanent storage those images may be renamed to prevent conflict with existing files.
	 * The end result is an order list that doesn't obviously reflect the saved URLs for newly-uploaded files,
	 * and this code rectifies that.
	 *
	 * @param array $file_order
	 * @param array $files
	 * @param int $form_id
	 * @param string $input_name
	 *
	 * @return array
	 *
	 * @since 1.5
	 */
	protected function get_new_file_order( $file_order, $files, $form_id, $input_name ) {

		$uploaded_files_tmp = [];
		if ( isset( \GFFormsModel::$uploaded_files[ $form_id ][ $input_name ] ) && is_array( \GFFormsModel::$uploaded_files[ $form_id ][ $input_name ] ) ) {
			$uploaded_files_tmp = array_reverse( \GFFormsModel::$uploaded_files[ $form_id ][ $input_name ] );
		}

		/* Get the URL for newly uploaded images */
		$uploaded_files = [];

		foreach ( array_reverse( $files ) as $index => $file ) {
			if ( ! isset( $uploaded_files_tmp[ $index ] ) ) {
				break;
			}

			$uploaded_files[ $uploaded_files_tmp[ $index ]['temp_filename'] ] = $file;
		}

		$uploaded_files = array_reverse( $uploaded_files );

		/* Match the newly-uploaded image URLs with the user-defined order */
		foreach ( $file_order as $index => $file ) {
			if ( ! isset( $uploaded_files[ $file['temp_filename'] ] ) ) {
				continue;
			}

			$file_order[ $index ]['uploaded_filename'] = $uploaded_files[ $file['temp_filename'] ];
		}

		/* Return the ordered list of images by the URL */
		return array_map( [ $this, 'strip_protocol_from_url' ], array_column( $file_order, 'uploaded_filename' ) );
	}

	/**
	 * Compare entry files with the new list of files submitted and delete and they have been removed from the UI
	 * Previously we deleted files as soon as they were removed form the UI, but that caused sync issues with the DB
	 * if the entry wasn't saved. Deleting when the form is submitted is better as we let Gravity Forms et al. handle
	 * permissions checks and if they're up to the point of saving we just handle it.
	 *
	 * @param string $value
	 * @param array $existing_uploaded_files
	 * @param int|null $entry_id
	 *
	 * @return string
	 *
	 * @since 2.4.0
	 */
	protected function maybe_delete_stored_files( $value, $existing_uploaded_files, $entry_id = null ) {
		$files = json_decode( $value, true ) ?: [];
		if ( $files === null || ! is_array( $files ) ) {
			return $value;
		}

		$existing_uploaded_files = array_map( [ $this, 'strip_protocol_from_url' ], $existing_uploaded_files );

		/* Delete any files no longer in the array (ignoring the protocol) */
		foreach ( $files as $url ) {
			if ( ! in_array( $this->strip_protocol_from_url( $url ), $existing_uploaded_files, true ) ) {
				$files = $this->delete_file( $url, $entry_id, $files );
			}
		}

		return wp_json_encode( array_values( $files ), JSON_UNESCAPED_UNICODE );
	}

	/**
	 * Parse the posted upload information and group new and existing files together
	 *
	 * @param array $form
	 * @param array $lead
	 * @param string $input_name
	 *
	 * @return array
	 */
	public function group_uploaded_files( $form, $lead, $input_name ) {
		$newly_uploaded_files    = [];
		$existing_uploaded_files = [];

		$value                   = (array) json_decode( rgar( $lead, (string) $this->id ), true );
		$protocol_stripped_value = array_map( [ $this, 'strip_protocol_from_url' ], $value );

		$posted_files   = \GFCommon::json_decode( stripslashes( \GFForms::post( 'gform_uploaded_files' ) ) );
		$uploaded_files = isset( \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] ) ? \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] : [];
		$tmp_location   = $this->get_tmp_upload_path( $form['id'] );

		foreach ( (array) $uploaded_files as $file ) {
			/* Check for newly uploaded file */
			if ( ! empty( $file['temp_filename'] ) ) {
				/* Verify the file is currently in the form's tmp directory */
				$temp_filepath = $tmp_location . wp_basename( $file['temp_filename'] );
				if ( is_file( $temp_filepath ) ) {
					$newly_uploaded_files[] = $file;
				}

				continue;
			}

			/* If we don't have any posted files, as a fallback we'll look for the file in the existing upload directory */
			if ( ! isset( $posted_files[ $input_name ] ) ) {
				$uploaded_file = \GFFormsModel::get_file_upload_path( $form['id'], $file['uploaded_filename'], false );
				if ( is_file( $uploaded_file['path'] ) ) {
					$existing_uploaded_files[] = $uploaded_file['url'];
				}

				continue;
			}

			/*
			 * When prepopulating an Image Hopper field the image file might be located elsewhere on the server
			 * (eg. Media Library or another form) and using `\GFFormsModel::get_file_upload_path()` may yield
			 * an invalid image path. To correct this, provided the URL is from the same domain as this site and
			 * the filename exactly matches Gravity Forms sanitized filename, we will use the URL direct from the
			 * $_POST data.
			 */

			/* Use a loop here because the images may be out of order and this is the most reliable method of parsing */
			foreach ( $posted_files[ $input_name ] as $raw_file ) {

				$url = $raw_file['uploaded_filename'];

				/* Skip over any URLs that don't include the current filename */
				if ( strpos( $url, '/' . wp_basename( $file['uploaded_filename'] ) ) === false ) {
					continue;
				}

				/* Look at the existing entry URLs (excluding the protocol) */
				if ( in_array( $this->strip_protocol_from_url( $url ), $protocol_stripped_value, true ) ) {
					$existing_uploaded_files[] = $url;
					break;
				}

				/* Look for the existing hash key for this URL to verify it */
				$entry_id  = isset( $lead['id'] ) ? $lead['id'] : 0;
				$path_meta = gform_get_meta( $entry_id, self::get_file_upload_path_meta_key_hash( $url ) );
				if ( ! empty( $path_meta['file_name'] ) ) {
					$existing_uploaded_files[] = $url;
					break;
				}

				/*
				 * If a developer has used the filter `image_hopper_allowed_prepopulated_url`, they will be in charge of
				 * checking the allowed prepopulated URLs. If the filter isn't used, we will fallback to the current site
				 * URL as the only valid domain.
				 */
				$validated_via_hook = has_filter( 'image_hopper_allowed_prepopulated_url' );
				if ( $validated_via_hook && apply_filters( 'image_hopper_allowed_prepopulated_url', false, $url, $form, $lead, $input_name ) ) {
					$existing_uploaded_files[] = $url;

					continue;
				}

				/*
				 * The filter above wasn't used to validate, so fallback to the current Site URL and filename match
				 */
				if ( stripos( $this->strip_protocol_from_url( $url ), $this->strip_protocol_from_url( site_url() ) ) === 0 && pathinfo( $url, PATHINFO_BASENAME ) === $file['uploaded_filename'] ) {
					$existing_uploaded_files[] = $url;
					break;
				}
			}
		}

		return [ $newly_uploaded_files, $existing_uploaded_files ];
	}

	/**
	 * Get Gravity Forms tmp path in a backwards-compatible way
	 *
	 * @param int $form_id
	 *
	 * @return string
	 *
	 * @since 2.16.4
	 */
	protected function get_tmp_upload_path( $form_id ) {
		if ( method_exists( \GFFormsModel::class, 'get_tmp_upload_location' ) ) {
			$tmp_location = \GFFormsModel::get_tmp_upload_location( $form_id );
			if ( isset( $tmp_location['path'] ) ) {
				return $tmp_location['path'];
			}
		}

		return \GFFormsModel::get_upload_path( $form_id ) . DIRECTORY_SEPARATOR . 'tmp' . DIRECTORY_SEPARATOR;
	}

	/**
	 * Delete a file, it's metadata, and remove it from the $files list
	 *
	 * @param string $url
	 * @param int $entry_id
	 * @param array $files
	 *
	 * @return array
	 *
	 * @since 2.4.2
	 */
	public function delete_file( $url, $entry_id, $files ) {
		$file_path = \GFFormsModel::get_physical_file_path( $url, $entry_id );
		$file_path = apply_filters( 'gform_file_path_pre_delete_file', $file_path, $url );

		/* Attempt to delete */
		if ( is_file( $file_path ) ) {
			unlink( $file_path );
			if ( $entry_id !== null ) {
				gform_delete_meta( $entry_id, self::get_file_upload_path_meta_key_hash( $url ) );
			}
		}

		/* Remove URL from saved value */
		$key = array_search( $url, $files, true );
		if ( $key !== false ) {
			unset( $files[ $key ] );
		}

		return $files;
	}

	/**
	 * Retrieve the input value on submission.
	 *
	 * @param string    $standard_name            The input name used when accessing the $_POST.
	 * @param string    $custom_name              The dynamic population parameter name.
	 * @param array     $field_values             The dynamic population parameter names with their corresponding values to be populated.
	 * @param bool|true $get_from_post_global_var Whether to get the value from the $_POST array as opposed to $field_values.
	 *
	 * @return array|string
	 */
	public function get_input_value_submission( $standard_name, $custom_name = '', $field_values = array(), $get_from_post_global_var = true ) {

		$form_id = $this->formId;
		/* phpcs:ignore WordPress.Security.NonceVerification.Missing */
		if ( ! empty( $_POST[ 'is_submit_' . $form_id ] ) && $get_from_post_global_var ) {
			$value = rgpost( $standard_name, false );
			$value = \GFFormsModel::maybe_trim_input( $value, $form_id, $this );

			return $value;
		} elseif ( $this->allowsPrepopulate ) {
			return \GFFormsModel::get_parameter_value( $custom_name, $field_values, $this );
		}
	}

	/**
	 * Remove the protocol (http: or https:) from a URL
	 *
	 * @param string $url
	 *
	 * @return string
	 */
	public function strip_protocol_from_url( $url ) {
		/* ignore invalid inputs */
		if ( ! is_string( $url ) ) {
			return $url;
		}

		if ( stripos( $url, 'http://' ) !== 0 && stripos( $url, 'https://' ) !== 0 ) {
			return $url;
		}

		return preg_replace( '~^https?:~i', '', $url );
	}

	/**
	 * Return the image width/height
	 *
	 * @param string $url
	 * @param int    $entry_id
	 *
	 * @return array
	 *
	 * @since 2.13.3
	 */
	public function get_image_dimensions( $url, $entry_id ) {
		global $_gform_lead_meta;

		/* Get image meta data*/
		$meta_key  = self::get_file_upload_path_meta_key_hash( $url );
		$cache_key = get_current_blog_id() . '_' . $entry_id . '_' . $meta_key;

		/* If the cached meta data does not have the path saved then force a DB query for fresh info */
		if ( empty( $_gform_lead_meta[ $cache_key ]['path'] ) ) {
			unset( $_gform_lead_meta[ $cache_key ] );
		}

		$file_meta = gform_get_meta( $entry_id, $meta_key );

		/* If the meta data is still missing, try to calculate the info */
		if ( empty( $file_meta['path'] ) ) {
			try {
				$path     = FileHelper::url_to_path( $url );
				$filename = wp_basename( $path );
				$dirname  = trailingslashit( dirname( $path ) );
				$url      = trailingslashit( dirname( $url ) );

				$file_meta['path']      = $dirname;
				$file_meta['url']       = $url;
				$file_meta['file_name'] = $filename;
			} catch ( \Exception $e ) {
				return [ '', '' ];
			}
		}

		/* If the width/height is not currently saved to the metadata then get and save it */
		if ( empty( $file_meta['image_width'] ) || empty( $file_meta['image_height'] ) ) {
			$width  = '';
			$height = '';

			if ( isset( $file_meta['path'], $file_meta['file_name'] ) && is_file( trailingslashit( $file_meta['path'] ) . $file_meta['file_name'] ) ) {
				$results = wp_getimagesize( trailingslashit( $file_meta['path'] ) . $file_meta['file_name'] );

				$width  = ! empty( $results[0] ) ? (int) $results[0] : '';
				$height = ! empty( $results[1] ) ? (int) $results[1] : '';
			}

			$file_meta['image_width']  = $width;
			$file_meta['image_height'] = $height;

			gform_update_meta( $entry_id, $meta_key, $file_meta );
		}

		return [ $file_meta['image_width'], $file_meta['image_height'] ];
	}

	/**
	 * Do field validation
	 *
	 * @param mixed $value
	 * @param array $form
	 *
	 * @return void
	 *
	 * @since 2.14.0
	 */
	public function validate( $value, $form ) {
		$input_name = 'input_' . $this->id;

		\GFCommon::log_debug( __METHOD__ . '(): Validating field ' . $input_name );

		/*
		 * Skip remainder of the field validation if running an older version of GravityView.
		 * To accurately validate this field when editing with GV requires code merged into 2.40.0
		 * https://github.com/GravityKit/GravityView/pull/2326
		 */
		if ( defined( 'GV_PLUGIN_VERSION' ) && version_compare( GV_PLUGIN_VERSION, '2.40', '<' ) && ! empty( $_POST['is_gv_edit_entry'] ) ) { //phpcs:ignore WordPress.Security.NonceVerification.Missing
			parent::validate( '', $form );

			return;
		}

		/* If editing an existing entry, grab the entry to help match existing uploads */
		$entry_id = ! empty( $_REQUEST['lid'] ) ? (int) $_REQUEST['lid'] : 0; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$entry_id = apply_filters( 'image_hopper_validation_entry_id', $entry_id, $form );
		$entry    = $entry_id > 0 ? \GFFormsModel::get_lead( $entry_id ) : [];

		$value = rgar( $entry, (string) $this->id, '' );
		$value = json_decode( $value, true ) !== null ? $value : '';

		\GFCommon::log_debug( __METHOD__ . '(): Using entry ID #' . $entry_id . ' to get existing uploaded files for field ' . $input_name );

		do_action( 'image_hopper_pre_field_validation', $value, $form, $input_name, $entry_id, $entry, $this );

		list( $newly_uploaded_files, $existing_uploaded_files ) = $this->group_uploaded_files( $form, $entry ?: [], $input_name );

		$total_files = count( $newly_uploaded_files ) + count( $existing_uploaded_files );

		\GFCommon::log_debug( __METHOD__ . '(): Total files detected: ' . $total_files );

		/* Do isRequired check based on sanitized inputs */
		if ( $this->isRequired && $total_files === 0 ) {
			\GFCommon::log_debug( __METHOD__ . '(): Field required but no new or existing files detected, failing validation.' );

			$this->failed_validation  = true;
			$this->validation_message = esc_html__( 'This field is required.', 'image-hopper' );

			return;
		}

		/* Validate file extension of newly-uploaded files */
		$allowed_extensions = ! empty( $this->allowedExtensions ) ? \GFCommon::clean_extensions( explode( ',', strtolower( $this->allowedExtensions ) ) ) : [];
		foreach ( $newly_uploaded_files as $file_name ) {
			\GFCommon::log_debug( __METHOD__ . '(): Validating file upload for ' . $file_name['uploaded_filename'] );
			$info = pathinfo( rgar( $file_name, 'uploaded_filename' ) );

			if ( empty( $allowed_extensions ) ) {
				if ( \GFCommon::file_name_has_disallowed_extension( rgar( $file_name, 'uploaded_filename' ) ) ) {
					\GFCommon::log_debug( __METHOD__ . '(): The file has a disallowed extension, failing validation.' );

					$this->failed_validation  = true;
					$this->validation_message = empty( $this->errorMessage ) ? esc_html__( 'The uploaded file type is not allowed.', 'image-hopper' ) : $this->errorMessage;

					return;
				}
			} elseif ( ! empty( $info['basename'] ) && ! \GFCommon::match_file_extension( rgar( $file_name, 'uploaded_filename' ), $allowed_extensions ) ) {
					\GFCommon::log_debug( __METHOD__ . '(): The file is of a type that cannot be uploaded, failing validation.' );

					$this->failed_validation  = true;
					$this->validation_message = empty( $this->errorMessage ) ? sprintf(
						/* translators: a list of file extensions that are allowed */
						esc_html__( 'The uploaded file type is not allowed. Must be one of the following: %s', 'image-hopper' ),
						strtolower( implode( ', ', \GFCommon::clean_extensions( explode( ',', $this->allowedExtensions ) ) ) )
					) : $this->errorMessage;

					return;
			}
		}

		/* Do Max File validation */
		if ( ! rgblank( $this->maxFiles ) ) {
			$limit = (int) $this->maxFiles;
			if ( $total_files > $limit ) {
				\GFCommon::log_debug( __METHOD__ . '(): The total number of files ' . $total_files . ' exceeds the max files limit of ' . $limit );

				$this->failed_validation  = true;
				$this->validation_message = ''; // let Filepond show the validation error

				return;
			}
		}
	}
}
