<?php

namespace GFPDF\Helper\Fields;

use GFPDF\Helper\Helper_Abstract_Fields;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @package GFPDF\Helper\Fields
 *
 * @since 1.0
 */
class Field_Image_Hopper extends Helper_Abstract_Fields {

	/**
	 * Display the HTML version of this field
	 *
	 * @param string $value
	 * @param bool   $show_label
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function html( $value = '', $show_label = true ) {
		$files = $this->value();
		$html  = '';

		$supported_images = [ 'jpg', 'jpeg', 'png', 'gif', 'bmp' ];
		if ( function_exists( 'imagewebp' ) ) {
			$supported_images[] = 'webp';
		}

		if ( function_exists( 'imageavif' ) ) {
			$supported_images[] = 'avif';
		}

		if ( count( $files ) === 0 ) {
			return parent::html( $html );
		}

		$image_html     = '';
		$non_image_html = '';

		foreach ( $files as $file ) {
			/* Convert URL to path */
			$path = $this->misc->convert_url_to_path( $file );
			if ( $path !== false ) {
				$file = $path;
			}

			/* Output image if the file is a supported image, otherwise display a link */
			$extension = pathinfo( wp_parse_url( $file, PHP_URL_PATH ), PATHINFO_EXTENSION );
			if ( in_array( strtolower( $extension ), $supported_images, true ) ) {
				$image_html .= sprintf(
					'<div class="ih-image-container"><img src="%s" class="ih-image" /></div>',
					$path !== false ? esc_attr( $file ) : esc_url( $file )
				);
			} else {
				$non_image_html .= sprintf(
					'<div><a href="%1$s">%2$s</a></div>',
					esc_attr( $file ),
					wp_basename( $file )
				);
			}
		}

		$image_html = sprintf(
			'<div class="ih-image-group ih-image-group-field-%1$d">%2$s</div>',
			$this->field->id,
			$image_html
		);

		$html = $image_html . $non_image_html;

		return parent::html( $html );
	}

	/**
	 * Generate this field's $form_data values
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function form_data() {
		$data     = [];
		$label    = $this->get_label();
		$value    = $this->value();
		$field_id = $this->field->id;

		/* Backwards compatibility support for v3 */
		if ( 0 === count( $value ) ) {
			$data[ $field_id . '.' . $label ] = [];
			$data[ $field_id ]                = [];
			$data[ $label ]                   = [];

			/* Path Format */
			$data[ $field_id . '_path' ]                = [];
			$data[ $field_id . '.' . $label . '_path' ] = [];

			/* Secured Download URLs */
			$data[ $field_id . '_secured' ]                = [];
			$data[ $field_id . '.' . $label . '_secured' ] = [];
		}

		foreach ( $value as $image ) {

			$data[ $field_id . '.' . $label ][] = $image;
			$data[ $field_id ][]                = $image;
			$data[ $label ][]                   = $image;

			$path = $this->misc->convert_url_to_path( $image );

			/* Path Format */
			$data[ $field_id . '_path' ][]                = $path;
			$data[ $field_id . '.' . $label . '_path' ][] = $path;

			/* Secured Download URLs */
			$data[ $field_id . '_secured' ][]                = $this->field->get_download_url( $image );
			$data[ $field_id . '.' . $label . '_secured' ][] = $this->field->get_download_url( $image );
		}

		return [ 'field' => $data ];
	}

	/**
	 * Get the field value in a ready-to-use format
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function value() {
		if ( $this->has_cache() ) {
			return $this->cache();
		}

		$value = $this->get_value();
		$files = [];

		if ( ! empty( $value ) ) {
			$paths = json_decode( $value, true );
			if ( $paths === null ) {
				$paths = [ $value ];
			}

			if ( is_array( $paths ) && count( $paths ) > 0 ) {
				foreach ( $paths as $path ) {
					$files[] = $path;
				}
			}
		}

		$this->cache( $files );

		return $this->cache();
	}
}
