<?php

namespace ImageHopper\ImageHopper\FirstParty;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @since 2.2
 */
class UserRegistration {

	/**
	 * @var object $_instance If available, contains an instance of this class.
	 *
	 * @since 2.2
	 */
	private static $_instance = null;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return self $_instance An instance of this class.
	 *
	 * @since 2.2
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * @since 2.2
	 */
	public function init() {
		add_filter( 'gform_user_registration_user_data_pre_populate', [ $this, 'user_registration_prepopulate' ], 10, 3 );
		add_action( 'gform_pre_submission', [ $this, 'undo_handle_existing_images_submission' ], 20 );
	}

	/**
	 * If the User Register Feed has an Image Hopper field linked prepopulate it
	 *
	 * @param array $mapped_fields
	 * @param array $form
	 * @param array $feed
	 *
	 * @return array
	 *
	 * @since 2.2
	 */
	public function user_registration_prepopulate( $mapped_fields, $form, $feed ) {
		$mapped_field_ids = array_map( 'intval', array_keys( $mapped_fields ) );

		foreach ( $form['fields'] as $field ) {
			if ( ! in_array( $field->id, $mapped_field_ids, true ) ) {
				continue;
			}

			if ( $field->type === 'image_hopper' ) {
				$value = rgar( $mapped_fields, $field->id );

				if ( empty( $value ) ) {
					continue;
				}

				/* check if file has been "deleted" via form UI and skip */
				$upload_files = json_decode( rgpost( 'gform_uploaded_files' ), ARRAY_A );
				$input_name   = "input_{$field->id}";
				if ( is_array( $upload_files ) && array_key_exists( $input_name, $upload_files ) && ! $upload_files[ $input_name ] ) {
					break;
				}

				/* if $uploaded_files array is not set for this form at all, init as array */
				if ( ! isset( \GFFormsModel::$uploaded_files[ $form['id'] ] ) ) {
					\GFFormsModel::$uploaded_files[ $form['id'] ] = [];
				}

				/*
				 * Check if this field's key has been set in the $uploaded_files array, if not add this file
				 * (otherwise, a new image may have been uploaded which we don't want to overwrite)
				 */
				if ( ! isset( \GFFormsModel::$uploaded_files[ $form['id'] ][ "input_{$field->id}" ] ) ) {
					$paths  = is_string( $value ) ? explode( ',', $value ) : $value;
					$values = [];
					foreach ( $paths as $url ) {
						$values[] = [
							'temp_filename'     => '',
							'uploaded_filename' => trim( $url ),
							'url'               => trim( $url ),
						];
					}

					\GFFormsModel::$uploaded_files[ $form['id'] ][ "input_{$field->id}" ] = $values;
				}
			}
		}

		return $mapped_fields;
	}

	/**
	 * The user registration pre-loads existing images on form submission, which causes problems with Image Hopper Editor
	 * Undo this behaviour so the native Image Hopper field can handle hydration and saving correctly.
	 *
	 * @param array $form
	 *
	 * @return void
	 *
	 * @since 2.7.1
	 */
	public function undo_handle_existing_images_submission( $form ) {
		global $_gf_uploaded_files;

		/* Ignore if using Stripe Elements */
		if ( rgget( 'payment_intent' ) ) {
			return;
		}

		/* Unset the cached file data for Image Hopper fields */
		foreach ( \GFAPI::get_fields_by_type( $form, 'image_hopper' ) as $field ) {
			$input_id = 'input_' . $field->id;
			if ( isset( $_gf_uploaded_files[ $input_id ] ) ) {
				unset( $_gf_uploaded_files[ $input_id ] );
			}
		}
	}
}
