<?php

namespace ImageHopper\ImageHopper\FirstParty;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

class AdvancedPostCreation {

	/**
	 * @var object $_instance If available, contains an instance of this class.
	 *
	 * @since
	 */
	private static $_instance = null;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return self $_instance An instance of this class.
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * @since
	 */
	public function init() {
		add_filter( 'gform_advancedpostcreation_file_fields_choices', [ $this, 'add_apc_support' ], 10, 2 );
		add_action( 'gform_post_save_feed_settings', [ $this, 'process_ih_mergetag_in_post_content' ], 10, 4 );

		/* Ensure featured image is set correctly */
		add_filter( 'gform_advancedpostcreation_post', [ $this, 'add_featured_image_fix' ] );
		add_filter( 'gform_advancedpostcreation_update_post_data', [ $this, 'add_featured_image_fix' ] );
		add_filter( 'gform_advancedpostcreation_field_value', [ $this, 'do_featured_image_fix' ], 10, 4 );
		add_action( 'update_post_metadata', [ $this, 'remove_featured_image_fix' ], 10, 3 );

		/* Ensure files are processed correctly when updating the post */
		add_action( 'gform_advancedpostcreation_pre_update_post', [ $this, 'mark_ih_for_apc_post_update' ] );
		add_action( 'gform_advancedpostcreation_post_update_post', [ $this, 'unmark_ih_for_apc_post_update' ] );
	}

	/**
	 * Add Image Hopper to the APC Upload field settings
	 *
	 * @param array   $choices     List of upload fields
	 * @param array   $form        Current Gravity Form
	 *
	 * @return array
	 *
	 * @since 1.2.0
	 */
	public function add_apc_support( $choices, $form ) {
		$fields = \GFAPI::get_fields_by_type( $form, [ 'image_hopper' ] );

		foreach ( $fields as $field ) {
			$choices[] = [
				'value' => $field->id,
				'label' => \GFCommon::get_label( $field ),
			];
		}

		return $choices;
	}

	/**
	 * Look for Image Hopper mergetags in the Post Content setting and, if found, auto-add them to the Add to Media Library setting.
	 *
	 * @param integer $feed_id  Feed Id
	 * @param integer $form_id  Form Id
	 * @param array   $settings Feed Settings
	 * @param object  $apc_class    instance of GF_Advanced_Post_Creation
	 *
	 * @since 1.2.0
	 */
	public function process_ih_mergetag_in_post_content( $feed_id, $form_id, $settings, $apc_class ) {
		if ( ! $apc_class instanceof \GF_Advanced_Post_Creation ) {
			return;
		}

		$form = \GFAPI::get_form( $form_id );
		if ( is_wp_error( $form ) ) {
			return;
		}

		/* Do regex match here */
		preg_match_all( '/{[^{]*?:(\d+(\.\d+)?)(:(.*?))?}/mi', $settings['postContent'], $matches, PREG_SET_ORDER );

		/* Get all matching field ids */
		$field_ids = [];
		foreach ( $matches as $match ) {
			$field_ids[] = $match[1];
		}

		/* Check if any of the matches are IH fields and insert into media library setting */
		$settings['postMedia'] = isset( $settings['postMedia'] ) && is_array( $settings['postMedia'] ) ? $settings['postMedia'] : [];
		$image_hoppers         = \GFFormsModel::get_fields_by_type( $form, 'image_hopper' );
		foreach ( $image_hoppers as $field ) {
			/* Insert Image Hopper field if it exists on current set of fields and skip if it already exists in settings's postMedia array */
			if ( in_array( (string) $field->id, $settings['postMedia'], true ) || ! in_array( (string) $field->id, $field_ids, true ) ) {
				continue;
			}

			$settings['postMedia'][] = (string) $field->id;
		}

		/* Update settings */
		$apc_class->update_feed_meta( $feed_id, $settings );
	}

	/**
	 * @since 2.3
	 */
	public function add_featured_image_fix( $post ) {
		add_filter( 'gform_addon_field_value', [ $this, 'do_featured_image_fix' ], 10, 4 );

		return $post;
	}

	/**
	 * @param $field_value
	 * @param $form
	 * @param $entry
	 * @param $field_id
	 *
	 * @return string
	 *
	 * @since 2.3
	 */
	public function do_featured_image_fix( $field_value, $form, $entry, $field_id ) {
		$field = \GFAPI::get_field( $form, $field_id );
		if ( $field !== false && $field->type === 'image_hopper' ) {
			$field_value = array_map( 'trim', explode( ',', $field_value ) )[0];
		}

		return $field_value;
	}

	/**
	 * @since 2.3
	 */
	public function remove_featured_image_fix( $short_circuit, $object_id, $meta_key ) {
		if ( $meta_key === '_thumbnail_id' ) {
			remove_filter( 'gform_addon_field_value', [ $this, 'do_featured_image_fix' ] );
		}

		return $short_circuit;
	}

	/**
	 * Force IH fields to become File Upload fields when updating a post via APC
	 *
	 * @return void
	 *
	 * @since 2.8.6
	 */
	public function mark_ih_for_apc_post_update() {
		add_filter( 'image_hopper_field_input_type', [ $this, 'mark_field_as_fileupload' ] );
	}

	/**
	 * Remove hook that forces IH fields to become File Upload fields when updating a post via APC
	 *
	 * @return void
	 *
	 * @since 2.8.6
	 */
	public function unmark_ih_for_apc_post_update() {
		remove_filter( 'image_hopper_field_input_type', [ $this, 'mark_field_as_fileupload' ] );
	}

	/**
	 * @param string $type
	 *
	 * @return string
	 *
	 * @since 2.13.6
	 */
	public function mark_field_as_fileupload( $type ) {
		return 'fileupload';
	}
}
