<?php

namespace ImageHopper\ImageHopper\Fields;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * Class SlimField
 *
 * Polyfill this field type so users can continue to view Slim fields in their Entry Details page. Disable field for data input.
 *
 * @package ImageHopper\ImageHopper\Fields
 *
 * @since 1.0.0
 */
class SlimField extends \GF_Field_FileUpload {

	/**
	 * @var string
	 *
	 * @since 1.0.0
	 */
	public $type = 'slim';

	/**
	 * Disable button in Form Editor
	 *
	 * @param array $field_groups
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function add_button( $field_groups ) {
		return $field_groups;
	}

	/**
	 * Make sure nothing shows up on the front end for this field any more
	 *
	 * @param array  $form
	 * @param string $value
	 * @param null   $entry
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_field_input( $form, $value = '', $entry = null ) {
		return '';
	}

	/**
	 * Make sure nothing shows up on the front end for this field any more
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function get_field_label_class() {
		return 'gf_hidden';
	}

	/**
	 * Ensure the field entry data can still be viewed when viewing Entry List
	 *
	 * @param array|string $value
	 * @param array        $entry
	 * @param string       $field_id
	 * @param array        $columns
	 * @param array        $form
	 *
	 * @return array|string
	 *
	 * @since 1.0.0
	 */
	public function get_value_entry_list( $file_url, $entry, $field_id, $columns, $form ) {
		if ( empty( $file_url ) ) {
			return $file_url;
		}

		$icon_url = \GFEntryList::get_icon_url( $file_url );
		$file_url = esc_attr( $this->get_download_url( $file_url ) );
		$title    = esc_attr__( 'Click to view', 'image-hopper' );

		return "<a href='{$file_url}' target='_blank' title='{$title}'><img src='{$icon_url}' alt=''/></a>";
	}

	/**
	 * Ensure the field entry data can still be viewed when viewing Entry Details
	 *
	 * @param array|string $value
	 * @param string       $currency
	 * @param false        $use_text
	 * @param string       $format
	 * @param string       $media
	 *
	 * @return array|string
	 *
	 * @since 1.0.0
	 */
	public function get_value_entry_detail( $file_url, $currency = '', $use_text = false, $format = 'html', $media = 'screen' ) {
		if ( empty( $file_url ) ) {
			return $file_url;
		}

		$file_path      = $file_url;
		$file_path_info = pathinfo( $file_path );
		$file_name      = $file_path_info['basename'];

		// no fun on 1.x
		$force_download = method_exists( $this, 'get_modifiers' ) && in_array( 'download', $this->get_modifiers(), true );
		$file_url       = $this->get_download_url( $file_url, $force_download );

		$field_ssl = apply_filters( 'gform_secure_file_download_is_https', true, $file_url, $this );

		if ( \GFCommon::is_ssl() && strpos( $file_url, 'http:' ) !== false && $field_ssl === true ) {
			$file_url = str_replace( 'http:', 'https:', $file_url );
		}

		$file_url = str_replace( ' ', '%20', apply_filters( 'gform_fileupload_entry_value_file_path', $file_url, $this ) );
		/* phpcs:ignore WordPress.WP.I18n.TextDomainMismatch */
		$click_to_view_text = __( 'Click to view image', 'slim-gravityforms' );

		return $format === 'text' ? $file_path . PHP_EOL : '<a href="' . esc_url( $file_url ) . '" target="_blank" title="' . esc_attr( $click_to_view_text ) . '"><span>' . esc_html( $file_name ) . '</span><img style="max-width:100%" src="' . esc_url( $file_url ) . '" alt=""/></a>';
	}

	/**
	 * Disable field validation
	 *
	 * @param array|string $value
	 * @param array        $form
	 *
	 * @since 1.0.0
	 */
	public function validate( $value, $form ) {
	}
}
