<?php

namespace ImageHopper\ImageHopper\API;

use ImageHopper\ImageHopper\ImageHopperAddOn;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * Class ImageManager
 *
 * @package ImageHopper\ImageHopper\API
 *
 * @since   1.0.0
 */
class ImageManager {

	/**
	 * Add detailed logging during image uploading to help with debugging
	 *
	 * @since 2.0.1
	 */
	public function preflight() {
		$ih = ImageHopperAddOn::get_instance();

		$ih->log_debug( 'Begin preflight image upload checks' );

		/* phpcs:ignore WordPress.Security.NonceVerification.Missing */
		$ih->log_debug( 'POST: ' . wp_json_encode( $_POST ) );

		$form_id        = absint( rgpost( 'form_id' ) );
		$form_unique_id = rgpost( 'gform_unique_id' );
		$form           = \GFAPI::get_form( $form_id );

		if ( empty( $form ) ) {
			$ih->log_error( 'File upload aborted because the form was not found.' );

			return;
		}

		if ( ! $form['is_active'] ) {
			$ih->log_error( 'File upload aborted because the form is not active.' );

			return;
		}

		if ( rgar( $form, 'requireLogin' ) ) {
			if ( ! is_user_logged_in() ) {
				$ih->log_error( 'File upload aborted because the current user is not logged in.' );

				return;
			}

			if ( ! wp_verify_nonce( rgpost( '_gform_file_upload_nonce_' . $form_id ), 'gform_file_upload_' . $form_id ) ) {
				$ih->log_error( 'File upload aborted because the verify_nonce check failed' );

				return;
			}
		}

		if ( ! ctype_alnum( $form_unique_id ) ) {
			$ih->log_error( 'File upload aborted because the ctype_alnum check failed' );

			return;
		}

		$field_id = absint( rgpost( 'field_id' ) );
		$field    = gf_apply_filters( [ 'gform_multifile_upload_field', $form['id'], $field_id ], \GFFormsModel::get_field( $form, $field_id ), $form, $field_id );

		if ( empty( $field ) ) {
			$ih->log_error( 'File upload aborted because the associated field could not be found in the form.' );

			return;
		}

		$field_type = \GFFormsModel::get_input_type( $field );
		if ( $field_type !== 'fileupload' ) {
			$ih->log_error( 'File upload aborted because the associated field is not a file upload field. The type returned was "' . $field_type . '".' );

			return;
		}

		$max_upload_size_in_bytes = $field->maxFileSize > 0 ? $field->maxFileSize * 1048576 : wp_max_upload_size();
		$max_upload_size_in_mb    = $max_upload_size_in_bytes / 1048576;

		/* Check for upload errors */
		if ( ! empty( $_FILES['file']['error'] ) ) {
			switch ( $_FILES['file']['error'] ) {
				case UPLOAD_ERR_INI_SIZE:
				case UPLOAD_ERR_FORM_SIZE:
					$ih->log_error( 'File ' . $_FILES['file']['name'] . ' exceeds size limit. Maximum file size: ' . $max_upload_size_in_mb . 'MB' );

					/* translators: %s = maximum file size in MB */
					$fileupload_validation_message = sprintf( __( 'File exceeds size limit. Maximum file size: %sMB', 'image-hopper' ), $max_upload_size_in_mb );
					break;

				default:
					$ih->log_error( 'The following error occurred while uploading: ' . $_FILES['file']['error'] );
					$fileupload_validation_message = __( 'There was an error while uploading the file.', 'image-hopper' );
			}

			$response = [
				'status' => 'error',
				'error'  => [
					'code'    => 104,
					'message' => esc_html( $fileupload_validation_message ),
				],
			];

			die( json_encode( $response ) );
		}

		if ( $_FILES['file']['size'] > 0 && $_FILES['file']['size'] > $max_upload_size_in_bytes ) {
			$ih->log_error( 'File upload aborted because the file exceeds the maximum file size limit. The maximum size is ' . $max_upload_size_in_mb . 'MB and the file size is ' . round( $_FILES['file']['size'] / 1048576, 2 ) . 'MB.' );

			return;
		}

		$ih->log_debug( 'Preflight checks completed.' );
	}

	/**
	 * Return the contents of a file in "limbo" (uploaded but entry not saved)
	 *
	 * @since 1.0.0
	 */
	public function restore() {

		$ih = ImageHopperAddOn::get_instance();
		$ih->log_debug( 'Begin restoring image' );

		$data = $this->get_file();

		if ( ! is_file( $data['file'] ) ) {
			$ih->log_error( 'The image cannot be found on the server. Looking for ' . $data['file'] );

			status_header( 404 );
			die();
		}

		status_header( 200 );
		send_nosniff_header();
		nocache_headers();

		header( 'Access-Control-Expose-Headers: Content-Disposition' );

		if ( function_exists( 'mime_content_type' ) ) {
			$mime = mime_content_type( $data['file'] );
			header( 'Content-Type: ' . $mime );
			$ih->log_debug( 'Mime type: ' . mime_content_type( $data['file'] ) );
		}

		$filename = str_replace( "{$data['unique_id']}_input_{$data['field_id']}_", '', $data['tmp_filename'] );
		$filename = substr( $filename, 17 );

		$ih->log_debug( 'Filename: ' . $filename );
		$ih->log_debug( 'Finish restoring image' );

		header( 'Content-Disposition: inline; filename="' . rawurlencode( $filename ) . "\"; filename*=UTF-8''" . rawurlencode( $filename ) );

		/* phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_readfile */
		readfile( $data['file'] );

		die();
	}

	/**
	 * Delete a file in "limbo" (uploaded but entry not saved)
	 *
	 * @since 1.0.0
	 */
	public function revert() {

		$ih = ImageHopperAddOn::get_instance();
		$ih->log_debug( 'Begin reverting (deleting) temporary image' );

		$data = $this->get_file();

		if ( is_file( $data['file'] ) ) {
			unlink( $data['file'] );
			$ih->log_debug( 'The following image has been removed ' . $data['file'] );
		}

		$ih->log_debug( 'Finish reverting (deleting) temporary image' );

		status_header( 200 );
		die();
	}

	/**
	 * Delete a file already saved to the server
	 *
	 * @since 1.0.0
	 *
	 * @depecated 2.4.0 Handled upon form submission in \ImageHopper\ImageHopper\Fields\ImageHopperField()
	 */
	public function delete() {
		die();
	}

	/**
	 * Do the common checks and data formatting for all requests of the API
	 *
	 * @since 1.0.0
	 */
	protected function get_file() {
		$ih = ImageHopperAddOn::get_instance();

		$ih->log_debug( 'Start request validation' );

		/* phpcs:ignore WordPress.Security.NonceVerification.Missing */
		$ih->log_debug( wp_json_encode( $_POST ) );

		if ( $_SERVER['REQUEST_METHOD'] !== 'POST' ) {
			$ih->log_error( 'Request method invalid. Should be POST but using ' . $_SERVER['REQUEST_METHOD'] );
			status_header( 404 );
			die();
		}

		status_header( 400 );

		send_nosniff_header();
		nocache_headers();

		$form_id      = (int) rgpost( 'form_id' );
		$field_id     = (int) rgpost( 'field_id' );
		$tmp_filename = rgpost( 'file' );

		$unique_id = rgpost( 'gform_unique_id' );

		/* security checks on the filename */
		if ( ! ctype_alnum( $unique_id ) ) {
			$ih->log_error( 'Unique ID failed alphanumeric test. Only letters or numbers aloud in ' . $unique_id );
			die();
		}

		if ( strpos( $tmp_filename, 'input_' . $field_id ) !== ( strlen( $unique_id ) + 1 ) ) {
			$ih->log_error( 'Invalid temporary filename for ' . $tmp_filename );
			die();
		}

		$form = \GFAPI::get_form( $form_id );
		if ( empty( $form ) || ! $form['is_active'] ) {
			$ih->log_error( 'Form not found or inactive for ID' . $form_id );
			die();
		}

		if ( rgar( $form, 'requireLogin' ) ) {
			if ( ! is_user_logged_in() ) {
				$ih->log_error( 'The form can only be completed by logged in users' );
				die();
			}

			$results = check_admin_referer( 'gform_file_upload_' . $form_id, '_gform_file_upload_nonce_' . $form_id );
			if ( $results === false ) {
				$ih->log_error( 'The nonce is invalid.' );
				die();
			}
		}

		$target_dir = \GFFormsModel::get_upload_path( $form_id ) . '/tmp/';
		$file       = $target_dir . $tmp_filename;

		$data = [
			'file'         => $file,
			'form_id'      => $form_id,
			'field_id'     => $field_id,
			'unique_id'    => $unique_id,
			'tmp_filename' => $tmp_filename,
		];

		$ih->log_debug( 'Finish request validation' );
		$ih->log_debug( wp_json_encode( $data ) );

		return $data;
	}
}
