<?php

namespace ImageHopper\ImageHopper\Fields;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @since 2.13.0
 */
class FieldConversion {

	private static $_instance = null;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return self $_instance An instance of this class.
	 *
	 * @since 2.13.0
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * @since 2.13.0
	 */
	public function init() {
		/* Handle field migration */
		add_filter( 'gform_form_update_meta', [ $this, 'maybe_migrate_single_file_upload_field_value' ], 10, 3 );
	}

	/**
	 * Check if a Single File Upload field has been converted to Image Hopper, then update the field and DB
	 *
	 * @param array  $form_meta
	 * @param int    $form_id
	 * @param string $meta_name
	 *
	 * @return array
	 *
	 * @since 2.13.0
	 */
	public function maybe_migrate_single_file_upload_field_value( $form_meta, $form_id, $meta_name ) {
		if ( $meta_name !== 'display_meta' ) {
			return $form_meta;
		}

		$converted_fields = [];
		foreach ( $form_meta['fields'] as &$field ) {
			if ( $field->type !== 'image_hopper' ) {
				continue;
			}

			/* check whether the IH field was just converted from a single file upload field */
			if ( ! $field->convertedSingleFileUploadToImageHopper ) {
				continue;
			}

			/* normalize the field and store the field ID */
			unset( $field->convertedSingleFileUploadToImageHopper );
			$converted_fields[] = $field->id;
		}

		unset( $field );

		if ( empty( $converted_fields ) ) {
			return $form_meta;
		}

		/* check if database supports JSON_ARRAY. If not, skip */
		if ( ! $this->check_db_supports_json() ) {
			return $form_meta;
		}

		/* Convert the Single File Upload fields to expected JSON array format for multi file fields */
		$this->migrate_single_file_upload_field_value( $form_id, $converted_fields );

		return $form_meta;
	}

	/**
	 * Check if the database supports the required JSON function
	 *
	 * @internal JSON_ARRAY may be added to earlier version than this, but it's hard to find the specifics. GF uses these for the JSON_CONTAINS function check
	 *
	 * @return bool
	 */
	protected function check_db_supports_json() {
		$db_type    = \GFCommon::get_dbms_type();
		$db_version = \GFCommon::get_db_version();

		if ( $db_type === 'MySQL' && version_compare( $db_version, '5.7.8', '>=' ) ) {
			return true;
		}

		if ( $db_type === 'MariaDB' && version_compare( $db_version, '10.2.25', '>=' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Update the format of converted Single File Upload fields in the database
	 *
	 * @param int        $form_id
	 * @param array<int> $fields
	 *
	 * @return void
	 */
	protected function migrate_single_file_upload_field_value( $form_id, $fields ) {
		global $wpdb;

		$field_ids = implode( ',', array_filter( array_map( 'absint', $fields ) ) );
		$table     = \GFFormsModel::get_entry_meta_table_name();

		/* phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared */
		/* $table will always be a safe string returned from GFs, while $field_ids has been sanitized above */
		$wpdb->query(
			$wpdb->prepare(
				"UPDATE `{$table}`
		SET meta_value = JSON_ARRAY(meta_value) 
		WHERE form_id = %d AND meta_key IN ({$field_ids})",
				$form_id
			)
		);
		/* phpcs:enable */
	}
}
