<?php

namespace ImageHopper\ImageHopper\ThirdParty\GravityView;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @since 2.4.1
 */
class EditEntry {
	public function init() {
		add_action( 'gravityview_edit_entry', [ $this, 'handle_max_files_validation' ], 5 );
		add_action( 'gravityview/edit_entry/before_update', [ $this, 'strip_cache_buster_from_post' ] );
		add_action( 'gform_after_update_entry', [ $this, 'gravityview_mask_ih_post_field_file_superglobal' ] );
		add_filter( 'gform_pre_validation', [ $this, 'gravityview_edit_gform_pre_validation' ] );
	}

	/**
	 * Remove cache busting value from IH Post Image field during edit submission
	 *
	 * @param array $form
	 *
	 * @return void
	 *
	 * @since 2.5.0
	 */
	public function strip_cache_buster_from_post( $form ) {
		$ih_post_fields = \GFFormsModel::get_fields_by_type( $form, 'image_hopper_post' );

		foreach ( $ih_post_fields as $field ) {
			$input_name           = 'input_' . $field->id;
			$url                  = rgpost( $input_name );
			$_POST[ $input_name ] = remove_query_arg( 't', $url );
		}
	}

	/**
	 * If a newly-updated file during GravityView's Edit Entry submission, mask the $_FILES superglobal so it triggers
	 * the GV functionality to add the Post images to the Media Library
	 *
	 * @param array $form
	 *
	 * @since 1.3.0
	 * @internal Moved out of the ImageHopperPostField class in version 2.7.0
	 */
	public function gravityview_mask_ih_post_field_file_superglobal( $form ) {
		$uploads = json_decode( stripslashes( \GFForms::post( 'gform_uploaded_files' ) ), true );
		if ( $uploads === null ) {
			return;
		}

		$ih_post_fields = \GFFormsModel::get_fields_by_type( $form, 'image_hopper_post' );

		foreach ( $ih_post_fields as $field ) {
			$input_name = 'input_' . $field->id;

			if ( ! isset( $uploads[ $input_name ] ) || ! is_array( $uploads[ $input_name ] ) ) {
				continue;
			}

			/* Check for a newly-uploaded Post image */
			$new_post_upload = array_filter(
				$uploads[ $input_name ],
				static function( $file ) {
					return ! empty( $file['temp_filename'] );
				}
			);

			if ( count( $new_post_upload ) > 0 ) {
				/* Ensure GravityView uploads to the media library */
				$_FILES[ $input_name ] = [
					'name' => 'mark as new file',
				];

				/* Reset the file store to resolve issues with the Image Editor */
				$uploads[ $input_name ] = $new_post_upload;
			}
		}

		$_POST['gform_uploaded_files'] = wp_json_encode( $uploads, JSON_UNESCAPED_UNICODE );
	}

	/**
	 * Correctly set the IH Post Field value during GravityView Edit Entry form submission
	 *
	 * @param array $form
	 *
	 * @return array
	 *
	 * @since 1.3.0
	 * @internal Moved out of the ImageHopperPostField class in version 2.7.0
	 */
	public function gravityview_edit_gform_pre_validation( $form ) {
		$gv_edit_entry        = \GravityView_Edit_Entry::getInstance();
		$gv_edit_entry_render = isset( $gv_edit_entry->instances['render'] ) ? $gv_edit_entry->instances['render'] : null;

		$ih_post_fields = \GFFormsModel::get_fields_by_type( $form, 'image_hopper_post' );
		foreach ( $ih_post_fields as $field ) {
			$input_name = 'input_' . $field->id;
			$form_id    = $form['id'];

			/* Get the correct value using the newly submitted data, or fallback to the existing entry data */
			$entry = $gv_edit_entry_render->get_entry();
			$value = isset( $entry[ $field->id ] ) ? $entry[ $field->id ] : null;
			if ( isset( \GFFormsModel::$uploaded_files[ $form_id ][ $input_name ] ) ) {
				$value = wp_json_encode( \GFFormsModel::$uploaded_files[ $form_id ][ $input_name ], JSON_UNESCAPED_UNICODE );
			}

			$gv_edit_entry_render->entry[ $input_name ] = $value;
		}

		return $form;
	}

	/**
	 * Increment the Max Files by the number of deleted images when editing and entry via GravityView.
	 * This is needed because GravityView does custom validation to verify the Max Files limit isn't reached,
	 * but Image Hopper doesn't delete existing images until the field is processed (@since 2.4). To work around this, we
	 * adjust Max Files up/down so we can still take advantage of the server side check.
	 *
	 * @return void
	 *
	 * @since 2.4.1
	 */
	public function handle_max_files_validation() {
		$form_id = rgpost( 'gform_submit' );
		if ( empty( $form_id ) ) {
			return;
		}

		$entry = gravityview_get_entry( \GravityView_frontend::is_single_entry() );
		if ( empty( $entry ) ) {
			return;
		}

		/* Increment Max Files by the number of deleted files */
		add_filter(
			'gform_validation_' . $form_id,
			function( $validation_results ) use ( $entry ) {
				return $this->adjust_max_files( $validation_results, $entry, 'increment' );
			},
			5
		);

		/* Restore Max Files to the original values */
		add_filter(
			'gform_validation_' . $form_id,
			function( $validation_results ) use ( $entry ) {
				return $this->adjust_max_files( $validation_results, $entry, 'decrement' );
			},
			15
		);
	}

	/**
	 * Finds the deleted files to be processed and increment or decrement the max files
	 *
	 * @param array $validation_results The form validation object
	 * @param array $entry A form entry object
	 * @param string $func A valid method name in the current class that'll increment or decrement the value
	 *
	 * @return array
	 */
	protected function adjust_max_files( $validation_results, $entry, $func ) {
		foreach ( $validation_results['form']['fields'] as &$field ) {
			if ( $field->type === 'image_hopper' && ! empty( $field->maxFiles ) && ! empty( $entry[ $field->id ] ) ) {
				$input_name = 'input_' . $field->id;
				if ( isset( \GFFormsModel::$uploaded_files[ $validation_results['form']['id'] ][ $input_name ] ) ) {
					$files                             = json_decode( $entry[ $field->id ], true );
					list( , $existing_uploaded_files ) = $field->group_uploaded_files( $validation_results['form'], $entry, $input_name );
					$field->maxFiles                   = $this->$func( $field->maxFiles, count( array_diff( $files, $existing_uploaded_files ) ) );
				}
			}
		}

		return $validation_results;
	}

	protected function increment( $val1, $val2 ) {
		return $val1 + $val2;
	}

	protected function decrement( $val1, $val2 ) {
		return $val1 - $val2;
	}
}
