<?php

namespace ImageHopper\ImageHopper\Licensing;

use ImageHopper\ImageHopper\ImageHopperAddOn;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * Trait LicensingChecks
 *
 * @package ImageHopper\ImageHopper\Licensing
 *
 * @since 1.0.0
 */
trait LicensingChecks {

	/**
	 * @var string the option name for the license key status
	 *
	 * @since 1.4
	 */
	protected $license_key_status_option_name = 'image_hopper_license_key_status';

	/**
	 * Determine if the license key is valid so the appropriate icon can be displayed next to the field.
	 *
	 * @param string $value The current value of the license_key field.
	 *
	 * @return bool|null
	 *
	 * @since 1.0.0
	 */
	public function is_license_valid( $value ) {
		if ( empty( $value ) ) {
			return null;
		}

		// Check license status, either using the cached data or polling the API
		$license_status = get_option( $this->license_key_status_option_name, '' );
		if ( ! isset( $license_status->timeout ) || $license_status->timeout < time() ) {
			$license_status = $this->do_license_check( 'check_license', $value );
			update_option( $this->license_key_status_option_name, $license_status );
		}

		$license_data = isset( $license_status->license ) ? $license_status->license : '';

		$valid = null;
		if ( empty( $license_data ) || $license_data === 'invalid' ) {
			$valid = false;
		} elseif ( $license_data === 'valid' ) {
			$valid = true;
		}

		return $valid;
	}

	/**
	 * Send a license check request to the store url.
	 *
	 * @param string $edd_action The action to perform (check_license, activate_license, or deactivate_license).
	 * @param string $license    The license key.
	 *
	 * @return object
	 *
	 * @since 1.0.0
	 */
	public function do_license_check( $edd_action, $license ) {
		$args = [
			'timeout' => 15,
			'body'    => [
				'edd_action'  => $edd_action,
				'license'     => trim( $license ),
				'item_name'   => rawurlencode( $this->_short_title ),
				'url'         => home_url(),
				'environment' => function_exists( 'wp_get_environment_type' ) ? wp_get_environment_type() : 'production',
			],
		];

		// Log request but mask license
		$request_body            = $args['body'];
		$request_body['license'] = substr( $license, 0, 1 ) . str_repeat( '*', strlen( $license ) - 2 ) . substr( $license, -1, 1 );

		ImageHopperAddOn::get_instance()->log_debug( 'Image Hopper License Check API Request Body: ' . print_r( $request_body, true ) ); //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

		$response = wp_remote_post( $this->_store_url, $args );

		// Log response
		ImageHopperAddOn::get_instance()->log_debug( 'Image Hopper License Check API Response Headers: ' . print_r( wp_remote_retrieve_headers( $response ), true ) ); //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
		ImageHopperAddOn::get_instance()->log_debug( 'Image Hopper License Check API Response Response Code: ' . print_r( wp_remote_retrieve_response_code( $response ), true ) ); //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r
		ImageHopperAddOn::get_instance()->log_debug( 'Image Hopper License Check API Response Response Body: ' . print_r( wp_remote_retrieve_body( $response ), true ) ); //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

		$response_object = json_decode( wp_remote_retrieve_body( $response ) );

		// gracefully handle errors
		if ( $response_object === null || ! is_object( $response_object ) ) {
			$response_object = new \StdClass();

			ImageHopperAddOn::get_instance()->log_error( 'Image Hopper License Check API Response failed: ' . print_r( $response_object, true ) ); //phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_print_r

		}

		// Set a timeout for the data, so we can refresh when needed
		$response_object->timeout = strtotime( '+3 hours' );

		return $response_object;
	}

	/**
	 * Handle license key activation or deactivation.
	 *
	 * @param array  $field         The field properties.
	 * @param string $field_setting The submitted value of the license_key field.
	 *
	 * @since 1.0.0
	 */
	public function license_validation( $field, $field_setting ) {
		$old_license = $this->get_plugin_setting( 'license_key' );

		if ( $old_license && $field_setting !== $old_license ) {
			// Send the remote request to deactivate the old license
			$this->do_license_check( 'deactivate_license', $old_license );
			delete_option( $this->license_key_status_option_name );
		}

		if ( ! empty( $field_setting ) ) {
			// Send the remote request to activate the new license
			$api_response = $this->do_license_check( 'activate_license', $field_setting );
			update_option( $this->license_key_status_option_name, $api_response );
			delete_site_transient( 'update_plugins' );
		}
	}

	/**
	 * Display appropriate message right after license key for users run Gravity Forms 2.5 or later
	 *
	 * @return string
	 *
	 * @since 1.4.0
	 */
	public function license_display_license_status_gf25() {
		$license_key = $this->get_plugin_setting( 'license_key' );

		if ( empty( $license_key ) ) {
			return '';
		}

		$license_key_status = get_option( $this->license_key_status_option_name, '' );

		// prior to IH 2.13 we only stored the license status as a string, so need to check if we have an object or not
		$license_key_status_response = isset( $license_key_status->license ) ? $license_key_status->license : $license_key_status;

		if ( $license_key_status_response === 'valid' ) {
			return sprintf( '<div class="alert gforms_note_success">%s</div>', esc_html__( 'Your support license key has been activated for this domain.', 'image-hopper' ) );
		}

		// The API check_license and activate_license return different data. Standardize first.
		$license_key_status_response = $license_key_status_response === 'invalid' && isset( $license_key_status->error ) ? $license_key_status->error : $license_key_status_response;

		$message = $this->get_license_error_message( $license_key_status_response );

		return sprintf( '<div class="alert gforms_note_error">%s</div>', wp_kses_post( $message ) );
	}

	/**
	 * Display appropriate message right after license key for users run Gravity Forms 2.4 or earlier
	 *
	 * @return string
	 *
	 * @since 1.4.0
	 */
	public function license_display_license_status_pre_gf25() {
		$license_key = $this->get_plugin_setting( 'license_key' );

		if ( empty( $license_key ) ) {
			return '';
		}

		$license_key_status = get_option( $this->license_key_status_option_name, '' );

		// prior to IH 2.13 we only stored the license status as a string, so need to check if we have an object or not
		$license_key_status_response = isset( $license_key_status->license ) ? $license_key_status->license : $license_key_status;

		if ( $license_key_status_response === 'valid' ) {
			return sprintf( "&nbsp;<i class='fa fa-check gf_keystatus_valid'></i> <span class='gf_keystatus_valid_text'>%s</span>", esc_html__( 'Your support license key has been activated for this domain.', 'image-hopper' ) );
		}

		// The API check_license and activate_license return different data. Standardize first.
		$license_key_status_response = $license_key_status_response === 'invalid' && isset( $license_key_status->error ) ? $license_key_status->error : $license_key_status_response;

		$message = $this->get_license_error_message( $license_key_status_response );

		return sprintf( "&nbsp;<i class='fa fa-times gf_keystatus_invalid'></i> <span class='gf_keystatus_invalid_text'>%s</span>", wp_kses_post( $message ) );
	}

	/**
	 * Setup plugin updater
	 *
	 * @since 1.0.0
	 */
	public function setup_plugin_updater() {
		$license_key = trim( (string) $this->get_plugin_setting( 'license_key' ) );

		new EDD_SL_Plugin_Updater(
			$this->_store_url,
			IMAGE_HOPPER_ADDON_FILE,
			[
				'version'   => IMAGE_HOPPER_ADDON_VERSION,
				'license'   => $license_key,
				'item_name' => $this->_short_title,
				'author'    => $this->_short_title,
			]
		);

		do_action( 'image_hopper_plugin_updater', $license_key, $this->_store_url, $this );

		add_filter( 'plugin_auto_update_setting_html', [ $this, 'auto_update_message' ], 10, 2 );
	}

	/**
	 * Filter the auto-update message on the plugins page.
	 *
	 * @param string $html        HTML of the auto-update message.
	 * @param string $plugin_file Plugin file.
	 *
	 * @return string
	 *
	 * @since 1.3.0
	 */
	public function auto_update_message( $html, $plugin_file ) {
		/* Only run on this plugin */
		if ( $plugin_file !== $this->_path ) {
			return $html;
		}

		if ( ! $this->is_license_valid( $this->get_plugin_setting( 'license_key' ) ) ) {
			return esc_html__( 'Auto-updates unavailable.', 'image-hopper' );
		}

		return $html;
	}

	/**
	 * @param $license_key_status
	 *
	 * @return string
	 */
	protected function get_license_error_message( $license_key_status ) {
		switch ( $license_key_status ) {
			case 'expired':
				/* translators: %1$s is an opening HTML anchor tag and %2$s is the closing tag */
				$message = sprintf( esc_html__( 'This license key has expired. %1$sPlease renew your license to continue receiving updates and support%2$s.', 'image-hopper' ), '<a href="https://imagehopper.tech/account/license-key/">', '</a>' );
				break;

			case 'revoked':
			case 'disabled':
				/* translators: %1$s is an opening HTML anchor tag and %2$s is the closing tag */
				$message = sprintf( esc_html__( 'This license key has been cancelled (most likely due to a refund request). %1$sPlease consider purchasing a new license%2$s.', 'image-hopper' ), '<a href="https://imagehopper.tech/#buy">', '</a>' );
				break;

			case 'inactive':
			case 'site_inactive':
				$message = esc_html__( 'Your license key is valid but does not match your current domain. This usually occurs if your domain URL changes. Please resave the settings to activate the license for this website.', 'image-hopper' );
				break;

			case 'no_activations_left':
				/* translators: %1$s is an opening HTML anchor tag and %2$s is the closing tag */
				$message = sprintf( esc_html__( 'This license key has reached its activation limit. %1$sPlease upgrade your license to increase the site limit (you only pay the difference)%2$s.', 'image-hopper' ), '<a href="https://imagehopper.tech/account/license-key/">', '</a>' );
				break;

			case 'missing':
			case 'invalid':
				$message = esc_html__( 'This license key is invalid. Please check your key has been entered correctly.', 'image-hopper' );

				break;

			case 'item_name_mismatch':
			case 'invalid_item_id':
				$message = esc_html__( 'This license key is not valid. Please check your key is for this product.', 'image-hopper' );
				break;

			default:
				$message = esc_html__( 'An error occurred, please try again.', 'image-hopper' );
				break;
		}

		return $message;
	}
}
