<?php

namespace ImageHopper\ImageHopper\FirstParty;

use ImageHopper\ImageHopper\Fields\ImageHopperField;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */

/**
 * @since 2.5.1
 */
class Dropbox {

	/**
	 * @var object $_instance If available, contains an instance of this class.
	 *
	 * @since 2.5.1
	 */
	private static $_instance = null;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return self $_instance An instance of this class.
	 *
	 * @since 2.5.1
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * @since 2.5.1
	 */
	public function init() {
		add_filter( 'gform_save_field_value', [ $this, 'fix_dropbox_urls_on_save' ], 10, 3 );

		add_filter( 'image_hopper_pre_get_value_save_entry', [ $this, 'fix_existing_dropbox_filenames_on_edit' ], 5, 3 );

		add_filter( 'gform_dropbox_should_upload_file', [ $this, 'prevent_reprocessing_of_files' ], 10, 7 );
		add_filter( 'image_hopper_field_input_type', [ $this, 'is_dropbox_request' ] );
	}

	/**
	 * Check if currently processing a Dropbox upload request and mask field as fileupload
	 *
	 * @param string $type
	 *
	 * @return string
	 *
	 * @since 2.13.6
	 */
	public function is_dropbox_request( $type ) {

		if ( ( is_admin() && \GFForms::get_page() === 'form_settings_gravityformsdropbox' ) ||
			 ( defined( 'DOING_AJAX' ) && DOING_AJAX && rgpost( 'action' ) === 'gform_dropbox_upload' ) ) {
			return 'fileupload';
		}

		return $type;
	}

	/**
	 * Convert Image Hopper Dropbox URLs to their raw form for use as embedded images
	 *
	 * @param string           $value The saved field value
	 * @param array            $entry
	 * @param ImageHopperField $field
	 *
	 * @return string
	 *
	 * @since 2.1
	 */
	public function fix_dropbox_urls_on_save( $value, $entry, $field ) {
		if ( $field->type === 'image_hopper' && ! empty( $value ) ) {
			$files = json_decode( $value, true );

			if ( is_array( $files ) ) {
				$files = array_map(
					static function( $file ) {
						/* Ignore if not the expected string URL */
						if ( ! is_string( $file ) ) {
							return $file;
						}

						if ( strpos( $file, 'www.dropbox.com' ) === false || strpos( $file, 'raw=1' ) !== false ) {
							  return $file;
						}

						return rawurldecode( add_query_arg( [ 'raw' => '1' ], $file ) );
					},
					$files
				);

				$value = wp_json_encode( $files, JSON_UNESCAPED_UNICODE );
			}
		}

		return $value;
	}

	/**
	 * Resolve filename sanitizing issue for existing Dropbox files
	 *
	 * When Gravity Forms populates \GFFormsModel::$uploaded_files(). it sanitizes the filenames.
	 * This causes issues when editing entries with Dropbox URLs, as they aren't picked up as existing
	 * uploaded files due to the URL parameters `?dl=0&raw=1` getting converted to `dl0raw1`, which would
	 * cause Image Hopper to delete those files from the entry object as it assumed the user removed the
	 * file from the Hopper.
	 *
	 * @param string $value
	 * @param array  $form
	 * @param string $input_name
	 *
	 * @return string
	 *
	 * @since 2.5.1
	 */
	public function fix_existing_dropbox_filenames_on_edit( $value, $form, $input_name ) {
		/* If not Dropbox URLs in this request, ignore */
		if ( strpos( $value, 'www.dropbox.com' ) === false ) {
			return $value;
		}

		if ( empty( \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] ) ) {
			return $value;
		}

		$files = json_decode( $value );

		foreach ( \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] as $key => $file ) {
			/* Skip new files */
			if ( ! empty( $file['temp_filename'] ) ) {
				continue;
			}

			/* Drop the sanitized arguments from the end of the filename */
			foreach ( $files as $url ) {
				$filename = wp_basename( $url );
				if ( $file['uploaded_filename'] === sanitize_file_name( $filename ) ) {
					\GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ][ $key ]['uploaded_filename'] = $filename;
					break;
				}
			}
		}

		return $value;
	}

	/**
	 * If IH field, check if the URL points to Dropbox and exclude. This resolves encoding issues in the URL.
	 *
	 * @param bool $should_upload_file
	 * @param string $url
	 * @param array $existing
	 * @param \GF_Field $field
	 *
	 * @return bool
	 *
	 * @since 2.5.1
	 */
	public function prevent_reprocessing_of_files( $should_upload_file, $url, $existing, $field ) {

		if ( $field->type === 'image_hopper' && strpos( $url, 'https://www.dropbox.com' ) === 0 ) {
			return false;
		}

		return $should_upload_file;
	}
}
