<?php

namespace ImageHopper\ImageHopper\Fields;

/**
 * @package     Image Hopper
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://opensource.org/licenses/gpl-3.0.php GNU Public License
 */
class ImageHopperPostField extends ImageHopperField {

	/**
	 * @var string
	 *
	 * @since 1.3.0
	 */
	public $type = 'image_hopper_post';

	/**
	 * @var int
	 *
	 * @since 1.3.0
	 */
	public $maxFiles = 1;

	/**
	 * ImageHopperPostField constructor.
	 *
	 * Mask this field as a `post_image` at specific points in the execution process
	 *
	 * @param array $data
	 */
	public function __construct( $data = [] ) {
		parent::__construct( $data );

		/* Ensure Gravity Forms saves the image when the Post is created */
		add_filter( 'gform_disable_post_creation', [ $this, 'mask_ih_post_field' ] );
		add_action( 'gform_after_create_post', [ $this, 'unmask_ih_post_field' ] );

		/* Add Stripe Payment Elements support so meta data is set */
		add_action( 'gform_stripe_payment_element_pre_process_submission', [ $this, 'mask_ih_post_field' ] );
		add_action( 'gform_stripe_payment_element_pre_process_form', [ $this, 'unmask_ih_post_field' ] );

		/* Add GravityView Edit Entry support, when required */
		if ( ! class_exists( '\GravityView_Edit_Entry' ) ) {
			return;
		}

		$gv_edit_entry        = \GravityView_Edit_Entry::getInstance();
		$gv_edit_entry_render = isset( $gv_edit_entry->instances['render'] ) ? $gv_edit_entry->instances['render'] : null;

		if ( ! $gv_edit_entry_render->is_edit_entry_submission() ) {
			return;
		}

		/* Ensure the correct image shows after GV Entry updated */
		add_filter( 'gform_field_input', [ $this, 'mask_ih_post_field' ], 9 );
		add_filter( 'gform_field_input', [ $this, 'unmask_ih_post_field' ], 11 );

		/* Ensure the image is automatically linked to the post on update */
		add_action( 'gform_after_update_entry', [ $this, 'mask_ih_post_field' ] );
		add_action( 'gravityview/edit_entry/after_update', [ $this, 'unmask_ih_post_field' ] );
	}

	/**
	 * @return \GF_Field_Post_Image
	 */
	protected function get_post_image_field() {
		return new \GF_Field_Post_Image( get_object_vars( $this ) );
	}

	/**
	 * Returns the field's form editor icon.
	 *
	 * This could be an icon url or a gform-icon class.
	 *
	 * @since 1.4
	 *
	 * @return string
	 */
	public function get_form_editor_field_icon() {
		return 'gform-icon--post-image';
	}

	/**
	 * Return the field title, for use in the form editor.
	 *
	 * @return string
	 *
	 * @since 1.3.0
	 */
	public function get_form_editor_field_title() {
		return esc_attr__( 'Image Hopper Post Image', 'image-hopper' );
	}

	/**
	 * Assign the field button to the Advanced Fields group.
	 *
	 * @return array
	 *
	 * since
	 */
	public function get_form_editor_button() {
		return [
			'group' => 'post_fields',
			'text'  => esc_attr__( 'IH Post Image', 'image-hopper' ),
		];
	}

	/**
	 * Pretend the field is a `fileupload` under specific circumstances, otherwise default to `image_hopper_post`
	 *
	 * @return string
	 *
	 * @since 1.3.0
	 */
	public function get_input_type() {
		/* If the type has been masked, return the masked version */
		if ( $this->type !== 'image_hopper_post' ) {
			return apply_filters( 'image_hopper_post_image_field_input_type', $this->type, $this );
		}

		/* If any of these actions, return fileupload */
		if (
			$this->is_fileupload_page() ||
			$this->is_form_submission()
		) {
			return apply_filters( 'image_hopper_post_image_field_input_type', 'fileupload', $this );
		}

		/* Return default value */
		$type = empty( $this->inputType ) ? $this->type : $this->inputType;

		return apply_filters( 'image_hopper_post_image_field_input_type', $type, $this );
	}

	/**
	 * The settings to include in the form editor.
	 *
	 * @return array
	 *
	 * @since 1.3.0
	 */
	public function get_form_editor_field_settings() {
		return apply_filters(
			'image_hopper_enabled_field_settings',
			[
				'conditional_logic_field_setting',
				'error_message_setting',
				'label_setting',
				'label_placement_setting',
				'admin_label_setting',
				'rules_setting',
				'file_size_setting',
				'visibility_setting',
				'description_setting',
				'css_class_setting',
				'image_hopper_max_image_sizes',
				'image_hopper_output_quality',
				'image_hopper_minimum_image_size',
				'image_hopper_minimum_image_size_warning',
				'image_hopper_upscale_field',
				'post_image_setting',
				'post_image_featured_image',
				'image_hopper_rename_image_field',
			],
			$this
		);
	}

	/**
	 * The scripts to be included in the form editor.
	 *
	 * @return string
	 *
	 * @since 1.3.0
	 */
	public function get_form_editor_inline_script_on_page_render() {
		return apply_filters(
			'image_hopper_post_inline_javascript',
			sprintf(
				/* phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents*/
				file_get_contents( __DIR__ . '/../../js/admin/formEditorInlineScriptPost.js' ),
				esc_attr__( 'Post Image', 'image-hopper' ),
				$this->allowedExtensions,
				esc_html__( 'Convert', 'image-hopper' ),
				esc_attr__( 'Convert this field to/from a File Upload and Image Hopper field', 'image-hopper' )
			)
		);
	}

	/**
	 * Define the fields inner markup.
	 *
	 * @param array        $form  The Form Object currently being processed.
	 * @param string|array $value The field value. From default/dynamic population, $_POST, or a resumed incomplete submission.
	 * @param null|array   $entry Null or the Entry Object currently being edited.
	 *
	 * @return string
	 *
	 * @since 1.3.0
	 */
	public function get_field_input( $form, $value = '', $entry = null ) {

		/* Prevent editing on Entry Details page */
		if ( $this->is_entry_detail_edit() ) {
			$post_id = rgar( $entry, 'post_id' );
			if ( is_numeric( $post_id ) ) {
				/* translators: %1$s is the start of an anchor HTML tag and %2$s is the end of an anchor tag */
				return '<div>' . sprintf( __( 'You can %1$sedit this post%2$s from the post page.', 'image-hopper' ), "<a href='post.php?action=edit&post=$post_id'>", '</a>' ) . '</div>';
			}
		}

		/* Reset max files to default (fixed issue when converting to and from the default field) */
		$this->maxFiles = 1;

		$is_entry_detail = $this->is_entry_detail();
		$is_form_editor  = $this->is_form_editor();
		$is_admin        = $is_entry_detail || $is_form_editor;

		$id       = (int) $this->id;
		$form_id  = (int) $form['id'];
		$field_id = $is_entry_detail || $is_form_editor || $form_id === 0 ? "input_$id" : 'input_' . $form_id . "_$id";

		$class_suffix  = $is_entry_detail ? '_admin' : '';
		$disabled_text = $is_form_editor ? 'disabled="disabled"' : '';

		/* aria attributes. */
		$form_sub_label_placement  = rgar( $form, 'subLabelPlacement' );
		$field_sub_label_placement = $this->subLabelPlacement;
		$is_sub_label_above        = $field_sub_label_placement === 'above' || ( empty( $field_sub_label_placement ) && $form_sub_label_placement === 'above' );

		// Prepare accepted extensions message.
		$allowed_extensions    = join( ',', \GFCommon::clean_extensions( explode( ',', strtolower( $this->allowedExtensions ) ) ) );
		$extensions_message_id = 'extensions_message_' . $form_id . '_' . $id;
		$extensions_message    = sprintf(
			"<span id='%s'>%s</span>",
			$extensions_message_id,
			/* translators: %s is a list of file types */
			esc_attr( sprintf( __( 'Accepted file types: %s.', 'image-hopper' ), str_replace( ',', ', ', $allowed_extensions ) ) )
		);

		$required_attribute = $this->isRequired ? 'aria-required="true"' : '';
		$invalid_attribute  = $this->failed_validation ? 'aria-invalid="true"' : 'aria-invalid="false"';
		$aria_describedby   = $this->get_aria_describedby( array( $extensions_message_id ) );

		/*
		 * Get saved data in the required format
		 *
		 * If a non-empty string, it's been saved to the database in the format `url|:|title|:|caption|:|description`
		 * If an array, it's during form validation failure and only the title, caption, and description are passed in
		 */
		$title       = '';
		$caption     = '';
		$description = '';
		$alt         = '';
		if ( is_string( $value ) && strpos( $value, '|:|' ) !== false ) {
			$field_value = explode( '|:|', $value );
			$url         = isset( $field_value[0] ) ? $field_value[0] : '';
			$title       = isset( $field_value[1] ) ? $field_value[1] : '';
			$caption     = isset( $field_value[2] ) ? $field_value[2] : '';
			$description = isset( $field_value[3] ) ? $field_value[3] : '';
			$alt         = isset( $field_value[4] ) ? $field_value[4] : '';

			$value = wp_json_encode( [ $url ], JSON_UNESCAPED_UNICODE );
		} elseif ( is_array( $value ) ) {
			$title       = esc_attr( rgget( $this->id . '.1', $value ) );
			$alt         = esc_attr( rgget( $this->id . '.2', $value ) );
			$caption     = esc_attr( rgget( $this->id . '.4', $value ) );
			$description = esc_attr( rgget( $this->id . '.7', $value ) );
			$value       = '';
		}

		/* hiding meta fields for admin */
		$hidden_style      = "style='display:none;'";
		$alt_style         = ! $this->displayAlt && $is_admin ? $hidden_style : '';
		$title_style       = ! $this->displayTitle && $is_admin ? $hidden_style : '';
		$caption_style     = ! $this->displayCaption && $is_admin ? $hidden_style : '';
		$description_style = ! $this->displayDescription && $is_admin ? $hidden_style : '';

		$upload = $this->build_field_input( $form, $entry, $value );

		$tabindex = $this->get_tabindex();
		if ( $is_sub_label_above ) {
			$alt_field = $this->displayAlt || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_alt gform-grid-col' $alt_style><label for='%s_2' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_alt', $form_id ], __( 'Alternative Text', 'image-hopper' ), $form_id ) . "</label><input type='text' name='input_%d.2' id='%s_2' value='%s' $tabindex %s/></span>", $field_id, $id, $field_id, $alt, $disabled_text ) : '';
		} else {
			$alt_field = $this->displayAlt || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_alt gform-grid-col' $alt_style><input type='text' name='input_%d.2' id='%s_2' value='%s' $tabindex %s/><label for='%s_2' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_alt', $form_id ], __( 'Alternative Text', 'image-hopper' ), $form_id ) . '</label></span>', $id, $field_id, $alt, $disabled_text, $field_id ) : '';
		}

		$tabindex = $this->get_tabindex();
		if ( $is_sub_label_above ) {
			$title_field = $this->displayTitle || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_title gform-grid-col' $title_style><label for='%s_1' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_title', $form_id ], __( 'Title', 'image-hopper' ), $form_id ) . "</label><input type='text' name='input_%d.1' id='%s_1' value='%s' $tabindex %s/></span>", $field_id, $id, $field_id, $title, $disabled_text ) : '';
		} else {
			$title_field = $this->displayTitle || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_title gform-grid-col' $title_style><input type='text' name='input_%d.1' id='%s_1' value='%s' $tabindex %s/><label for='%s_1' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_title', $form_id ], __( 'Title', 'image-hopper' ), $form_id ) . '</label></span>', $id, $field_id, $title, $disabled_text, $field_id ) : '';
		}

		$tabindex = $this->get_tabindex();
		if ( $is_sub_label_above ) {
			$caption_field = $this->displayCaption || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_caption gform-grid-col' $caption_style><label for='%s_4' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_caption', $form_id ], __( 'Caption', 'image-hopper' ), $form_id ) . "</label><input type='text' name='input_%d.4' id='%s_4' value='%s' $tabindex %s/></span>", $field_id, $id, $field_id, $caption, $disabled_text ) : '';
		} else {
			$caption_field = $this->displayCaption || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_caption gform-grid-col' $caption_style><input type='text' name='input_%d.4' id='%s_4' value='%s' $tabindex %s/><label for='%s_4' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_caption', $form_id ], __( 'Caption', 'image-hopper' ), $form_id ) . '</label></span>', $id, $field_id, $caption, $disabled_text, $field_id ) : '';
		}

		$tabindex = $this->get_tabindex();
		if ( $is_sub_label_above ) {
			$description_field = $this->displayDescription || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_description gform-grid-col' $description_style><label for='%s_7' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_description', $form_id ], __( 'Description', 'image-hopper' ), $form_id ) . "</label><input type='text' name='input_%d.7' id='%s_7' value='%s' $tabindex %s/></span>", $field_id, $id, $field_id, $description, $disabled_text ) : '';
		} else {
			$description_field = $this->displayDescription || $is_admin ? sprintf( "<span class='ginput_full$class_suffix ginput_post_image_description gform-grid-col' $description_style><input type='text' name='input_%d.7' id='%s_7' value='%s' $tabindex %s/><label for='%s_7' class='gform-field-label gform-field-label--type-sub'>" . gf_apply_filters( [ 'gform_postimage_description', $form_id ], __( 'Description', 'image-hopper' ), $form_id ) . '</label></span>', $id, $field_id, $description, $disabled_text, $field_id ) : '';
		}

		return sprintf(
			"<div class='ginput_complex ginput_container ginput_container_image_hopper ginput_container_image_hopper_post  gform-theme__no-reset--children'>%s</div>",
			$upload . $alt_field . $title_field . $caption_field . $description_field
		);
	}

	/**
	 * Filter out the previously uploaded images when handling form submission, and format per Post Image specification
	 *
	 * @param string $value
	 * @param array  $form
	 * @param string $input_name
	 * @param int    $lead_id
	 * @param array  $lead
	 *
	 * @return string
	 *
	 * @since 1.3.0
	 */
	public function get_value_save_entry( $value, $form, $input_name, $lead_id, $lead ) {
		/* Handle moving uploaded images to permanent storage */
		$is_save_and_continue = rgpost( 'gform_save' );
		if ( $this->is_form_submission() && ! $is_save_and_continue ) {
			$value = parent::get_value_save_entry( $value, $form, $input_name, $lead_id, $lead );
			$value = json_decode( $value, true );
			$url   = ! empty( $value[0] ) ? trim( $value[0] ) : '';
		} elseif ( $this->is_form_submission() && $is_save_and_continue ) {
			$file_path = \GFFormsModel::get_file_upload_path( $form['id'], $value );
			$url       = $file_path['url'];
		}

		/* If the URL is empty at this point, we will load it from the entry */
		if ( empty( $url ) ) {
			$url = explode( '|:|', rgar( $lead, (string) $this->id ) )[0];
		}

		if ( empty( $url ) || ! \GFCommon::is_valid_url( $url ) ) {
			return '';
		}

		/* phpcs:disable WordPress.Security.NonceVerification.Missing */
		$image_title       = isset( $_POST[ "{$input_name}_1" ] ) ? wp_strip_all_tags( $_POST[ "{$input_name}_1" ] ) : '';
		$image_caption     = isset( $_POST[ "{$input_name}_4" ] ) ? wp_strip_all_tags( $_POST[ "{$input_name}_4" ] ) : '';
		$image_description = isset( $_POST[ "{$input_name}_7" ] ) ? wp_strip_all_tags( $_POST[ "{$input_name}_7" ] ) : '';
		$image_alt         = isset( $_POST[ "{$input_name}_2" ] ) ? wp_strip_all_tags( $_POST[ "{$input_name}_2" ] ) : '';
		/* phpcs:enable */

		return $url . '|:|' . $image_title . '|:|' . $image_caption . '|:|' . $image_description . '|:|' . $image_alt;
	}

	/**
	 * Masquerade as the Post Image field
	 *
	 * @param string $value
	 *
	 * @return mixed
	 *
	 * @since 1.3.0
	 */
	public function mask_ih_post_field( $value = '' ) {
		$this->type = 'post_image';

		return $value;
	}

	/**
	 * Un-masquerade the field
	 *
	 * @param string $value
	 *
	 * @return mixed
	 *
	 * @since 1.3.0
	 */
	public function unmask_ih_post_field( $value = '' ) {
		$this->type = 'image_hopper_post';

		return $value;
	}

	/**
	 * If a validation error on the GravityView Edit Entry page, apply a filter to correct the IH field output
	 *
	 * @param array $results
	 *
	 * @return array
	 *
	 * @since 1.3.0
	 * @depecated 2.16.1
	 */
	public function gravityview_fix_ih_post_field_value( $results ) {
		return $results;
	}

	/**
	 * Fix the IH field display during validation failure
	 *
	 * @param $field_value
	 * @param $field
	 * @param $gv_edit_entry_render
	 *
	 * @return array
	 *
	 * @since 1.3.0
	 * @depecated 2.16.1
	 */
	public function gravityview_edit_field_value( $field_value, $field, $gv_edit_entry_render ) {
		return $field_value;
	}

	/**
	 * If a newly-updated file during GravityView's Edit Entry submission, mask the $_FILES superglobal so it triggers
	 * the GV functionality to add the image to the Media Library
	 *
	 * @param array $form
	 *
	 * @since 1.3.0
	 */
	public function gravityview_mask_ih_post_field_file_superglobal( $form ) {
		$uploads = json_decode( stripslashes( \GFForms::post( 'gform_uploaded_files' ) ), true );
		if ( $uploads === null ) {
			return;
		}

		$ih_post_fields = \GFFormsModel::get_fields_by_type( $form, 'image_hopper_post' );

		foreach ( $ih_post_fields as $field ) {
			$input_name = 'input_' . $field->id;

			if ( ! is_array( $uploads[ $input_name ] ) ) {
				continue;
			}

			/* Check for a newly-uploaded Post image */
			$new_post_upload = array_filter(
				$uploads[ $input_name ],
				static function( $file ) {
					return ! empty( $file['temp_filename'] );
				}
			);

			if ( count( $new_post_upload ) > 0 ) {
				/* Ensure GravityView uploads to the media library */
				$_FILES[ $input_name ] = [
					'name' => 'mark as new file',
				];

				/* Reset the file store to resolve issues with the Image Editor */
				$uploads[ $input_name ] = $new_post_upload;
			}
		}

		$_POST['gform_uploaded_files'] = wp_json_encode( $uploads, JSON_UNESCAPED_UNICODE );
	}

	/**
	 * @param string $value
	 * @param array  $entry
	 * @param string $field_id
	 * @param array  $columns
	 * @param array  $form
	 *
	 * @return string
	 *
	 * @since 1.3
	 */
	public function get_value_entry_list( $value, $entry, $field_id, $columns, $form ) {
		return $this->get_post_image_field()->get_value_entry_list( $value, $entry, $field_id, $columns, $form );
	}

	/**
	 * @param array|string $value
	 * @param string       $currency
	 * @param false        $use_text
	 * @param string       $format
	 * @param string       $media
	 *
	 * @return string
	 *
	 * @since 1.3
	 */
	public function get_value_entry_detail( $value, $currency = '', $use_text = false, $format = 'html', $media = 'screen' ) {
		$value = $this->get_post_image_field()->get_value_entry_detail( $value, $currency, $use_text, $format, $media );

		$value = str_replace( "width='100'", 'style="max-width:100%"', $value );

		return $value;
	}

	/**
	 * @param array $field_values
	 * @param bool  $get_from_post_global_var
	 *
	 * @return array
	 *
	 * @since 1.3
	 */
	public function get_value_submission( $field_values, $get_from_post_global_var = true ) {
		return $this->get_post_image_field()->get_value_submission( $field_values, $get_from_post_global_var );
	}

	/**
	 * @param array|string $value
	 * @param string       $input_id
	 * @param array        $entry
	 * @param array        $form
	 * @param string       $modifier
	 * @param array|string $raw_value
	 * @param bool         $url_encode
	 * @param bool         $esc_html
	 * @param string       $format
	 * @param bool         $nl2br
	 *
	 * @return string
	 *
	 * @since 1.3
	 */
	public function get_value_merge_tag( $value, $input_id, $entry, $form, $modifier, $raw_value, $url_encode, $esc_html, $format, $nl2br ) {
		return $this->get_post_image_field()->get_value_merge_tag( $value, $input_id, $entry, $form, $modifier, $raw_value, $url_encode, $esc_html, $format, $nl2br );
	}

	/**
	 * @param array  $entry
	 * @param string $input_id
	 * @param false  $use_text
	 * @param false  $is_csv
	 *
	 * @return string|null
	 */
	public function get_value_export( $entry, $input_id = '', $use_text = false, $is_csv = false ) {
		if ( empty( $input_id ) ) {
			$input_id = $this->id;
		}

		return rgar( $entry, $input_id );
	}

	/**
	 * Stores the physical file paths as extra entry meta data.
	 *
	 * @since 2.4.0
	 *
	 * @param array $form  The form object being saved.
	 * @param array $entry The entry object being saved.
	 *
	 * @return array The array that contains the file URLs and their corresponding physical paths.
	 */
	public function get_extra_entry_metadata( $form, $entry ) {

		// Leave only the file URL in the entry value so when parent saves the file path information the URL is a valid file URL.
		$post_image_info     = explode( '|:|', rgget( $this->id, $entry ) );
		$entry[ $this->id ]  = rgar( $post_image_info, 0 );
		$this->multipleFiles = false;

		return parent::get_extra_entry_metadata( $form, $entry );
	}
}
