<?php

namespace ImageHopper\ImageHopperEditor;

use ImageHopper\ImageHopper\Licensing\EDD_SL_Plugin_Updater;

/**
 * @package     Image Hopper Editor
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */

/**
 * Class ImageHopperAddOn
 *
 * @package ImageHopper\ImageHopperEditor
 *
 * @since   1.0
 */
class Editor {

	/**
	 * @var object $_instance If available, contains an instance of this class.
	 *
	 * @since 1.0.0
	 */
	private static $_instance = null;

	/**
	 * Returns an instance of this class, and stores it in the $_instance property.
	 *
	 * @return object $_instance An instance of this class.
	 *
	 * @since 1.0.0
	 */
	public static function get_instance() {
		if ( self::$_instance === null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * Runs on the WordPress "init" hook
	 *
	 * @since 1.0.0
	 */
	public function init() {
		add_action( 'image_hopper_plugin_updater', [ $this, 'setup_plugin_updater' ], 10, 2 );
		add_filter( 'image_hopper_input_field', [ $this, 'inject_field_settings' ], 10, 5 );
		add_filter( 'image_hopper_pre_get_value_save_entry', [ $this, 'override_image_on_update' ], 10, 5 );

		add_filter( 'image_hopper_scripts', [ $this, 'scripts' ] );
		add_filter( 'image_hopper_styles', [ $this, 'styles' ] );
	}

	/**
	 * Runs on the WordPress "init_admin" hook
	 *
	 * @since 1.0.0
	 */
	public function init_admin() {
		/* Register new Field editor settings for Image Hopper */
		add_action( 'gform_field_standard_settings', [ $this, 'field_enable_editor_settings' ], 10, 2 );
		add_action( 'gform_field_appearance_settings', [ $this, 'field_enable_editor_appearance_settings' ], 10, 2 );

		add_filter( 'image_hopper_inline_javascript', [ $this, 'load_editor_inline_javascript' ] );

		add_filter( 'image_hopper_enabled_field_settings', [ $this, 'register_editor_field_settings' ] );

		/* Load the Crop and Upscale features if pre Image Hopper 1.5 */
		if ( version_compare( IMAGE_HOPPER_ADDON_VERSION, '1.5.0', '<' ) ) {
			add_action( 'image_hopper_field_max_image_sizes_post_display', [ $this, 'add_crop_field_setting' ] );
			add_action( 'gform_field_advanced_settings', [ $this, 'add_upscale_field_setting' ], 10, 2 );
			add_filter( 'gform_tooltips', [ $this, 'add_tooltips' ] );
		}
	}

	/**
	 * Register our field JavaScript
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function scripts( $scripts ) {

		/* Consolidate the strings into one localized global */
		foreach ( $scripts as &$script ) {
			if ( $script['handle'] === 'image_hopper_js' ) {
				$script['strings'] = array_merge( $script['strings'], $this->get_strings() );

				break;
			}
		}

		$scripts[] = [
			'handle'  => 'image_hopper_editor_js',
			'src'     => plugins_url( '', IMAGE_HOPPER_EDITOR_ADDON_FILE ) . '/dist/image-hopper-editor.js',
			'version' => defined( 'WP_DEBUG' ) && WP_DEBUG === true ? time() : IMAGE_HOPPER_EDITOR_ADDON_VERSION,
			'deps'    => [ 'image_hopper_js' ],
			'enqueue' => [
				[
					'field_types' => [
						'image_hopper',
						'image_hopper_post',
					],
				],
			],
		];

		return $scripts;
	}

	/**
	 * Set the module tag to load the modern JS
	 *
	 * @param string $tag The generated script tag
	 * @param string $handle The JS registered ID
	 *
	 * @return string
	 *
	 * @since 2.0
	 *
	 * @deprecated 2.6.3. Including as a module breaks CDNs due to CORS
	 */
	public function register_javascript_module( $tag, $handle ) {
		switch ( $handle ) {
			case 'image_hopper_editor_js':
				$tag = str_replace( '<script ', '<script type="module" ', $tag );
				break;
		}

		return $tag;
	}

	/**
	 * @return array
	 *
	 * @since 2.0
	 */
	protected function get_strings() {
		return [
			'editorL10n' => [
				/* Core */
				/* translators: reset the image back to its original state */
				'labelReset'                             => esc_html__( 'Reset', 'image-hopper-editor' ),
				'labelDefault'                           => esc_html__( 'Default', 'image-hopper-editor' ),
				/* translators: change the image editor mode to automatic */
				'labelAuto'                              => esc_html__( 'Auto', 'image-hopper-editor' ),
				'labelNone'                              => esc_html__( 'None', 'image-hopper-editor' ),
				/* translators: edit the image */
				'labelEdit'                              => esc_html__( 'Edit', 'image-hopper-editor' ),
				/* translators: close the image editor */
				'labelClose'                             => esc_html__( 'Close', 'image-hopper-editor' ),
				/* translators: %s will be replaced by a feature which isn't supported in the current web browser */
				'labelSupportError'                      => esc_html__( '%s not supported in this browser', 'image-hopper-editor' ),

				/* defaults */
				'labelColor'                             => esc_html__( 'Color', 'image-hopper-editor' ),
				'labelWidth'                             => esc_html__( 'Width', 'image-hopper-editor' ),
				'labelSize'                              => esc_html__( 'Size', 'image-hopper-editor' ),
				'labelOffset'                            => esc_html__( 'Offset', 'image-hopper-editor' ),
				'labelAmount'                            => esc_html__( 'Amount', 'image-hopper-editor' ),
				'labelInset'                             => esc_html__( 'Inset', 'image-hopper-editor' ),
				'labelRadius'                            => esc_html__( 'Radius', 'image-hopper-editor' ),

				/* translators: different font sizes */
				'labelSizeExtraSmall'                    => esc_html__( 'Extra small', 'image-hopper-editor' ),
				/* translators: different font sizes */
				'labelSizeSmall'                         => esc_html__( 'Small', 'image-hopper-editor' ),
				/* translators: different font sizes */
				'labelSizeMediumSmall'                   => esc_html__( 'Medium small', 'image-hopper-editor' ),
				/* translators: different font sizes */
				'labelSizeMedium'                        => esc_html__( 'Medium', 'image-hopper-editor' ),
				/* translators: different font sizes */
				'labelSizeMediumLarge'                   => esc_html__( 'Medium large', 'image-hopper-editor' ),
				/* translators: different font sizes */
				'labelSizeLarge'                         => esc_html__( 'Large', 'image-hopper-editor' ),
				/* translators: different font sizes */
				'labelSizeExtraLarge'                    => esc_html__( 'Extra large', 'image-hopper-editor' ),

				'labelButtonRevert'                      => esc_html__( 'Revert', 'image-hopper-editor' ),
				'labelButtonCancel'                      => esc_html__( 'Cancel', 'image-hopper-editor' ),
				'labelButtonUndo'                        => esc_html__( 'Undo', 'image-hopper-editor' ),
				'labelButtonRedo'                        => esc_html__( 'Redo', 'image-hopper-editor' ),
				'labelButtonExport'                      => esc_html__( 'Done', 'image-hopper-editor' ),

				'statusLabelButtonClose'                 => esc_html__( 'Close', 'image-hopper-editor' ),
				'statusLabelLoadImage'                   => esc_html__( 'Creating preview…', 'image-hopper-editor' ),
				/* translators: tells the user the software is ready for a new image to be uploaded */
				'statusLabelLoadImageWaiting'            => esc_html__( 'Waiting for image', 'image-hopper-editor' ),
				/* translators: a message shown when an image size is too small. %1$s is the minimum width and %2$s is the minimum height (both in pixels) */
				'statusLabelLoadImageImageTooSmall'      => sprintf( esc_html__( 'Minimum size is %1$s × %2$spx', 'image-hopper-editor' ), '{minWidth}', '{minHeight}' ),
				'statusLabelLoadImageError'              => esc_html__( 'Error loading image', 'image-hopper-editor' ),
				'statusLabelLoadImageCreatingPreview'    => esc_html__( 'Loading image…', 'image-hopper-editor' ),

				'statusLabelProcessImage'                => esc_html__( 'Processing image…', 'image-hopper-editor' ),
				'statusLabelProcessImageErrorProcessing' => esc_html__( 'Error processing image', 'image-hopper-editor' ),
				'statusLabelProcessImageUploading'       => esc_html__( 'Uploading image…', 'image-hopper-editor' ),
				'statusLabelProcessImageErrorUploading'  => esc_html__( 'Error uploading image', 'image-hopper-editor' ),

				/* Shapes */
				'shapeLabelButtonSelectSticker'          => esc_html__( 'Select image', 'image-hopper-editor' ),
				'shapeTitleButtonTextLayoutAutoWidth'    => esc_html__( 'Auto width', 'image-hopper-editor' ),
				'shapeTitleButtonTextLayoutAutoHeight'   => esc_html__( 'Auto height', 'image-hopper-editor' ),
				'shapeTitleButtonTextLayoutFixedSize'    => esc_html__( 'Fixed size', 'image-hopper-editor' ),

				'shapeTitleButtonFlipHorizontal'         => esc_html__( 'Flip horizontal', 'image-hopper-editor' ),
				'shapeTitleButtonFlipVertical'           => esc_html__( 'Flip vertical', 'image-hopper-editor' ),
				'shapeTitleButtonRemove'                 => esc_html__( 'Remove', 'image-hopper-editor' ),
				'shapeTitleButtonDuplicate'              => esc_html__( 'Duplicate', 'image-hopper-editor' ),
				/* translators: move a shape in front of all other shapes */
				'shapeTitleButtonMoveToFront'            => esc_html__( 'Move to front', 'image-hopper-editor' ),

				'shapeLabelInputText'                    => esc_html__( 'Edit text', 'image-hopper-editor' ),

				'shapeLabelInputCancel'                  => esc_html__( 'Cancel', 'image-hopper-editor' ),
				'shapeLabelInputConfirm'                 => esc_html__( 'Confirm', 'image-hopper-editor' ),

				/* translators: disable a shapes stroke/outline */
				'shapeLabelStrokeNone'                   => esc_html__( 'No outline', 'image-hopper-editor' ),

				/* translators: Normal/Bold/Italic/Bold Italic is in the context of a font */
				'shapeLabelFontStyleNormal'              => esc_html__( 'Normal', 'image-hopper-editor' ),
				/* translators: Normal/Bold/Italic/Bold Italic is in the context of a font */
				'shapeLabelFontStyleBold'                => esc_html__( 'Bold', 'image-hopper-editor' ),
				/* translators: Normal/Bold/Italic/Bold Italic is in the context of a font */
				'shapeLabelFontStyleItalic'              => esc_html__( 'Italic', 'image-hopper-editor' ),
				/* translators: Normal/Bold/Italic/Bold Italic is in the context of a font */
				'shapeLabelFontStyleItalicBold'          => esc_html__( 'Bold Italic', 'image-hopper-editor' ),

				'shapeTitleBackgroundColor'              => esc_html__( 'Fill color', 'image-hopper-editor' ),

				'shapeTitleFontFamily'                   => esc_html__( 'Font', 'image-hopper-editor' ),
				'shapeTitleFontSize'                     => esc_html__( 'Font size', 'image-hopper-editor' ),
				'shapeTitleFontStyle'                    => esc_html__( 'Font style', 'image-hopper-editor' ),
				/* translators: text line height control */
				'shapeTitleLineHeight'                   => esc_html__( 'Leading', 'image-hopper-editor' ),

				/* translators: Start and End refer to each end of a drawn line */
				'shapeTitleLineStart'                    => esc_html__( 'Start', 'image-hopper-editor' ),
				/* translators: Start and End refer to each end of a drawn line */
				'shapeTitleLineEnd'                      => esc_html__( 'End', 'image-hopper-editor' ),
				'shapeTitleStrokeWidth'                  => esc_html__( 'Line width', 'image-hopper-editor' ),
				'shapeTitleStrokeColor'                  => esc_html__( 'Line color', 'image-hopper-editor' ),

				/* translators: the following are drawing tool used when annotating/decorating an image: Bar, Circle, Square, Arrow */
				'shapeTitleLineDecorationBar'            => esc_html__( 'Bar', 'image-hopper-editor' ),
				/* translators: the following are drawing tool used when annotating/decorating an image: Bar, Circle, Square, Arrow */
				'shapeTitleLineDecorationCircle'         => esc_html__( 'Circle', 'image-hopper-editor' ),
				/* translators: the following are drawing tool used when annotating/decorating an image: Bar, Circle, Square, Arrow */
				'shapeTitleLineDecorationSquare'         => esc_html__( 'Square', 'image-hopper-editor' ),
				/* translators: the following are drawing tool used when annotating/decorating an image: Bar, Circle, Square, Arrow */
				'shapeTitleLineDecorationArrow'          => esc_html__( 'Arrow', 'image-hopper-editor' ),
				/* translators: refer to enabling or disabling fill/background color for the shapes */
				'shapeTitleLineDecorationCircleSolid'    => esc_html__( 'Circle solid', 'image-hopper-editor' ),
				/* translators: refer to enabling or disabling fill/background color for the shapes */
				'shapeTitleLineDecorationSquareSolid'    => esc_html__( 'Square solid', 'image-hopper-editor' ),
				/* translators: refer to enabling or disabling fill/background color for the shapes */
				'shapeTitleLineDecorationArrowSolid'     => esc_html__( 'Arrow solid', 'image-hopper-editor' ),

				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorTransparent'             => esc_html__( 'Transparent', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorWhite'                   => esc_html__( 'White', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorSilver'                  => esc_html__( 'Silver', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorGray'                    => esc_html__( 'Gray', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorBlack'                   => esc_html__( 'Black', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorNavy'                    => esc_html__( 'Navy', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorBlue'                    => esc_html__( 'Blue', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorAqua'                    => esc_html__( 'Aqua', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorTeal'                    => esc_html__( 'Teal', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorOlive'                   => esc_html__( 'Olive', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorGreen'                   => esc_html__( 'Green', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorYellow'                  => esc_html__( 'Yellow', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorOrange'                  => esc_html__( 'Orange', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorRed'                     => esc_html__( 'Red', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorMaroon'                  => esc_html__( 'Maroon', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorFuchsia'                 => esc_html__( 'Fuchsia', 'image-hopper-editor' ),
				/* translators: the following are the names of colors or in the context of color management: Transparent, White, Silver, Gray, Black, Navy, Blue, Aqua, Teal, Olive, Green, Yellow, Orange, Red, Maroon, Fuchsia, Purple */
				'shapeTitleColorPurple'                  => esc_html__( 'Purple', 'image-hopper-editor' ),

				'shapeTitleTextColor'                    => esc_html__( 'Font color', 'image-hopper-editor' ),
				'shapeTitleTextAlign'                    => esc_html__( 'Text align', 'image-hopper-editor' ),
				'shapeTitleTextAlignLeft'                => esc_html__( 'Left align text', 'image-hopper-editor' ),
				'shapeTitleTextAlignCenter'              => esc_html__( 'Center align text', 'image-hopper-editor' ),
				'shapeTitleTextAlignRight'               => esc_html__( 'Right align text', 'image-hopper-editor' ),

				/* translators: the following are drawing tool used when annotating/decorating an image: Sharpie, Rectangle, Ellipse, Arrow, Line, Text, Stickers, and Eraser */
				'shapeLabelToolSharpie'                  => esc_html__( 'Sharpie', 'image-hopper-editor' ),
				/* translators: a drawing tool which will delete any shape added */
				'shapeLabelToolEraser'                   => esc_html__( 'Eraser', 'image-hopper-editor' ),
				/* translators: a drawing tool which adds a rectangle/square to an image */
				'shapeLabelToolRectangle'                => esc_html__( 'Rectangle', 'image-hopper-editor' ),
				/* translators: a drawing tool that adds an ellipse or circle to an image */
				'shapeLabelToolEllipse'                  => esc_html__( 'Ellipse', 'image-hopper-editor' ),
				/* translators: a drawing tool which adds an arrow to an image */
				'shapeLabelToolArrow'                    => esc_html__( 'Arrow', 'image-hopper-editor' ),
				/* translators: a drawing tool that adds a line to an image */
				'shapeLabelToolLine'                     => esc_html__( 'Line', 'image-hopper-editor' ),
				/* translators: a drawing tool that adds text/text box to an image */
				'shapeLabelToolText'                     => esc_html__( 'Text', 'image-hopper-editor' ),
				/* translators: a drawing tool which places an image on top of another image */
				'shapeLabelToolPreset'                   => esc_html__( 'Stickers', 'image-hopper-editor' ),

				/* Annotate */
				/* translators: draw or add shapes directly on the image eg. the image could be flipped and any decorated shapes will be flipped as well */
				'annotateLabel'                          => esc_html__( 'Annotate', 'image-hopper-editor' ),

				/* Decorate */
				/* translators: draw or add shapes on a new layer above the image eg. the image could be flipped and any decorated shapes will remain in the same position */
				'decorateLabel'                          => esc_html__( 'Decorate', 'image-hopper-editor' ),

				/* Stickers */
				/* translators: an image placed on top of another image */
				'stickerLabel'                           => esc_html__( 'Sticker', 'image-hopper-editor' ),

				/* Crop */
				/* translators: image cropping controls */
				'cropLabel'                              => esc_html__( 'Crop', 'image-hopper-editor' ),
				/* translators: image cropping controls that recenters the image on the page after a crop has taken place */
				'cropLabelButtonRecenter'                => esc_html__( 'Recenter', 'image-hopper-editor' ),
				/* translators: image cropping control */
				'cropLabelButtonRotateLeft'              => esc_html__( 'Rotate left', 'image-hopper-editor' ),
				/* translators: image cropping control */
				'cropLabelButtonRotateRight'             => esc_html__( 'Rotate right', 'image-hopper-editor' ),
				/* translators: image cropping control */
				'cropLabelButtonFlipHorizontal'          => esc_html__( 'Flip horizontal', 'image-hopper-editor' ),
				/* translators: image cropping control */
				'cropLabelButtonFlipVertical'            => esc_html__( 'Flip vertical', 'image-hopper-editor' ),
				/* translators: image cropping control */
				'cropLabelSelectPreset'                  => esc_html__( 'Crop shape', 'image-hopper-editor' ),
				/* translators: image cropping control which will allow cropping outside of the current image dimensions */
				'cropLabelCropBoundary'                  => esc_html__( 'Crop boundary', 'image-hopper-editor' ),
				/* translators: image cropping control which prevents the image being cropped outside its dimensions */
				'cropLabelCropBoundaryEdge'              => esc_html__( 'Edge of image', 'image-hopper-editor' ),
				/* translators: disable an image editing feature */
				'cropLabelCropBoundaryNone'              => esc_html__( 'None', 'image-hopper-editor' ),
				/* translators: image cropping control */
				'cropLabelTabRotation'                   => esc_html__( 'Rotation', 'image-hopper-editor' ),
				/* translators: image cropping control */
				'cropLabelTabZoom'                       => esc_html__( 'Zoom', 'image-hopper-editor' ),

				/* Filter */
				/* translators: photo filter controls (like Instagram) */
				'filterLabel'                            => esc_html__( 'Filter', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelChrome'                      => esc_html__( 'Chrome', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelFade'                        => esc_html__( 'Fade', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelCold'                        => esc_html__( 'Cold', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelWarm'                        => esc_html__( 'Warm', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelPastel'                      => esc_html__( 'Pastel', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelMonoDefault'                 => esc_html__( 'Mono', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelMonoNoir'                    => esc_html__( 'Noir', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelMonoWash'                    => esc_html__( 'Wash', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelMonoStark'                   => esc_html__( 'Stark', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelSepiaDefault'                => esc_html__( 'Sepia', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelSepiaBlues'                  => esc_html__( 'Blues', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelSepiaRust'                   => esc_html__( 'Rust', 'image-hopper-editor' ),
				/* translators: the name of a photo filter */
				'filterLabelSepiaColor'                  => esc_html__( 'Color', 'image-hopper-editor' ),

				/* Finetune */
				/* translators: finetune = standard image adjustment controls like brightness/contrast/exposure */
				'finetuneLabel'                          => esc_html__( 'Finetune', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelBrightness'                => esc_html__( 'Brightness', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelContrast'                  => esc_html__( 'Contrast', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelSaturation'                => esc_html__( 'Saturation', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelExposure'                  => esc_html__( 'Exposure', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelTemperature'               => esc_html__( 'Temperature', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelGamma'                     => esc_html__( 'Gamma', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelClarity'                   => esc_html__( 'Clarity', 'image-hopper-editor' ),
				/* translators: a finetune control setting */
				'finetuneLabelVignette'                  => esc_html__( 'Vignette', 'image-hopper-editor' ),

				/* Resize */
				'resizeLabel'                            => esc_html__( 'Resize', 'image-hopper-editor' ),
				'resizeLabelFormCaption'                 => esc_html__( 'Image output size', 'image-hopper-editor' ),
				/* translators: shorthand for the image width in pixels */
				'resizeLabelInputWidth'                  => esc_html__( 'w', 'image-hopper-editor' ),
				/* translators: the image's width */
				'resizeTitleInputWidth'                  => esc_html__( 'Width', 'image-hopper-editor' ),
				/* translators: shorthand for the image height in pixels */
				'resizeLabelInputHeight'                 => esc_html__( 'h', 'image-hopper-editor' ),
				/* translators: the image's height */
				'resizeTitleInputHeight'                 => esc_html__( 'Height', 'image-hopper-editor' ),
				'resizeTitleButtonMaintainAspectRatio'   => esc_html__( 'Maintain aspect ratio', 'image-hopper-editor' ),

				/* Frame */
				/* translators: Picture Frames for images */
				'frameLabel'                             => esc_html__( 'Frame', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelMatSharp'                     => esc_html__( 'Mat', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelMatRound'                     => esc_html__( 'Bevel', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelLineSingle'                   => esc_html__( 'Line', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelLineMultiple'                 => esc_html__( 'Zebra', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelEdgeSeparate'                 => esc_html__( 'Inset', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelEdgeOverlap'                  => esc_html__( 'Plus', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelEdgeCross'                    => esc_html__( 'Lumber', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelCornerHooks'                  => esc_html__( 'Hook', 'image-hopper-editor' ),
				/* translators: the name of the image picture frame style */
				'frameLabelPolaroid'                     => esc_html__( 'Polaroid', 'image-hopper-editor' ),

				/* Redact */
				'redactLabel'                            => esc_html__( 'Redact', 'image-hopper-editor' ),
			],
		];
	}

	/**
	 * Register our field CSS
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public function styles( $styles ) {
		$styles[] = [
			'handle'  => 'image_hopper_editor_css',
			'src'     => plugins_url( '', IMAGE_HOPPER_EDITOR_ADDON_FILE ) . '/dist/editor.style.build.css',
			'version' => defined( 'WP_DEBUG' ) && WP_DEBUG === true ? time() : IMAGE_HOPPER_EDITOR_ADDON_VERSION,
			'enqueue' => [
				[
					'field_types' => [
						'image_hopper',
						'image_hopper_post',
					],
				],
			],
		];

		return $styles;
	}

	/**
	 * Setup EDD updater class for the plugin
	 *
	 * @param string $license_key
	 * @param string $store_url
	 *
	 * @since 1.0
	 */
	public function setup_plugin_updater( $license_key, $store_url ) {
		new EDD_SL_Plugin_Updater(
			$store_url,
			IMAGE_HOPPER_EDITOR_ADDON_FILE,
			[
				'version'   => IMAGE_HOPPER_EDITOR_ADDON_VERSION,
				'license'   => $license_key,
				'item_name' => 'Image Hopper Editor',
				'author'    => 'Image Hopper',
			]
		);
	}

	/**
	 * Add additional settings to the Image Hopper field
	 *
	 * @param array $settings
	 *
	 * @return array
	 *
	 * @since 1.0
	 */
	public function register_editor_field_settings( $settings ) {
		$settings = array_merge(
			$settings,
			[
				'image_hopper_enable_editor',
				'image_hopper_editor_instant_edit',
				'image_hopper_editor_features',
				'image_hopper_editor_individual_features',
				'image_hopper_editor_default_screen',
				'image_hopper_upscale_field',
				'image_hopper_default_filter',
				'image_hopper_editor_appearance',
				'image_hopper_editor_default_markup_tool',
				'image_hopper_editor_default_markup_color',
			]
		);

		return $settings;
	}

	/**
	 * Include additional inline JavaScript for the IH field in the form editor
	 *
	 * @param string $js
	 *
	 * @return string
	 *
	 * @since 1.0
	 */
	public function load_editor_inline_javascript( $js ) {
		if ( version_compare( IMAGE_HOPPER_ADDON_VERSION, '1.5.0', '<' ) ) {
			/* phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents */
			return $js . file_get_contents( __DIR__ . '/../js/admin/formEditorInlineScriptPreIH1.5.js' );
		}

		/* phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents */
		return $js . file_get_contents( __DIR__ . '/../js/admin/formEditorInlineScript.js' );
	}

	/**
	 * Add field settings tooltip(s)
	 *
	 * @param array $tooltips
	 *
	 * @return array
	 *
	 * @since 1.0
	 */
	public function add_tooltips( $tooltips ) {
		$tooltips['ih_crop_to_dimensions'] = '<h6>' . esc_html__( 'Crop to Dimensions', 'image-hopper-editor' ) . '</h6>' . esc_html__( 'When enabled, the image will be cropped to the width and height specified (unless the image is already smaller, in which case it will be constrained to the aspect ratio). If you enable the Image Editor, the crop box will be locked to the same aspect ratio.', 'image-hopper-editor' );

		$tooltips['ih_upscale_image'] = '<h6>' . esc_html__( 'Upscale Image', 'image-hopper-editor' ) . '</h6>' . esc_html__( 'When enabled, images smaller than the crop size dimensions will automatically upscale to match. This ensures uniformity for all images, but does effect the overall image quality when upscaling occurs.', 'image-hopper-editor' );

		return $tooltips;
	}

	/**
	 * Output the mark-up for the Editor Fields
	 *
	 * @param int $position The position the settings should be located at.
	 * @param int $form_id  The ID of the form currently being edited.
	 *
	 * @since 1.0.0
	 */
	public function field_enable_editor_settings( $position, $form_id ) {
		if ( $position === 1600 ) {
			?>
			<li class="image_hopper_enable_editor field_setting">
				<label class="section_label"><?php esc_html_e( 'Editor', 'image-hopper-editor' ); ?></label>

				<input type="checkbox" id="field_enable_editor" onclick="SetFieldProperty('enableEditor', this.checked); ToggleEditorFields(this.checked)" onkeypress="SetFieldProperty('enableEditor', this.checked); ToggleEditorFields(this.checked)" />
				<label for="field_enable_editor" class="inline"><?php esc_html_e( 'Enable Image Editor', 'image-hopper-editor' ); ?></label>
			</li>

			<li class="image_hopper_editor_instant_edit field_setting">
				<input type="checkbox" id="field_instant_edit" onclick="SetFieldProperty('instantEdit', this.checked);" onkeypress="SetFieldProperty('instantEdit', this.checked);" />
				<label for="field_instant_edit" class="inline"><?php esc_html_e( 'Automatically open the editor on upload', 'image-hopper-editor' ); ?></label>
			</li>

			<li class="image_hopper_editor_features field_setting">
				<label for="editor_features_container" class="section_label">
					<?php esc_html_e( 'Features', 'image-hopper-editor' ); ?>
				</label>

				<div id="editor_features_container">
					<input type="radio" id="editor_features_presets_essentials" name="editor_features_presets" value="essentials" onclick="ToggleFeatures(this.value);" onkeypress="ToggleFeatures(this.value);" />
					<label for="editor_features_presets_essentials" class="inline"><?php esc_html_e( 'Essentials', 'image-hopper-editor' ); ?></label>

					<input type="radio" id="editor_features_presets_color" name="editor_features_presets" value="color" onclick="ToggleFeatures(this.value);" onkeypress="ToggleFeatures(this.value);" />
					<label for="editor_features_presets_color" class="inline"><?php esc_html_e( 'Color', 'image-hopper-editor' ); ?></label>

					<input type="radio" id="editor_features_presets_annotate" name="editor_features_presets" value="annotate" onclick="ToggleFeatures(this.value);" onkeypress="ToggleFeatures(this.value);" />
					<label for="editor_features_presets_annotate" class="inline"><?php esc_html_e( 'Annotate', 'image-hopper-editor' ); ?></label>

					<input type="radio" id="editor_features_presets_all" name="editor_features_presets" value="all" onclick="ToggleFeatures(this.value);" onkeypress="ToggleFeatures(this.value);" />
					<label for="editor_features_presets_all" class="inline"><?php esc_html_e( 'All', 'image-hopper-editor' ); ?></label>

					<input type="radio" id="editor_features_presets_custom" name="editor_features_presets" value="custom" onclick="ToggleFeatures(this.value);" onkeypress="ToggleFeatures(this.value);" />
					<label for="editor_features_presets_custom" class="inline"><?php esc_html_e( 'Custom', 'image-hopper-editor' ); ?></label>

				</div>
			</li>

			<li class="image_hopper_editor_individual_features field_setting" style="column-count: 3">
				<?php foreach ( $this->get_editor_sections() as $sid => $section ): ?>
					<div id="<?php echo esc_attr( 'editor_individual_features_container_' . str_replace( 'field_', '', $sid ) ); ?>" class="editor_individual_features_container" style="break-inside: avoid;<?php echo $sid !== 'field_crop' ? 'margin-top: 20px' : ''; ?>">

						<?php
						$keys = array_keys( $section['subfields'] );
						$keys = implode(
							',',
							array_map(
								static function( $string ) {
									return "'$string'";
								},
								$keys
							)
						);
						?>

						<div>
							<input type="checkbox" id="<?php echo esc_attr( $sid ); ?>" name="editor_views" onclick="SaveIndividualFeature(this.getAttribute('id'), this.checked, [<?php echo esc_attr( $keys ); ?>], this.checked);" onkeypress="SaveIndividualFeature(this.getAttribute('id'), this.checked, [<?php echo esc_attr( $keys ); ?>], this.checked);" />
							<label for="<?php echo esc_attr( $sid ); ?>" class="inline"><b><?php echo esc_html( $section['label'] ); ?></b></label>
						</div>

						<?php
						foreach ( $section['subfields'] as $id => $label ):
							if ( $id === 'field_line' ) {
								continue;
							}
							?>
							<div>
								<input type="checkbox" id="<?php echo esc_attr( $id ); ?>" onclick="SaveIndividualFeature(this.getAttribute('id'), this.checked, '<?php echo esc_attr( $sid ); ?>');" onkeypress="SaveIndividualFeature(this.getAttribute('id'), this.checked, '<?php echo esc_attr( $sid ); ?>');" />
								<label for="<?php echo esc_attr( $id ); ?>" class="inline"><?php echo esc_html( $label ); ?></label>
							</div>
						<?php endforeach; ?>
					</div>
				<?php endforeach ?>
			</li>
			<?php
		}
	}

	public function field_enable_editor_appearance_settings( $position, $form_id ) {
		if ( $position === 500 ) {
			$sections = $this->get_editor_sections();

			?>
			<li class="image_hopper_editor_appearance field_setting">
				<?php /* translators: the Light/Dark preview mode setting label */ ?>
				<label for="editor_appearance" class="section_label"><?php esc_html_e( 'Editor Appearance', 'image-hopper-editor' ); ?></label>

				<select id="editor_appearance" onchange="SetFieldProperty('editorAppearance', this.value);">
					<?php /* translators: In reference to Light or Dark preview mode for the UI, depending on the system settings */ ?>
					<option value="auto"><?php esc_html_e( 'Automatic', 'image-hopper-editor' ); ?></option>
					<?php /* translators: In reference to Light preview mode for the UI */ ?>
					<option value="light"><?php esc_html_e( 'Light', 'image-hopper-editor' ); ?></option>
					<?php /* translators: In reference to Dark preview mode for the UI */ ?>
					<option value="dark"><?php esc_html_e( 'Dark', 'image-hopper-editor' ); ?></option>
				</select>
			</li>

			<li class="image_hopper_editor_default_screen field_setting">
				<label for="editor_default_screen" class="section_label"><?php esc_html_e( 'Default Editor View', 'image-hopper-editor' ); ?></label>

				<select id="editor_default_screen" onchange="SetFieldProperty('defaultEditorView', this.value);">
					<option value=""><?php esc_html_e( 'Select a default editor view', 'image-hopper-editor' ); ?></option>
				</select>
			</li>

			<li class="image_hopper_default_filter field_setting">
				<label for="editor_default_filter" class="section_label"><?php esc_html_e( 'Default Filter', 'image-hopper-editor' ); ?></label>

				<select id="editor_default_filter" onchange="SetFieldProperty('defaultFilter', this.value);">
					<option value=""><?php esc_html_e( 'Select a default image filter', 'image-hopper-editor' ); ?></option>
					<?php foreach ( $sections['field_filter']['subfields'] as $key => $label ): ?>
						<option value="<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $label ); ?></option>
					<?php endforeach; ?>
				</select>
			</li>

			<li class="image_hopper_editor_default_markup_tool field_setting">
				<label for="editor_default_markup_tool" class="section_label"><?php esc_html_e( 'Default Markup Tool', 'image-hopper-editor' ); ?></label>

				<select id="editor_default_markup_tool" onchange="SetFieldProperty('markupDefaultTool', this.value);">
					<option value=""><?php esc_html_e( 'Select a default tool', 'image-hopper-editor' ); ?></option>
					<?php foreach ( $sections['field_markup']['subfields'] as $key => $label ): ?>
						<option value="<?php echo esc_attr( $key ); ?>"><?php echo esc_html( $label ); ?></option>
					<?php endforeach; ?>
				</select>
			</li>

			<li class="image_hopper_editor_default_markup_color field_setting">
				<label for="editor_markup_color" class="section_label">
					<?php esc_html_e( 'Default Markup Color', 'image-hopper-editor' ); ?>
				</label>

				<?php \GFFormDetail::color_picker( 'ih_editor_markup_color', 'SetIhMarkupColor' ); ?>
			</li>
			<?php
		}
	}

	/**
	 * Include crop setting next to the Downsize Image setting
	 *
	 * @since 1.0
	 */
	public function add_crop_field_setting() {
		ob_start();
		?>

		<input type="checkbox" id="editor_enable_crop_to_size" onclick="SetFieldProperty('cropToSize', this.checked); ToggleUpscaleImage(this.checked)" onkeypress="SetFieldProperty('cropToSize', this.checked);ToggleUpscaleImage(this.checked)" />

		<label for="editor_enable_crop_to_size" class="inline">
			<?php esc_html_e( 'Crop to dimensions', 'image-hopper-editor' ); ?>
			<?php gform_tooltip( 'ih_crop_to_dimensions' ); ?>
		</label>

		<?php
		/* phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped */
		echo ob_get_clean();
	}

	/**
	 * Output the mark-up for the Upscale field setting
	 *
	 * @param int $position The position the settings should be located at.
	 * @param int $form_id  The ID of the form currently being edited.
	 *
	 * @since 1.0.0
	 */
	public function add_upscale_field_setting( $position, $form_id ) {
		if ( $position === 450 ) {
			?>
			<li class="image_hopper_upscale_field field_setting">
				<label class="section_label"><?php esc_html_e( 'Editor', 'image-hopper-editor' ); ?></label>

				<input type="checkbox" id="editor_enable_upscale_image" onclick="SetFieldProperty('upscaleImageToCropSize', this.checked)" onkeypress="SetFieldProperty('upscaleImageToCropSize', this.checked);" />
				<label for="editor_enable_upscale_image" class="inline">
					<?php esc_html_e( 'Upscale to crop dimensions if image too small', 'image-hopper-editor' ); ?>
					<?php gform_tooltip( 'ih_upscale_image' ); ?>
				</label>
			</li>
			<?php
		}
	}

	/**
	 * Inject the <input /> tag with the editor field settings
	 *
	 * @param string    $input
	 * @param           $form
	 * @param           $entry
	 * @param           $value
	 * @param \GF_Field $field
	 *
	 * @return string
	 *
	 * @since 1.0
	 */
	public function inject_field_settings( $input, $form, $entry, $value, $field ) {

		$settings = [
			'enableEditor'           => $field->enableEditor,
			'instantEdit'            => $field->instantEdit,
			'cropToSize'             => $field->cropToSize,
			'upscaleImageToCropSize' => $field->upscaleImageToCropSize,
			'defaultEditorView'      => $field->defaultEditorView,
			'defaultFilter'          => $field->defaultFilter,
			'editorAppearance'       => $field->editorAppearance,
			'markupDefaultColor'     => ! empty( $field->markupDefaultColor ) ? $field->markupDefaultColor : '#ff4136',
			'markupDefaultTool'      => $field->markupDefaultTool,
			'enabledFeatures'        => ! empty( $field->enabledFeatures ) ? json_decode( $field->enabledFeatures, true ) : [],
		];

		return str_replace( '/>', "data-editor='" . esc_attr( wp_json_encode( $settings, JSON_UNESCAPED_UNICODE ) ) . "' />", $input );
	}

	/**
	 * If editing a WooCommerce Gravity Forms entry, correct old Editor images
	 *
	 * @param string $value
	 * @param array  $form
	 * @param string $input_name
	 *
	 * @return string
	 *
	 * @since 1.0.1
	 *
	 * @depecated 2.7.2
	 */
	public function correct_value_for_woo_gf_entry( $value, $form, $input_name, $entry_id, $entry ) {
		return $value;
	}

	/**
	 * When the editor is enabled and the user updates an existing image, we need to delete the old image so the new one can
	 * seamlessly take its place.
	 *
	 * @param string $value Current field value
	 * @param array $form Gravity Forms object
	 * @param string $input_name Post field ID
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	public function override_image_on_update( $value, $form, $input_name, $entry_id, $entry ) {

		if ( empty( \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] ) ) {
			return $value;
		}

		/* Check the image editor is enabled for this field */
		$field = \GFAPI::get_field( $form, ltrim( $input_name, 'input_' ) );
		if ( $field === false || $field->enableEditor !== true ) {
			return $value;
		}

		/* Check we have a value already stored */
		$files = $value !== '' ? json_decode( $value, true ) : [];

		/* Merge the existing uploaded files into the files list - added in IH 2.4.1 */
		if ( method_exists( $field, 'group_uploaded_files' ) ) {
			list( , $existing_uploaded_files ) = $field->group_uploaded_files( $form, $entry, $input_name );
			$files                             = array_unique( array_merge( $files, $existing_uploaded_files ) );
		}

		if ( count( $files ) === 0 ) {
			return $value;
		}

		/* Find out if there's any images uploaded matching the name of an existing stored image */
		$stored = array_filter(
			(array) \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ],
			static function( $file ) {
				return empty( $file['temp_filename'] );
			}
		);

		$new = array_filter(
			(array) \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ],
			static function( $file ) {
				return ! empty( $file['temp_filename'] );
			}
		);

		/* $stored can contain the full URL to the image, while $new only has the filename, so we need to fix that first */
		$stored_column = array_column( $stored, 'uploaded_filename' );
		array_walk(
			$stored_column,
			static function( &$url ) {
				$url = wp_basename( $url );
			}
		);

		$matches = array_intersect( $stored_column, array_column( $new, 'uploaded_filename' ) );

		/* Now we have our updated image list, delete any stored images */
		foreach ( $stored as $file ) {
			if ( ! in_array( wp_basename( $file['uploaded_filename'] ), $matches, true ) ) {
				continue;
			}

			foreach ( $files as $key => $url ) {
				if ( strpos( $url, $file['uploaded_filename'] ) === false ) {
					continue;
				}

				/* Added in IH 2.4.2 */
				$file_path = \GFFormsModel::get_physical_file_path( $url, $entry_id );
				$file_path = apply_filters( 'gform_file_path_pre_delete_file', $file_path, $url );

				/* Attempt to delete */
				if ( is_file( $file_path ) ) {
					unlink( $file_path );
				}

				unset( $files[ $key ] );

				/* Remove the edited image from the main list */
				foreach ( \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ] as $index => $item ) {
					if ( wp_basename( $item['uploaded_filename'] ) === wp_basename( $url ) ) {
						unset( \GFFormsModel::$uploaded_files[ $form['id'] ][ $input_name ][ $index ] );
						break;
					}
				}

				/* Place the newly edited image in the correct order, so it won't get appended to the end of the list */
				$posted_files = \GFCommon::json_decode( stripslashes( \GFForms::post( 'gform_uploaded_files' ) ) );
				if ( empty( $posted_files[ $input_name ] ) ) {
					continue;
				}

				foreach ( $posted_files[ $input_name ] as $index => $item ) {
					/* Find the original image and save the index */
					if ( $item['uploaded_filename'] === $url ) {
						$correct_image_index = $index;
					}

					/* Find the newly uplodated/edited image */
					if ( isset( $correct_image_index ) && wp_basename( $url ) === $item['uploaded_filename'] ) {
						/* Copy the image data to the original position, then remove from the array */
						$posted_files[ $input_name ][ $correct_image_index ] = $item;
						unset( $posted_files[ $input_name ][ $index ] );

						break;
					}
				}

				/* Save the order so Image Hopper Core can handle the reordering */
				$_POST['gform_uploaded_files'] = addslashes( wp_json_encode( $posted_files, JSON_UNESCAPED_UNICODE ) );

				break;
			}
		}

		return wp_json_encode( $files, JSON_UNESCAPED_UNICODE );
	}

	/**
	 * @return array[]
	 *
	 * @since 2.11
	 */
	protected function get_editor_sections() {
		$sections = [
			'field_crop'   => [
				'label'     => esc_html__( 'Crop', 'image-hopper-editor' ),
				'subfields' => [
					'field_rotate_left'     => esc_html__( 'Rotate Left', 'image-hopper-editor' ),
					'field_rotate_right'    => esc_html__( 'Rotate Right', 'image-hopper-editor' ),
					'field_flip_horizontal' => esc_html__( 'Flip Horizontal', 'image-hopper-editor' ),
					'field_flip_vertical'   => esc_html__( 'Flip Vertical', 'image-hopper-editor' ),
					'field_manual_rotation' => esc_html__( 'Manual Rotation', 'image-hopper-editor' ),
					'field_show_size'       => esc_html__( 'Display Size', 'image-hopper-editor' ),
				],
			],

			'field_resize' => [
				'label'     => esc_html__( 'Resize', 'image-hopper-editor' ),
				'subfields' => [],
			],

			'field_redact' => [
				'label'     => esc_html__( 'Redact', 'image-hopper-editor' ),
				'subfields' => [],
			],

			'field_filter' => [
				'label'     => esc_html__( 'Filter', 'image-hopper-editor' ),
				'subfields' => [
					'field_pastel'      => esc_html__( 'Pastel', 'image-hopper-editor' ),
					'field_chrome'      => esc_html__( 'Chrome', 'image-hopper-editor' ),
					'field_fade'        => esc_html__( 'Fade', 'image-hopper-editor' ),
					'field_warm'        => esc_html__( 'Warm', 'image-hopper-editor' ),
					'field_cold'        => esc_html__( 'Cold', 'image-hopper-editor' ),

					'field_mono'        => esc_html__( 'Mono', 'image-hopper-editor' ),
					'field_mono_noir'   => esc_html__( 'Mono Noir', 'image-hopper-editor' ),
					'field_mono_wash'   => esc_html__( 'Mono Wash', 'image-hopper-editor' ),
					'field_mono_stark'  => esc_html__( 'Mono Stark', 'image-hopper-editor' ),

					'field_sepia'       => esc_html__( 'Sepia', 'image-hopper-editor' ),
					'field_sepia_blues' => esc_html__( 'Sepia Blues', 'image-hopper-editor' ),
					'field_sepia_rust'  => esc_html__( 'Sepia Rust', 'image-hopper-editor' ),
					'field_sepia_color' => esc_html__( 'Sepia Color', 'image-hopper-editor' ),
				],
			],

			'field_color'  => [
				'label'     => esc_html__( 'Finetune', 'image-hopper-editor' ),
				'subfields' => [
					'field_brightness'  => esc_html__( 'Brightness', 'image-hopper-editor' ),
					'field_contrast'    => esc_html__( 'Contrast', 'image-hopper-editor' ),
					'field_saturation'  => esc_html__( 'Saturation', 'image-hopper-editor' ),
					'field_exposure'    => esc_html__( 'Exposure', 'image-hopper-editor' ),
					'field_temperature' => esc_html__( 'Temperature', 'image-hopper-editor' ),
					'field_gamma'       => esc_html__( 'Gamma', 'image-hopper-editor' ),
					'field_clarity'     => esc_html__( 'Clarity', 'image-hopper-editor' ),
					'field_vignette'    => esc_html__( 'Vignette', 'image-hopper-editor' ),
				],
			],

			'field_markup' => [
				'label'     => esc_html__( 'Markup', 'image-hopper-editor' ),
				'subfields' => [
					'field_draw'   => esc_html__( 'Draw', 'image-hopper-editor' ),
					'field_line'   => esc_html__( 'Line', 'image-hopper-editor' ),
					'field_path'   => esc_html__( 'Path', 'image-hopper-editor' ),
					'field_arrow'  => esc_html__( 'Arrow', 'image-hopper-editor' ),
					'field_text'   => esc_html__( 'Text', 'image-hopper-editor' ),
					'field_square' => esc_html__( 'Square', 'image-hopper-editor' ),
					'field_circle' => esc_html__( 'Circle', 'image-hopper-editor' ),
				],
			],
		];

		return $sections;
	}

}
