import FilePondPluginFilePoster from 'filepond-plugin-file-poster'
import FilePondPluginImageEditor from '@pqina/filepond-plugin-image-editor'

import {
  // editor
  openEditor,
  processImage,
  createDefaultImageReader,
  createDefaultImageWriter,
  createDefaultImageOrienter,
  createDefaultShapePreprocessor,
  createDefaultImageScrambler,
  setPlugins,
  blobToFile,
  createMarkupEditorToolStyles,
  colorStringToColorArray,
  // plugins
  locale_en_gb,
  plugin_crop,
  plugin_crop_locale_en_gb,
  plugin_filter,
  plugin_filter_defaults,
  plugin_filter_locale_en_gb,
  plugin_finetune,
  plugin_finetune_defaults,
  plugin_finetune_locale_en_gb,
  plugin_annotate,
  plugin_annotate_locale_en_gb,
  plugin_resize,
  plugin_resize_locale_en_gb,
  plugin_sticker,
  plugin_sticker_locale_en_gb,
  plugin_frame,
  plugin_frame_defaults,
  plugin_frame_locale_en_gb,
  plugin_redact,
  plugin_redact_locale_en_gb,
  markup_editor_defaults,
  markup_editor_locale_en_gb
} from '@pqina/pintura'
import decodeHtmlSpecialEntities from './helpers/decodeHtmlSpecialEntities'
import stripPlaceholder from './helpers/stripPlaceholder'
import filterOptions from './helpers/filters/filters'
import defaultFilterMatrix from './helpers/filters/defaultFilterMatrix'
import getAspectRatioCalculation from './helpers/getAspectRatioCalculation'
import getCropFields from './helpers/pluginsConfig/getCropFields'
import getFilterProperties from './helpers/pluginsConfig/getFilterProperties'
import { getAnnotateProperties, getActiveAnnotateTool } from './helpers/pluginsConfig/getAnnotateProperties'
import getFinetuneControlConfig from './helpers/pluginsConfig/getFinetuneControlConfig'
import { enableLightMode, enableDarkMode } from './helpers/editorAppearance'
import { toBytes } from 'filepond/src/js/utils/toBytes'
import '../css/style.scss'

/**
 * @package     Image Hopper Editor
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://imagehopper.tech/commercial-license/ Commercial License
 */

((gform, $) => {
  const l10n = image_hopper_js_strings.editorL10n

  // Decode string html special entities
  for (const key of Object.keys(l10n)) {
    l10n[key] = decodeHtmlSpecialEntities(l10n[key])
  }

  gform.addAction('image_hopper_filepond_plugin_registration', FilePond => {
    setPlugins(
      plugin_crop,
      plugin_filter,
      plugin_finetune,
      plugin_annotate,
      plugin_resize,
      plugin_sticker,
      plugin_frame,
      plugin_redact,
    )

    FilePond.registerPlugin(
      FilePondPluginImageEditor,
      FilePondPluginFilePoster
    )
  })

  // Track the editor instance
  const editors = []

  // Track any GWiz Nested Form fields
  const nestedFormsObjects = []
  gform.addFilter( 'gpnf_modal_args', (args, formId, fieldId, nestedObject) => {
    nestedFormsObjects.push(nestedObject)
    return args
  }, 10, 4)

  gform.addFilter('image_hopper_filepond_config', (config, FilePond, $form, currentPage, formId, fieldId, entryId, files, field) => {
    // Get all the editor settings together
    const editor = $(field).data('editor')
    const cropToDimensions = editor.cropToSize
    const maxFileSize = toBytes($(field).data('max-file-size'))

    const isAlwaysTransformation = $(field).data('image-transform-output-quality-mode') === 'always'
    const isDefaultImageQuality = $(field).data('image-transform-output-quality') === 90

    // Map old names to new names to maintain backwards compatibility
    const enabledFeatures = {
      ...editor.enabledFeatures,
      field_finetune: editor.enabledFeatures.field_color,
      field_annotate: editor.enabledFeatures.field_markup
    }

    const topLevelGroup = ['crop', 'filter', 'color', 'markup', 'resize', 'redact'].filter((value) => enabledFeatures['field_' + value])

    // Downscale images & Crop to dimensions field value
    const cropWidth = parseInt($(field).data('image-resize-target-width'))
    const cropHeight = parseInt($(field).data('image-resize-target-height'))
    const cropAspectRatio = cropToDimensions ? getAspectRatioCalculation(cropWidth, cropHeight) : undefined

    // Output quality field value
    const outputQuality = $(field).data('imageTransformOutputQuality') / 100

    // Upscale to crop dimensions field value
    const upscaleImage = cropToDimensions && editor.upscaleImageToCropSize

    // Minimum size of the crop fields value
    const minimumWidth = $(field).data('image-validate-size-min-width')
    const minimumHeight = $(field).data('image-validate-size-min-height')

    // Resize tab minimum size
    const resizeMinSize = cropToDimensions ? { width: cropWidth, height: cropHeight } : { width: 1, height: 1 }

    let styleItemPanelAspectRatio = cropToDimensions ? cropHeight / cropWidth : config.styleItemPanelAspectRatio
    if (styleItemPanelAspectRatio > 1) {
      styleItemPanelAspectRatio = config.styleItemPanelAspectRatio
    }

    // Strip field_ prefix
    editor.defaultEditorView = stripPlaceholder(editor.defaultEditorView)

    // Set crop as default editor view
    let defaultEditorView = topLevelGroup.includes(editor.defaultEditorView) ? editor.defaultEditorView : topLevelGroup[0]
    switch (defaultEditorView) {
      case 'color':
        defaultEditorView = 'finetune'
        break

      case 'markup':
        defaultEditorView = 'annotate'
        break
    }

    // Backwards compat
    let topLevelGroupColorIndex = topLevelGroup.findIndex(a => a === 'color')
    if (topLevelGroupColorIndex !== -1) {
      topLevelGroup[topLevelGroupColorIndex] = 'finetune'
    }

    topLevelGroupColorIndex = topLevelGroup.findIndex(a => a === 'markup')
    if (topLevelGroupColorIndex !== -1) {
      topLevelGroup[topLevelGroupColorIndex] = 'annotate'
    }

    // Disable image editor if no group is selected
    const enableEditor = topLevelGroup.length > 0 && editor.enableEditor

    // Get editor plugin properties
    const cropFields = getCropFields(enabledFeatures)
    const filterProperties = getFilterProperties(enabledFeatures)
    const activeAnnotateProperties = getAnnotateProperties(enabledFeatures)
    const activeAnnotateTool = getActiveAnnotateTool(editor.markupDefaultTool)
    const finetuneControlConfiguration = getFinetuneControlConfig(plugin_finetune_defaults, enabledFeatures)

    config = {
      ...config,
      imageEditorAllowEdit: enableEditor,
      imageEditorInstantEdit: enableEditor && editor.instantEdit,
      imageEditor: {

        // Used to create the editor, receives editor configuration, should return an editor instance
        createEditor: openEditor,

        // Required, used for reading the image data
        imageReader: [createDefaultImageReader],

        // Optionally can leave out when not generating a preview thumbnail and/or output image
        imageWriter: [
          createDefaultImageWriter,
          {
            canvasMemoryLimit: 4096 * 4096,
            quality: outputQuality,
            targetSize: {
              width: cropWidth,
              height: cropHeight,
              fit: 'contain',
              upscale: upscaleImage
            },

            // if image has been resized below minimum threshold, pad the edges
            postprocessImageData: (imageData) => {
              return new Promise((resolve) => {
                // image larger than minimum, skip
                if (imageData.width >= minimumWidth && imageData.height >= minimumHeight) {
                  resolve(imageData)
                }

                // create a canvas element to handle the imageData
                const canvas = document.createElement('canvas')
                canvas.width = minimumWidth > imageData.width ? minimumWidth : imageData.width
                canvas.height = minimumHeight > imageData.height ? minimumHeight : imageData.height

                const ctx = canvas.getContext('2d')
                ctx.fillStyle = '#FFFFFF'
                ctx.fillRect(0, 0, canvas.width, canvas.height)

                // add the image to the center of the canvas
                const canvasX = (canvas.width - imageData.width) / 2
                const canvasY = (canvas.height - imageData.height) / 2
                ctx.putImageData(imageData, canvasX, canvasY)

                // returns the padded image
                resolve(
                  ctx.getImageData(0, 0, canvas.width, canvas.height)
                )
              })
            },

            // If the image size is larger than the max file size try reduce the image quality
            postprocessImageBlob: ({ blob }) => {
              if (blob.size < maxFileSize) {
                return blob
              }

              let quality = outputQuality - 0.25
              if (quality < .5) {
                quality = .5
              }

              return new Promise(async (resolve) => {
                const results = await processImage(blob, {
                  imageReader: createDefaultImageReader(),
                  imageWriter: createDefaultImageWriter({
                    quality: quality,
                  })
                })

                resolve(results.dest)
              })
            },
          }
        ],

        // Used to generate poster images, runs an editor in the background
        imageProcessor: processImage,

        // Pintura image editor options
        editorOptions: {
          handleEvent: (type, detail) => {
            // Save the editor instance
            if (type === 'init') {
              editors[`formId.fieldId`] = detail
            }

            // Handle Light/Dark mode
            if (type === 'loadstart') {
              const browserDarkModeEnabled = window.matchMedia('(prefers-color-scheme: dark)').matches
              const automaticAppearance = editor.editorAppearance === 'auto' || !editor.editorAppearance
              if (editor.editorAppearance === 'light' || (automaticAppearance && !browserDarkModeEnabled)) {
                enableLightMode()
              } else if (editor.editorAppearance === 'dark' || (automaticAppearance && browserDarkModeEnabled)) {
                enableDarkMode()
              }
            }

            if (type === 'show') {
              // Fix editor focus bug with GWiz Nested Forms
              nestedFormsObjects.map(nest => nest.trap ? nest.trap.pause() : '')

              // auto-focus the editor when it is opened
              editors[`formId.fieldId`]?.element.querySelector('button')?.focus()
            }

            if (type === 'hide') {
              // Restore normal GWiz Nested Forms functionality once the editor has closed
              nestedFormsObjects.map(nest => nest.trap ? nest.trap.unpause() : '')
            }
          },

          util: defaultEditorView,
          utils: topLevelGroup,

          // Set aspect ratio as a number
          imageCropAspectRatio: cropAspectRatio,

          // Set minimum size of the crop
          imageCropMinSize: {
            width: minimumWidth,
            height: minimumHeight
          },

          // Set minimum size on resize tab
          resizeMinSize: resizeMinSize,

          // Crop properties
          cropEnableSelectPreset: false,
          cropEnableButtonRotateLeft: cropFields.field_rotate_left,
          cropEnableButtonRotateRight: cropFields.field_rotate_right,
          cropEnableButtonFlipHorizontal: cropFields.field_flip_horizontal,
          cropEnableButtonFlipVertical: cropFields.field_flip_vertical,
          cropEnableRotationInput: cropFields.field_manual_rotation,
          cropEnableInfoIndicator: cropFields.field_show_size,
          cropAutoCenterImageSelectionTimeout: 750,

          // Annotate properties
          ...markup_editor_defaults,
          shapePreprocessor: createDefaultShapePreprocessor(),

          annotateTools: activeAnnotateProperties,
          annotateActiveTool: activeAnnotateTool,
          enableMultiSelect: true,
          enableTapToAddText: true,
          markupEditorToolStyles: createMarkupEditorToolStyles({
            sharpie: {
              disableMove: false,
              strokeColor: colorStringToColorArray(editor.markupDefaultColor),
            },
            path: {
              strokeColor: colorStringToColorArray(editor.markupDefaultColor),
            },

            line: {
              strokeColor: colorStringToColorArray(editor.markupDefaultColor),
            },

            arrow: {
              strokeColor: colorStringToColorArray(editor.markupDefaultColor),
            },

            rectangle: {
              strokeColor: colorStringToColorArray(editor.markupDefaultColor),
              backgroundColor: colorStringToColorArray(editor.markupDefaultColor),
            },

            ellipse: {
              strokeColor: colorStringToColorArray(editor.markupDefaultColor),
              backgroundColor: colorStringToColorArray(editor.markupDefaultColor),
            },

            text: {
              color: colorStringToColorArray(editor.markupDefaultColor)
            }
          }),

          // Filters properties
          filterOptions: filterOptions(filterProperties),
          filterFunctions: plugin_filter_defaults.filterFunctions,

          // Set editor default properties
          imageOrienter: createDefaultImageOrienter(),

          // Finetune properties
          finetuneControlConfiguration: finetuneControlConfiguration,
          finetuneOptions: [
            ...plugin_finetune_defaults.finetuneOptions.map(item => finetuneControlConfiguration[item[0]] ? item : undefined).filter(item => item !== undefined)
          ],

          // Frame
          ...plugin_frame_defaults,

          // Redact
          imageScrambler: createDefaultImageScrambler({}),

          // Locale
          locale: {
            ...locale_en_gb,
            ...markup_editor_locale_en_gb,
            ...plugin_finetune_locale_en_gb,
            ...plugin_annotate_locale_en_gb,
            ...plugin_crop_locale_en_gb,
            ...plugin_filter_locale_en_gb,
            ...plugin_resize_locale_en_gb,
            ...plugin_sticker_locale_en_gb,
            ...plugin_frame_locale_en_gb,
            ...plugin_redact_locale_en_gb,

            // Locale overrides
            ...l10n,
            labelSupportError: (features) => l10n.labelSupportError.replace('%s', features.join(', ')),

            statusLabelLoadImage: (state) => {
              if (!state || !state.task) return l10n.statusLabelLoadImageWaiting
              if (state.error) {
                return state.error.code === 'IMAGE_TOO_SMALL'
                  ? l10n.statusLabelLoadImageImageTooSmall
                  : l10n.statusLabelLoadImageError
              }
              if (state.task === 'blob-to-bitmap') return l10n.statusLabelLoadImage
              return l10n.statusLabelLoadImageCreatingPreview
            },

            // processing status message
            statusLabelProcessImage: (state) => {
              if (!state || !state.task) return undefined
              if (state.task === 'store') {
                if (state.error) return l10n.statusLabelProcessImageErrorUploading
                return l10n.statusLabelProcessImageUploading
              }
              if (state.error) return l10n.statusLabelProcessImageErrorProcessing
              return l10n.statusLabelProcessImage
            }
          }
        }
      },

      styleItemPanelAspectRatio,
      filePosterMaxHeight: 600
    }

    return config
  }, 10, 9)

  gform.addAction('image_hopper_post_filepond_create', (FilePond, $form, currentPage, formId, fieldId, entryId, files, field) => {
    const pond = $('#field_' + formId + '_' + fieldId).find('.filepond--root')[0]
    const editor = $(field).data('editor')

    // Handle the default image filter when a new image is uploaded
    if(!editor.defaultFilter) {
      return
    }

    pond.addEventListener('FilePond:initfile', e => {
      if (e.detail.file.origin === FilePond.FileOrigin.INPUT) {
        e.detail.file.setMetadata('imageState', {
          colorMatrix: {
            filter: defaultFilterMatrix(editor.defaultFilter)
          },
        })
      }
    })
  }, 10, 8)
})(gform, jQuery)
