/**
 * @package     Image Hopper Editor
 * @copyright   Copyright (c) 2025, Image Hopper
 * @license     https://imagehopper.tech/commercial-license/ Commercial License
 */

/**
 * Show or hide the editor fields based on the toggle
 */
function ToggleEditorFields (checked) {
  const $sections = jQuery('li[class^="image_hopper_editor"]')

  checked ? $sections.show() : $sections.hide()
}

function ToggleUpscaleImage (checked) {
  const $upscaleImageSection = jQuery('li[class^="image_hopper_upscale_field"]')

  checked ? $upscaleImageSection.show() : $upscaleImageSection.hide()
}

/**
 * Control the saving of the feature preset
 * @param type string
 */
function ToggleFeatures (type) { // eslint-disable-line
  SetFieldProperty('featurePresets', type)

  const presets = {
    essentials: [
      'field_crop',
      'field_rotate_left',
      'field_rotate_right',
      'field_manual_rotation'
    ],
    color: [
      'field_filter',
      'field_pastel',
      'field_chrome',
      'field_warm',
      'field_cold',
      'field_mono',
      'field_sepia',
      'field_sepia_color',
      'field_color'
    ],
    annotate: [
      'field_markup',
      'field_draw',
      'field_arrow',
      'field_text',
      'field_crop',
      'field_manual_rotation',
      'field_redact'
    ],
    all: jQuery('.editor_individual_features_container input[type="checkbox"]').map(function () {
      return this.id
    }).get()
  }

  if (presets[type] !== undefined) {
    const value = {}

    jQuery('.editor_individual_features_container input[type="checkbox"]').prop('checked', false)

    for (let i = 0; i < presets[type].length; i++) {
      const id = presets[type][i]

      jQuery('#' + id).prop('checked', true)

      value[id] = true
    }

    SetFieldProperty('enabledFeatures', JSON.stringify(value))
  }

  setupDefaultEditorViewSetting()
}

/**
 * Correctly display all selected group feature labels for the Default Editor View setting
 */
function setupDefaultEditorViewSetting (dontTriggerChange) {
  // Add group item to Editor Default Screen
  const $editorScreen = jQuery('#editor_default_screen')
  let editorScreenCurrentVal = $editorScreen.val()
  const preset = jQuery('#editor_features_container input[name="editor_features_presets"]:checked').val()

  switch (preset) {
    case 'all':
    case 'essentials':
      editorScreenCurrentVal = 'field_crop'
      break

    case 'color':
      editorScreenCurrentVal = 'field_filter'
      break

    case 'annotate':
      editorScreenCurrentVal = 'field_markup'
      break
  }

  // Clear out old values
  $editorScreen.find('option:nth-child(n+2)').remove()

  // Add any selected groups
  jQuery('.editor_individual_features_container input[name="editor_views"]').each(function () {
    if (jQuery(this).prop('checked')) {
      $editorScreen.append('<option value="' + jQuery(this).attr('id') + '">' + jQuery(this).next().text() + '</option>')
    }
  })

  // Reset the default value
  if ($editorScreen.find('option[value="' + editorScreenCurrentVal + '"]').length) {
    $editorScreen.val(editorScreenCurrentVal)
  } else {
    $editorScreen.val('')
  }

  if (!dontTriggerChange) {
    $editorScreen.trigger('change')
  }
}

/**
 * Control the saving of the feature settings
 *
 * @param id string         The setting ID
 * @param state bool        The state of the setting
 * @param groupId string    The setting's group ID
 * @param groupState bool   The state of the settings' group
 */
function SaveIndividualFeature (id, state, groupId, groupState) { // eslint-disable-line
  if (groupId !== undefined) {
    if (!Array.isArray(groupId)) {
      groupId = [groupId]
    }

    groupState = groupState !== undefined ? groupState : true

    for (let i = 0; i < groupId.length; i++) {
      const gid = groupId[i]
      const $group = jQuery('#' + gid)
      $group.prop('checked', groupState)

      SaveIndividualFeature($group.attr('id'), groupState)
    }
  }

  jQuery('#editor_features_presets_custom').prop('checked', true)
  SetFieldProperty('featurePresets', 'custom')

  const value = GetSelectedField().enabledFeatures !== undefined ? JSON.parse(GetSelectedField().enabledFeatures) : {}
  value[id] = state

  SetFieldProperty('enabledFeatures', JSON.stringify(value))

  if (jQuery('#' + id).attr('name') === 'editor_views') {
    setupDefaultEditorViewSetting()
  }
}

/**
 * Rehydrate the Image Hopper Field Settings
 */
gform.addAction('image_hopper_load_field_settings', function (field, form) {
  // Crop to Size
  jQuery('#editor_enable_crop_to_size').prop('checked', field.cropToSize === true)

  // Enable Image Editor Setting
  jQuery('#field_enable_editor').prop('checked', field.enableEditor === true)
  ToggleEditorFields(field.enableEditor === true)

  // Enable Editor Auto Open Setting
  jQuery('#field_instant_edit').prop('checked', field.instantEdit === true)

  // Upscale
  jQuery('#editor_enable_upscale_image').prop('checked', field.upscaleImageToCropSize === true)
  ToggleUpscaleImage(field.cropToSize)

  // Default Filter
  jQuery('#editor_default_filter').val(field.defaultFilter)

  // Feature Preset Setting
  if (field.featurePresets === undefined) {
    return jQuery('#editor_features_presets_essentials').prop('checked', true).trigger('click')
  }

  jQuery('#editor_features_presets_' + field.featurePresets).prop('checked', true)

  // Feature Individual Settings
  if (field.enabledFeatures) {
    const features = JSON.parse(field.enabledFeatures)

    for (const key of Object.keys(features)) {
      jQuery('#' + key).prop('checked', features[key])
    }

    // Default Editor Views
    setupDefaultEditorViewSetting(true)
    jQuery('#editor_default_screen').val(field.defaultEditorView)
  }
}, 10, 2)
