# MonsterInsights Frontend System - Technical Documentation

## Overview

The MonsterInsights frontend system manages all client-side tracking, events, and user interactions. The system is version-aware, with different implementations for Lite and Pro versions, while sharing a common core infrastructure.

## Directory Structure

```
monsterinsights/
├── includes/frontend/
│   ├── frontend.php              # Core frontend functionality
│   ├── class-tracking-abstract.php
│   ├── seedprod.php             # SeedProd integration
│   ├── tracking/                 # Tracking implementations
│   │   ├── class-tracking-gtag.php
│   │   └── class-tracking-preview.php
│   ├── events/                  # Event tracking
│   │   └── class-gtag-events.php
│   └── verified-badge/          # Verification badge feature
│
├── pro/includes/frontend/       # Pro-specific features
│   ├── class-frontend.php      # Enhanced frontend functionality
│   └── privacy-guard.php       # Privacy features
│
└── lite/includes/frontend/     # Lite-specific features
    └── class-frontend.php      # Basic frontend functionality
```

## Core Components

### 1. Frontend Base (`frontend.php`)

Main functionality for frontend tracking and events:

```php
// Key Functions
monsterinsights_tracking_script()     // Main tracking script
monsterinsights_events_tracking()     // Events handling
monsterinsights_rss_link_tagger()    // RSS UTM tracking
monsterinsights_add_admin_bar_menu() // Admin bar integration
```

### 2. Tracking System

1. **Abstract Tracking Class**
   ```php
   class MonsterInsights_Tracking_Abstract {
       protected $tracking_type = '';
       protected $is_debug;
       
       abstract public function frontend_output();
       abstract public function get_frontend_code();
   }
   ```

2. **Tracking Implementations**
   - GTag Tracking (`class-tracking-gtag.php`)
   - Preview Mode (`class-tracking-preview.php`)

### 3. Events System

```php
class MonsterInsights_Gtag_Events {
    // Core event tracking
    public function load_frontend_events()
    public function track_custom_event()
    public function track_after_pageview()
}
```

## Version-Specific Features

### 1. Lite Version Features

```php
// Basic tracking features
- Standard pageview tracking
- Basic event tracking
- Simple user interaction tracking
- Basic eCommerce tracking
```

### 2. Pro Version Features

```php
// Enhanced tracking features
- Advanced eCommerce tracking
- Custom dimensions
- Form tracking
- Media tracking
- Privacy protection
- Enhanced user tracking
```

## Implementation Details

### 1. Tracking Script Integration

```php
// 1. Check if tracking should be loaded
if ( ! monsterinsights_skip_tracking() ) {
    
    // 2. Determine tracking mode
    $mode = is_preview() ? 'preview' : MonsterInsights()->get_tracking_mode();
    
    // 3. Load appropriate tracking
    do_action( 'monsterinsights_tracking_before_' . $mode );
    
    // 4. Output tracking code
    $tracking->frontend_output();
    
    do_action( 'monsterinsights_tracking_after_' . $mode );
}
```

### 2. Event Tracking

```php
// 1. Register events
add_action('template_redirect', 'monsterinsights_events_tracking');

// 2. Track specific events
monsterinsights_track_event('category', 'action', 'label', value);

// 3. Custom event tracking
do_action('monsterinsights_custom_event', $event_data);
```

### 3. Admin Bar Integration

```php
// 1. Add menu item
add_action('admin_bar_menu', 'monsterinsights_add_admin_bar_menu');

// 2. Load required scripts
add_action('wp_enqueue_scripts', 'monsterinsights_frontend_admin_bar_scripts');
```

## Key Features

### 1. Tracking Capabilities

- Pageview tracking
- Event tracking
- eCommerce tracking
- Form submissions
- File downloads
- Outbound links
- Affiliate links
- Custom dimensions

### 2. Privacy Features

- GDPR compliance
- Cookie consent integration
- IP anonymization
- User opt-out support
- Data retention controls

### 3. Performance Optimization

- Async script loading
- Script debugging tools
- Caching compatibility
- Performance monitoring

## Development Guidelines

### 1. Adding Custom Events

```php
// 1. Register custom event
add_filter('monsterinsights_frontend_tracking_options_gtag', function($options) {
    $options['custom_event'] = array(
        'name' => 'event_name',
        'trigger' => 'trigger_element'
    );
    return $options;
});

// 2. Track custom event
monsterinsights_track_custom_event($event_data);
```

### 2. Best Practices

1. **Performance**
   - Use async loading
   - Minimize blocking scripts
   - Optimize event triggers

2. **Compatibility**
   - Test with caching plugins
   - Check theme compatibility
   - Verify with page builders

3. **Security**
   - Sanitize tracking data
   - Validate event parameters
   - Handle user permissions

## Troubleshooting

### Common Issues

1. **Tracking Not Working**
   - Verify tracking code presence
   - Check for JavaScript errors
   - Validate GTM integration

2. **Event Tracking Issues**
   - Debug event triggers
   - Check event parameters
   - Verify data in GA

3. **Admin Bar Problems**
   - Check user permissions
   - Verify script loading
   - Debug CSS conflicts

## Additional Resources

- [Google Analytics 4 Documentation](https://developers.google.com/analytics/devguides/collection/ga4)
- [GTM Implementation Guide](https://www.monsterinsights.com/docs/implementing-gtm/)
- [Event Tracking Guide](https://www.monsterinsights.com/docs/event-tracking/) 