# MonsterInsights Email System - Technical Documentation

## Overview

The MonsterInsights email system provides a robust framework for sending notification emails to users. The system is version-aware, with different implementations for Lite and Pro versions, while sharing a common core infrastructure.

## Directory Structure

```
monsterinsights/
├── includes/emails/
│   ├── class-emails.php         # Core email functionality
│   └── templates/               # Base email templates
│       ├── body-default.php     # Default email body
│       ├── body-default-plain.php
│       ├── footer-default.php   # Default email footer
│       └── header-default.php   # Default email header
│
├── lite/includes/emails/        # Lite-specific implementations
│   ├── summaries.php           # Basic email summaries
│   ├── summaries-infoblocks.php
│   └── templates/              # Lite-specific templates
│
└── pro/includes/emails/        # Pro-specific implementations
    ├── summaries.php          # Enhanced email summaries
    ├── summaries-infoblocks.php
    └── templates/             # Pro-specific templates
```

## Core Components

### 1. Email Base Class (`class-emails.php`)

The `MonsterInsights_WP_Emails` class serves as the foundation for all email functionality:

```php
class MonsterInsights_WP_Emails {
    private $from_address;
    private $from_name;
    private $reply_to;
    private $cc;
    private $content_type;
    private $headers;
    private $html = true;
    private $template;
    protected $args;
}
```

Key Features:
- Template-based email generation
- HTML and plain text support
- Custom headers and CC support
- Dynamic content processing
- Attachment handling

### 2. Template System

1. **Base Templates**:
   - Header template with styling
   - Body templates (HTML and plain text)
   - Footer template with unsubscribe links

2. **Template Loading Priority**:
   ```php
   [Theme Directory]/monsterinsights/emails/
   [Plugin Directory]/pro/includes/emails/templates/
   [Plugin Directory]/lite/includes/emails/templates/
   [Plugin Directory]/includes/emails/templates/
   ```

## Version-Specific Features

### 1. Lite Version

```php
// Basic email features
- Simple notification templates
- Basic analytics summaries
- Core notification system
```

### 2. Pro Version

```php
// Enhanced email features
- Advanced notification templates
- Detailed analytics summaries
- Custom scheduling options
- Additional notification types
```

## Implementation Details

### 1. Sending Emails

```php
$email = new MonsterInsights_WP_Emails('template-name');
$email->set_args([
    'header' => [...],
    'body'   => [...],
    'footer' => [...]
]);
$email->send($to, $subject, $message, $attachments);
```

### 2. Email Hooks

```php
// Pre-send actions
add_action('monsterinsights_email_send_before', ...);

// Post-send actions
add_action('monsterinsights_email_send_after', ...);

// Content filters
add_filter('monsterinsights_email_content_type', ...);
add_filter('monsterinsights_email_headers', ...);
```

### 3. Template Tags

```php
{site_name}     // Website name
{date}          // Current date
{time}          // Current time
{unsubscribe}   // Unsubscribe link
{preview_link}  // Email preview link
```

## Email Types

### 1. Summary Emails

- Weekly/monthly analytics summaries
- Traffic reports
- Goal completion notifications
- Custom alert notifications

### 2. System Notifications

- Plugin updates
- License notifications
- Error alerts
- Security notifications

### 3. User Engagement

- Welcome emails
- Feature announcements
- Tips and tutorials
- Survey requests

## Development Guidelines

### 1. Creating Custom Email Templates

```php
// 1. Create template file
/templates/email-custom-template.php

// 2. Register template
add_filter('monsterinsights_email_templates', function($templates) {
    $templates['custom'] = 'Custom Template';
    return $templates;
});

// 3. Use template
$email = new MonsterInsights_WP_Emails('custom');
```

### 2. Best Practices

1. **Template Design**
   - Use responsive layouts
   - Test across email clients
   - Include plain text versions

2. **Content Handling**
   - Sanitize dynamic content
   - Use template tags consistently
   - Support internationalization

3. **Performance**
   - Cache template files
   - Optimize images
   - Batch processing for bulk emails

## Troubleshooting

### Common Issues

1. **Email Delivery**
   - Check SMTP configuration
   - Verify email headers
   - Monitor spam scores

2. **Template Loading**
   - Check template hierarchy
   - Verify file permissions
   - Debug template paths

3. **Content Display**
   - Test email clients
   - Validate HTML
   - Check character encoding

## Additional Resources

- [WordPress Mail Functions](https://developer.wordpress.org/reference/functions/wp_mail/)
- [Email Template Best Practices](https://www.monsterinsights.com/docs/email-summaries/)
- [HTML Email Standards](https://www.email-standards.org/) 