# MonsterInsights Site Notes - Technical Documentation

## Overview

The Site Notes system allows users to create, manage, and track important notes related to their website's analytics and performance. It provides a structured way to document website changes, updates, promotions, and other significant events that might impact analytics data.

## Core Components

### 1. Controller (`Controller.php`)

The `MonsterInsights_SiteNotes_Controller` class serves as the main controller for the Site Notes system:

```php
class MonsterInsights_SiteNotes_Controller {
    private $db;  // Database instance
    
    // Core functionality
    public function run()
    public function load_dependencies()
    public function add_hooks()
}
```

Key responsibilities:
- Initializing the system
- Managing AJAX endpoints
- Handling note operations
- Integration with WordPress admin
- Chart data preparation
- Meta box management

### 2. Database (`Database.php`)

The `MonsterInsights_Site_Notes_DB_Base` class handles all database operations:

```php
class MonsterInsights_Site_Notes_DB_Base {
    // Core database operations
    public function install()
    public function create($data)
    public function get($post_id)
    public function get_items($args)
    public function get_categories($args)
}
```

Features:
- Custom post type management
- Category taxonomy
- CRUD operations
- Query optimization

## Data Structure

### 1. Custom Post Type

```php
'monsterinsights_note' => [
    'public' => false,
    'supports' => ['title', 'author', 'custom-fields'],
    'capability_type' => 'post'
]
```

### 2. Custom Taxonomy

```php
'monsterinsights_note_category' => [
    'hierarchical' => false,
    'public' => false
]
```

### 3. Note Structure

```php
[
    'id' => int,
    'note_title' => string,
    'note_date' => 'Y-m-d',
    'status' => string,
    'important' => boolean,
    'medias' => array,
    'category' => [
        'id' => int,
        'name' => string,
        'background_color' => string
    ]
]
```

## Key Features

### 1. Note Management

- Create/Edit/Delete notes
- Categorization
- Media attachments
- Important flag
- Date tracking
- Status management (publish, trash)

### 2. Categories

Default categories:
- Website Updates
- Blog Post
- Promotion

Features:
- Custom category creation
- Color coding
- Category filtering
- Bulk management

### 3. Analytics Integration

- Chart annotations
- Traffic correlation
- Performance impact tracking
- Event documentation

## Implementation Details

### 1. AJAX Endpoints

```php
// Note Operations
wp_ajax_monsterinsights_vue_get_notes
wp_ajax_monsterinsights_vue_get_note
wp_ajax_monsterinsights_vue_save_note
wp_ajax_monsterinsights_vue_trash_notes
wp_ajax_monsterinsights_vue_restore_notes
wp_ajax_monsterinsights_vue_delete_notes

// Category Operations
wp_ajax_monsterinsights_vue_get_categories
wp_ajax_monsterinsights_vue_save_category
wp_ajax_monsterinsights_vue_delete_categories
```

### 2. Hooks and Filters

```php
// Core Hooks
add_action('init', 'register_meta')
add_action('wp_after_insert_post', 'create_note_with_post')
add_action('save_post', 'save_custom_fields')

// Report Integration
add_filter('monsterinsights_report_overview_data')
add_filter('monsterinsights_report_traffic_sessions_chart_data')
```

### 3. Meta Box Integration

- Post editor integration
- Custom fields management
- Note creation triggers

## Development Guidelines

### 1. Creating Notes Programmatically

```php
$note_data = [
    'note' => 'Note content',
    'category' => category_id,
    'important' => boolean,
    'date' => 'Y-m-d',
    'medias' => array
];

MonsterInsights_SiteNotes_Controller::get_instance()->create_note($note_data);
```

### 2. Adding Custom Categories

```php
$category_data = [
    'name' => 'Category Name',
    'background_color' => '#HEX'
];

$db->create_category($category_data);
```

### 3. Best Practices

1. **Data Validation**
   - Sanitize note content
   - Validate dates
   - Check permissions

2. **Performance**
   - Use proper indexing
   - Cache category data
   - Optimize queries

3. **Security**
   - Check nonces
   - Verify capabilities
   - Sanitize output

## Troubleshooting

### Common Issues

1. **Notes Not Displaying**
   - Check user permissions
   - Verify AJAX endpoints
   - Validate data structure

2. **Category Issues**
   - Check taxonomy registration
   - Verify term relationships
   - Check meta data

3. **Integration Problems**
   - Verify hooks execution
   - Check filter priorities
   - Debug data flow

## Additional Resources

- [WordPress Custom Post Types](https://developer.wordpress.org/plugins/post-types/)
- [WordPress Taxonomies](https://developer.wordpress.org/plugins/taxonomies/)
- [AJAX in WordPress](https://developer.wordpress.org/plugins/javascript/ajax/) 