# MonsterInsights Notifications System - Technical Documentation

## Overview

The MonsterInsights notifications system provides timely, actionable insights and recommendations to users based on their analytics data and plugin usage. The system is designed to be extensible, with each notification implemented as a separate class.

## Directory Structure

```
includes/admin/notifications/
├── notification-audience.php              # Audience demographics notifications
├── notification-bounce-rate.php           # Bounce rate alerts
├── notification-events.php                # Events tracking notifications
├── notification-mobile-device-*.php       # Mobile traffic insights
├── notification-upgrade-*.php             # Pro features promotions
└── notification-install-*.php             # Plugin installation recommendations
```

## Core Components

### 1. Base Notification Class

All notifications extend the `MonsterInsights_Notification_Event` base class, which provides:

```php
class MonsterInsights_Notification_Event {
    public $notification_id;           // Unique identifier
    public $notification_interval;     // Frequency in days
    public $notification_type;         // License types that can see this
    public $notification_category;     // Category (insight, alert, etc)
    public $notification_priority;     // Display priority
}
```

### 2. Notification Types

1. **Insights**
   - Audience demographics
   - Traffic patterns
   - User behavior

2. **Alerts**
   - Bounce rate changes
   - Traffic drops
   - Multiple tracking codes

3. **Recommendations**
   - Feature upgrades
   - Plugin installations
   - Configuration suggestions

## Implementation Details

### 1. Notification Structure

Each notification class follows this pattern:

```php
final class MonsterInsights_Notification_Example extends MonsterInsights_Notification_Event {
    // Notification metadata
    public $notification_id = 'unique_id';
    public $notification_interval = 30; // days
    public $notification_type = array('basic', 'lite', 'pro');
    public $notification_category = 'insight';
    public $notification_priority = 2;

    // Prepare notification content
    public function prepare_notification_data($notification) {
        $data = $this->get_notification_data();
        // Build notification array
        return $notification;
    }

    // Fetch required data
    public function get_notification_data() {
        // Get and process data
        return $data;
    }
}
```

### 2. Data Sources

Notifications can pull data from:
- Google Analytics reports
- WordPress site settings
- User interaction history
- Plugin configuration
- External APIs

### 3. Display Logic

Notifications are filtered based on:
1. User license type
2. Notification interval
3. Site configuration
4. Previous dismissals
5. Priority level

## Key Features

### 1. Audience Insights
- Geographic distribution
- Language preferences
- Device usage patterns
- Traffic sources

### 2. Performance Monitoring
- Bounce rate tracking
- Traffic trend analysis
- Mobile performance
- Event tracking

### 3. Upgrade Recommendations
- Feature-based suggestions
- Traffic-based recommendations
- Integration opportunities
- Pro version benefits

## Development Guidelines

### 1. Creating New Notifications

1. **File Naming**
   ```
   notification-{feature-name}.php
   ```

2. **Class Structure**
   ```php
   final class MonsterInsights_Notification_{Feature} extends MonsterInsights_Notification_Event {
       // Required properties
       public $notification_id;
       public $notification_interval;
       public $notification_type;
       public $notification_category;
       public $notification_priority;

       // Required methods
       public function prepare_notification_data();
       public function get_notification_data();
   }
   ```

### 2. Best Practices

1. **Data Handling**
   - Cache API responses
   - Validate data before display
   - Handle empty states gracefully

2. **User Experience**
   - Clear, actionable messages
   - Relevant recommendations
   - Non-intrusive timing

3. **Performance**
   - Efficient data queries
   - Proper caching
   - Minimal API calls

## Integration Points

### 1. WordPress Admin
- Dashboard widget integration
- Settings page notifications
- Report page alerts

### 2. MonsterInsights Core
- Reports integration
- Settings validation
- License management

### 3. External Services
- Google Analytics
- WordPress.org API
- MonsterInsights API

## Troubleshooting

### Common Issues

1. **Missing Notifications**
   - Check notification interval
   - Verify user permissions
   - Validate data availability

2. **Data Discrepancies**
   - Clear cache
   - Check API connectivity
   - Verify report data

3. **Display Problems**
   - Review WordPress version
   - Check theme compatibility
   - Verify CSS loading

## Additional Resources

- [WordPress Admin Notices API](https://developer.wordpress.org/reference/hooks/admin_notices/)
- [MonsterInsights Notifications API Documentation](https://github.com/awesomemotive/monsterinsights-plugin-site)
- [Google Analytics API Reference](https://developers.google.com/analytics/devguides/reporting/core/v4) 