# MonsterInsights Admin - Technical Documentation

## Overview

The admin section of MonsterInsights handles all administrative functionality of the plugin, including settings management, reporting interface, authentication, licensing, and the Vue.js-based admin dashboard.

## Directory Structure

```
includes/admin/
├── pages/                    # Admin page templates
├── reports/                  # Report generation classes
├── site-notes/              # Site notes functionality
├── feature-feedback/        # Feature feedback system
├── licensing/               # License management
├── notifications/          # Admin notifications system
├── admin.php               # Core admin functionality
├── admin-assets.php        # Admin scripts and styles
├── api-auth.php            # Google Analytics authentication
├── routes.php              # Vue.js REST API routes
├── ajax.php                # AJAX handlers
├── common.php              # Shared admin functions
├── tracking.php            # Admin-side tracking
└── various utility files
```

## Core Components

### 1. Admin Menu System (`admin.php`)

- **Main Menu Registration**
  - Registers the MonsterInsights menu in WordPress admin
  - Handles submenu organization
  - Controls menu visibility based on user roles
  - Manages Pro vs Lite menu differences

- **Menu Features**
  ```php
  function monsterinsights_admin_menu() {
      // Main menu registration
      add_menu_page(...);
      
      // Dynamic submenu items
      add_submenu_page(...);
  }
  ```

### 2. REST API Routes (`routes.php`)

The `MonsterInsights_Rest_Routes` class manages all Vue.js admin interface API endpoints:

#### Key Endpoints:
- **Authentication & Profile**
  - `wp_ajax_monsterinsights_vue_get_license`
  - `wp_ajax_monsterinsights_vue_get_profile`

- **Settings Management**
  - `wp_ajax_monsterinsights_vue_get_settings`
  - `wp_ajax_monsterinsights_vue_update_settings`
  - `wp_ajax_monsterinsights_vue_update_settings_bulk`

- **Reporting**
  - `wp_ajax_monsterinsights_vue_get_report_data`
  - `wp_ajax_monsterinsights_vue_grab_popular_posts_report`

- **Plugin Management**
  - `wp_ajax_monsterinsights_vue_get_addons`
  - `wp_ajax_monsterinsights_vue_install_plugin`

### 3. Authentication System (`api-auth.php`)

Handles Google Analytics authentication and profile management:

- OAuth2 authentication flow
- GA4 property management
- Measurement Protocol implementation
- Manual UA/GA4 configuration support

### 4. Settings Management

Settings are managed through a combination of components:

1. **Settings Storage**
   - WordPress options API integration
   - Network vs Site settings handling
   - Default value management

2. **Settings Interface**
   - Vue.js based settings pages
   - Real-time validation
   - Import/Export functionality

3. **Settings Security**
   ```php
   private function handle_sanitization($field, $value) {
       // Sanitization based on field type
   }
   ```

### 5. Reporting System

The reporting system is built with several components:

1. **Report Generation**
   - API data fetching
   - Data processing and formatting
   - Cache management

2. **Report Types**
   - Overview Reports
   - Real-Time Data
   - Custom Dimensions
   - eCommerce Reports (Pro)

3. **Report Display**
   - Vue.js based interface
   - Interactive charts and graphs
   - Export capabilities

## Development Guidelines

### 1. Adding New Admin Pages

1. Create page template in `pages/`
2. Register in admin menu system
3. Add necessary Vue components
4. Implement required API endpoints

### 2. Implementing New Settings

1. Add setting to schema
2. Implement sanitization
3. Add Vue.js form components
4. Update REST API endpoints

### 3. Adding New Reports

1. Create report class in `reports/`
2. Implement data fetching
3. Add necessary API endpoints
4. Create Vue.js components

### 4. Security Considerations

- Always use nonce verification
- Implement capability checks
- Sanitize all inputs
- Escape all outputs

Example:
```php
check_ajax_referer('mi-admin-nonce', 'nonce');
if (!current_user_can('monsterinsights_save_settings')) {
    return;
}
```

## Asset Management

### 1. Script Loading

Scripts are managed in `admin-assets.php`:
- Conditional loading based on page
- Version-based cache busting
- Dependencies management

### 2. Stylesheet Organization

- Main admin styles
- Report-specific styles
- Vue.js component styles

## Notifications System

The admin notifications system provides:

1. **Notification Types**
   - Updates
   - Alerts
   - Recommendations
   - Marketing

2. **Notification Management**
   - Priority handling
   - Dismissal system
   - User-specific notices

## Troubleshooting

### Common Issues

1. **Authentication Problems**
   - Check OAuth configuration
   - Verify API credentials
   - Review error logs

2. **Report Loading Issues**
   - Check API quota
   - Verify cache system
   - Review API responses

3. **Settings Saving Problems**
   - Check user capabilities
   - Verify nonce
   - Review sanitization

## Future Considerations

1. **Performance Optimization**
   - Implement better caching
   - Optimize API calls
   - Improve asset loading

2. **Security Enhancements**
   - Regular security audits
   - Implementation of rate limiting
   - Enhanced validation

3. **Feature Development**
   - Enhanced reporting
   - Improved user interface
   - Better integration options

## Additional Resources

- [MonsterInsights API Documentation](https://developers.monsterinsights.com/)
- [WordPress Plugin Guidelines](https://developer.wordpress.org/plugins/)
- [Vue.js Best Practices](https://vuejs.org/guide/best-practices/) 