<?php

namespace GiveStripe\Settings\Controllers;

use Exception;
use Give\PaymentGateways\Stripe\Models\AccountDetail;
use Give\PaymentGateways\Stripe\Repositories\Settings;
use GiveStripe\Settings\DataTransferObjects\UpdateStripeManualAccountDetailsDto;
use GiveStripe\Settings\Facades\StripeApiKeys;

/**
 * Class UpdateStripeManualAccountDetailsController
 * @package GiveStripe\Settings\Controllers
 * @unlreased
 */
class UpdateStripeManualAccountDetailsController
{
    /**
     * Prevent the use of legacy secret keys that start with "sk_" in favor of the ones starting with "rk_" and are generated by the GiveWP Stripe App
     * @since 2.4.0
     */
    public function __invoke()
    {
        $this->validateRequest();
        $requestData = UpdateStripeManualAccountDetailsDto::fromArray(give_clean($_POST));

        if ($this->hasLegacySecretKey($requestData)) {
            wp_send_json_error([
                'error' => esc_html__(
                    'Keys starting with "sk_" are not allowed. Install the GiveWP Stripe App to get the proper keys.',
                    'give-stripe'
                ),
            ]);
        }

        try {
            $stripeAccount = AccountDetail::fromArray($requestData->toArray());
            give(Settings::class)->updateStripeAccount($stripeAccount);
        } catch (Exception $e) {
            wp_send_json_error();
        }

        wp_send_json_success();
    }

    /**
     * @since 2.4.0
     */
    private function validateRequest()
    {
        if (!current_user_can('manage_give_settings')) {
            die('Invalid request.');
        }
    }

    /**
     * @since 2.6.0
     */
    private function hasLegacySecretKey(UpdateStripeManualAccountDetailsDto $requestData): bool
    {
        return StripeApiKeys::isSecretKey($requestData->liveSecretKey) || StripeApiKeys::isSecretKey($requestData->testSecretKey);
    }
}
