<?php

namespace GiveRecurring\PaymentGateways\Stripe\Actions;

use Give\Donations\Models\Donation;
use Give\Donations\Models\DonationNote;
use Give\Framework\PaymentGateways\Commands\SubscriptionSynced;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\Support\ValueObjects\Money;
use Give\PaymentGateways\Gateways\Stripe\Traits\CanSetupStripeApp;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionPeriod;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use Stripe\Invoice;
use Stripe\Exception\ApiErrorException;
use Stripe\Subscription as StripeSubscription;

/**
 * Synchronizes a GiveWP subscription with its corresponding Stripe subscription
 *
 * @since 2.14.0
 */
class SyncStripeSubscription
{
    use CanSetupStripeApp;

    /**
     * @since 2.14.0
     *
     * @throws PaymentGatewayException
     */
    public function __invoke(Subscription $subscription): SubscriptionSynced
    {
        try {
            // Setup Stripe app for this form
            $this->setupStripeApp($subscription->donationFormId);

            // Retrieve the subscription from Stripe
            $stripeSubscription = StripeSubscription::retrieve($subscription->gatewaySubscriptionId);

            // Update subscription details from Stripe
            $this->syncSubscriptionDetails($subscription, $stripeSubscription);

            // Get subscription invoices from Stripe
            $invoices = $this->getStripeSubscriptionInvoices($stripeSubscription);

            // Create missing renewal donations and get present donations
            [$missingDonations, $presentDonations] = $this->processMissingRenewals($subscription, $invoices);

            return new SubscriptionSynced(
                $subscription, // Do not save the subscription here, let the handler do it
                $missingDonations,
                $presentDonations,
                __('Stripe subscriptions can be synchronized as far back as available invoice history allows.', 'give-recurring')
            );

        } catch (ApiErrorException $e) {
            throw new PaymentGatewayException(
                sprintf(
                    'Unable to synchronize subscription with Stripe. %s',
                    $e->getMessage()
                ),
                $e->getCode(),
                $e
            );
        } catch (\Exception $e) {
            throw new PaymentGatewayException(
                sprintf(
                    'An error occurred while synchronizing subscription. %s',
                    $e->getMessage()
                ),
                $e->getCode(),
                $e
            );
        }
    }

    /**
     * Sync subscription details from Stripe subscription
     *
     * @since 2.14.0
     *
     * @return void
     */
    protected function syncSubscriptionDetails(Subscription $subscription, StripeSubscription $stripeSubscription): void
    {
        // Sync subscription status
        $subscription->status = $this->mapStripeStatusToGiveStatus($stripeSubscription->status);

        // Note: We don't sync the createdAt date as it should remain as originally set when the subscription was created

        // Get the subscription item to sync billing period and frequency
        if (!empty($stripeSubscription->items->data)) {
            $subscriptionItem = $stripeSubscription->items->data[0];

            if (isset($subscriptionItem->price)) {
                $price = $subscriptionItem->price;

                // Sync billing period
                if (isset($price->recurring->interval)) {
                    $subscription->period = $this->mapStripeIntervalToGivePeriod($price->recurring->interval);
                }

                // Sync frequency
                if (isset($price->recurring->interval_count)) {
                    $subscription->frequency = $price->recurring->interval_count;
                }
            }
        }
    }

    /**
     * Get invoices for the Stripe subscription
     *
     * @since 2.14.0
     *
     * @throws ApiErrorException
     */
    protected function getStripeSubscriptionInvoices(StripeSubscription $stripeSubscription): array
    {
        $invoices = [];
        $hasMore = true;
        $startingAfter = null;

        while ($hasMore) {
            $params = [
                'subscription' => $stripeSubscription->id,
                'status' => 'paid',
                'limit' => 100,
            ];

            if ($startingAfter) {
                $params['starting_after'] = $startingAfter;
            }

            $invoiceCollection = Invoice::all($params);
            $invoices = array_merge($invoices, $invoiceCollection->data);

            $hasMore = $invoiceCollection->has_more;

            if ($hasMore && !empty($invoiceCollection->data)) {
                $startingAfter = end($invoiceCollection->data)->id;
            }
        }

        return $invoices;
    }

    /**
     * Process missing renewal donations from Stripe invoices
     *
     * @since 2.14.0
     *
     * @throws \Exception
     */
    protected function processMissingRenewals(Subscription $subscription, array $invoices): array
    {
        $missingDonations = [];
        $presentDonations = $subscription->donations;
        $existingTransactionIds = [];

        // Get existing transaction IDs
        foreach ($presentDonations as $donation) {
            if ($donation->gatewayTransactionId) {
                $existingTransactionIds[] = $donation->gatewayTransactionId;
            }
        }

        // Process each invoice to create missing renewals
        foreach ($invoices as $invoice) {
            // Skip the initial invoice (it should already exist)
            if ($this->isInitialInvoice($subscription, $invoice)) {
                continue;
            }

            // Skip if we already have a donation for this transaction
            $transactionId = $invoice->payment_intent ?? $invoice->charge;
            if (!$transactionId || in_array($transactionId, $existingTransactionIds)) {
                continue;
            }

            // Create renewal donation
            $renewalDonation = $this->createRenewalDonationFromInvoice($subscription, $invoice);
            if ($renewalDonation) {
                $missingDonations[] = $renewalDonation;
                $existingTransactionIds[] = $renewalDonation->gatewayTransactionId;
            }
        }

        return [$missingDonations, $presentDonations];
    }

    /**
     * Check if the invoice is the initial subscription invoice
     *
     * @since 2.14.0
     *
     */
    protected function isInitialInvoice(Subscription $subscription, \Stripe\Invoice $invoice): bool
    {
        $initialDonation = $subscription->initialDonation();
        $invoiceTransactionId = $invoice->payment_intent ?? $invoice->charge;

        return $initialDonation && $initialDonation->gatewayTransactionId === $invoiceTransactionId;
    }

    /**
     * Create a renewal donation from a Stripe invoice
     *
     * @since 2.14.0
     *
     * @throws \Exception
     */
    protected function createRenewalDonationFromInvoice(Subscription $subscription, \Stripe\Invoice $invoice): ?Donation
    {
        $initialDonation = $subscription->initialDonation();
        if (!$initialDonation) {
            return null;
        }

        // Convert Stripe amount (cents) to dollars
        $amountInDollars = $invoice->amount_paid / 100;
        $currency = strtoupper($invoice->currency);

        $renewalDonation = $subscription->createRenewal([
            'gatewayTransactionId' => $invoice->payment_intent ?? $invoice->charge,
            'amount' => new Money($amountInDollars * 100, $currency),
        ]);

        // Add donation note
        DonationNote::create([
            'donationId' => $renewalDonation->id,
            'content' => sprintf(
                __('Renewal donation created during subscription synchronization. Stripe Invoice ID: %s', 'give-recurring'),
                $invoice->id
            )
        ]);

        return $renewalDonation;
    }

    /**
     * Map Stripe subscription status to GiveWP subscription status
     *
     * @since 2.14.0
     *
     */
    protected function mapStripeStatusToGiveStatus(string $stripeStatus): SubscriptionStatus
    {
        switch ($stripeStatus) {
            case 'active':
                return SubscriptionStatus::ACTIVE();
            case 'canceled':
                return SubscriptionStatus::CANCELLED();
            case 'incomplete':
            case 'incomplete_expired':
                return SubscriptionStatus::PENDING();
            case 'past_due':
                return SubscriptionStatus::FAILING();
            case 'paused':
                return SubscriptionStatus::PAUSED();
            case 'unpaid':
                return SubscriptionStatus::FAILING();
            case 'trialing':
                return SubscriptionStatus::ACTIVE();
            default:
                return SubscriptionStatus::ACTIVE();
        }
    }

    /**
     * Map Stripe billing interval to GiveWP subscription period
     *
     * @since 2.14.0
     *
     */
    protected function mapStripeIntervalToGivePeriod(string $stripeInterval): SubscriptionPeriod
    {
        switch ($stripeInterval) {
            case 'day':
                return SubscriptionPeriod::DAY();
            case 'week':
                return SubscriptionPeriod::WEEK();
            case 'month':
                return SubscriptionPeriod::MONTH();
            case 'year':
                return SubscriptionPeriod::YEAR();
            default:
                return SubscriptionPeriod::MONTH();
        }
    }
}
