<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Paystack\Traits;

use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;

/**
 * @since 2.16.0
 */
trait PaystackApiTrait
{
    /**
     * Make a request to the Paystack API
     *
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    protected function makePaystackApiRequest(string $method, string $endpoint, array $data = []): array
    {
        $secretKey = $this->getPaystackSecretKey();

        if (empty($secretKey)) {
            throw new PaymentGatewayException(__('Paystack secret key is not configured', 'give-recurring'));
        }

        $url = 'https://api.paystack.co/' . ltrim($endpoint, '/');

        $headers = [
            'Authorization' => 'Bearer ' . $secretKey,
            'Content-Type' => 'application/json',
            'Cache-Control' => 'no-cache',
        ];

        $args = [
            'method' => strtoupper($method),
            'headers' => $headers,
            'timeout' => 60,
        ];

        if (!empty($data) && in_array(strtoupper($method), ['POST', 'PUT', 'PATCH'])) {
            $args['body'] = wp_json_encode($data);
        } elseif (!empty($data) && strtoupper($method) === 'GET') {
            $url .= '?' . http_build_query($data);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Paystack API request failed: %s', 'give-recurring'),
                    $response->get_error_message()
                )
            );
        }

        $responseCode = wp_remote_retrieve_response_code($response);
        $responseBody = wp_remote_retrieve_body($response);
        $decodedResponse = json_decode($responseBody, true) ?? [];

        if ($responseCode >= 400) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Paystack API returned error %d: %s', 'give-recurring'),
                    $responseCode,
                    $decodedResponse['message'] ?? __('Unknown error occurred.', 'give-recurring')
                )
            );
        }

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new PaymentGatewayException(
                __('Invalid JSON response from Paystack API', 'give-recurring')
            );
        }

        return $decodedResponse;
    }

    /**
     * Get the Paystack secret key
     *
     * @return string
     */
    protected function getPaystackSecretKey(): string
    {
        return give_is_test_mode() ? give_get_option('paystack_test_secret_key') : give_get_option('paystack_live_secret_key');
    }

}
