<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Paystack\Actions;

use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Subscriptions\Models\Subscription;
use GiveRecurring\PaymentGatewayModules\Modules\Paystack\DataTransferObjects\PaystackPlan;
use GiveRecurring\PaymentGatewayModules\Modules\Paystack\Traits\PaystackApiTrait;

/**
 * @since 2.16.0
 */
class CreatePaystackPlan
{
    use PaystackApiTrait;

    /**
     * Handle creating or retrieving a Paystack plan
     *
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    public function handle(Subscription $subscription): PaystackPlan
    {
        // Create new plan
        $planData = [
            'name' => $this->generatePlanName($subscription),
            'amount' => $subscription->amount->formatToMinorAmount(),
            'interval' => $this->convertPeriodToPaystackInterval($subscription),
            'currency' => $subscription->amount->getCurrency()->getCode(),
            'invoice_limit' => $subscription->installments,
        ];

        $response = $this->makePaystackApiRequest('POST', 'plan', $planData);

        if (!isset($response['status']) || $response['status'] === false) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Unable to create Paystack plan: %s', 'give-recurring'),
                    isset($response['message']) ? $response['message'] : __('Unknown error', 'give-recurring')
                )
            );
        }

        return PaystackPlan::fromArray($response['data']);
    }

    /**
     * Generate a plan name for the subscription
     *
     * @since 2.16.0
     */
    protected function generatePlanName(Subscription $subscription): string
    {
        $donation = $subscription->initialDonation();
        $formTitle = get_the_title($subscription->donationFormId);

        return sprintf(
            '%s - %s %s',
            $formTitle,
            $subscription->amount->formatToDecimal(),
            $this->getIntervalDisplayName($subscription)
        );
    }

    /**
     * Convert GiveWP subscription period to Paystack interval
     *
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    protected function convertPeriodToPaystackInterval(Subscription $subscription): string
    {
        $period = $subscription->period->getValue();
        $frequency = $subscription->frequency;

        // Paystack supports: hourly, daily, weekly, monthly, quarterly, biannually, annually
        switch ($period) {
            case 'day':
                if ($frequency === 1) {
                    return 'daily';
                }
                break;
            case 'week':
                if ($frequency === 1) {
                    return 'weekly';
                }
                break;
            case 'month':
                if ($frequency === 1) {
                    return 'monthly';
                } elseif ($frequency === 3) {
                    return 'quarterly';
                } elseif ($frequency === 6) {
                    return 'biannually';
                }
                break;
            case 'year':
                if ($frequency === 1) {
                    return 'annually';
                }
                break;
        }

        throw new PaymentGatewayException(
            sprintf(
                __('Unsupported subscription interval: %s every %d %s(s)', 'give-recurring'),
                $period,
                $frequency,
                $period
            )
        );
    }

    /**
     * Get display name for the interval
     *
     * @since 2.16.0
     *
     */
    protected function getIntervalDisplayName(Subscription $subscription): string
    {
        $period = $subscription->period->getValue();
        $frequency = $subscription->frequency;

        if ($frequency === 1) {
            switch ($period) {
                case 'day':
                    return __('Daily', 'give-recurring');
                case 'week':
                    return __('Weekly', 'give-recurring');
                case 'month':
                    return __('Monthly', 'give-recurring');
                case 'year':
                    return __('Annually', 'give-recurring');
            }
        }

        return sprintf(
            __('Every %d %s(s)', 'give-recurring'),
            $frequency,
            $period
        );
    }
}
