<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\GoCardless;

use Exception;
use Give\Donations\Models\Donation;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\Http\Response\Types\RedirectResponse;
use Give\Framework\PaymentGateways\Commands\GatewayCommand;
use Give\Framework\PaymentGateways\Commands\RedirectOffsite;
use Give\Framework\PaymentGateways\Commands\SubscriptionSynced;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionAmountEditable;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionDashboardLinkable;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionTransactionsSynchronizable;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give\Framework\PaymentGateways\SubscriptionModule;
use Give\Framework\Support\ValueObjects\Money;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionPeriod;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use GiveGoCardless\Actions\CompleteGoCardlessRedirectFlow;
use GiveGoCardless\Actions\CreateGoCardlessRedirectFlow;
use GiveGoCardless\DataTransferObjects\GoCardlessRedirectFlow;
use GiveRecurring\PaymentGatewayModules\Modules\GoCardless\Actions\CancelGoCardlessSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\GoCardless\Actions\CreateGoCardlessSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\GoCardless\Actions\SyncGoCardlessSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\GoCardless\Actions\UpdateGoCardlessSubscriptionAmount;
use GiveRecurring\PaymentGatewayModules\Modules\GoCardless\DataTransferObjects\GoCardlessSubscription;

/**
 * @since 2.14.0
 */
class GoCardlessGatewaySubscriptionModule extends SubscriptionModule implements SubscriptionAmountEditable,
                                                                             SubscriptionDashboardLinkable,
                                                                             SubscriptionTransactionsSynchronizable
{
    /**
     * @since 2.14.0
     */
    public $secureRouteMethods = [
        'handleSubscriptionReturn'
    ];

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function createSubscription(
        Donation $donation,
        Subscription $subscription,
        $gatewayData
    ): GatewayCommand {

        $this->gateway->validateCurrency($donation);

        if (!$subscription->period->isOneOf(SubscriptionPeriod::WEEK(), SubscriptionPeriod::MONTH(), SubscriptionPeriod::YEAR())) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Period \'%s\' is not supported by GoCardless gateway.', 'give-gocardless'),
                    $subscription->period->getValue()
                )
            );
        }

        $redirectFlow = $this->createGoCardlessRedirectFlow($donation, $subscription, $gatewayData);
        return new RedirectOffsite($redirectFlow->redirectUrl);
    }

    /**
     * @since 2.14.0
     */
    public function getSubscriptionReturnURL(Donation $donation, Subscription $subscription, $gatewayData): string
    {
        $successReturnUrl = $this->gateway->generateSecureGatewayRouteUrl(
            'handleSubscriptionReturn',
            $donation->id,
            [
                'givewp-donation-id' => $donation->id,
                'givewp-subscription-id' => $subscription->id,
                'givewp-return-url' => $gatewayData['successUrl'],
            ]
        );

        return $successReturnUrl;
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function handleSubscriptionReturn(array $queryParams) : RedirectResponse
    {
        try {
            $donation = !empty($queryParams['givewp-donation-id']) ? Donation::find($queryParams['givewp-donation-id']) : '';
            $subscription = !empty($queryParams['givewp-subscription-id']) ? Subscription::find($queryParams['givewp-subscription-id']) : '';
            $redirectFlowId = !empty($queryParams['redirect_flow_id']) ? $queryParams['redirect_flow_id'] : '';

            if (!$donation || !$subscription || !$redirectFlowId) {
                throw new PaymentGatewayException(__('Invalid redirect flow request.', 'give-gocardless'));
            }

            $goCardlessSubscription = $this->createGoCardlessSubscription($subscription, $redirectFlowId, ['name' => $this->getSubscriptionDescription($subscription)]);
            $subscription->status = SubscriptionStatus::ACTIVE();
            $subscription->gatewaySubscriptionId = $goCardlessSubscription->id;
            $subscription->save();

            $donation = $subscription->initialDonation();
            $donation->status = DonationStatus::PROCESSING();
            $donation->save();

            return new RedirectResponse(esc_url_raw($queryParams['givewp-return-url']));

        } catch (Exception $e) {
            PaymentGatewayLog::error(
                __('[GoCardless] Payment Error: ', 'give-gocardless') . $e->getCode() . ' - ' . $e->getMessage(),
                [
                    'queryParams' => $queryParams,
                    'errorCode' => $e->getCode(),
                    'errorMessage' => $e->getMessage(),
                ]
            );

            throw new PaymentGatewayException($e->getCode() . ' - ' . $e->getMessage());
        }
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function cancelSubscription(Subscription $subscription): bool
    {
        try {

            $this->cancelGoCardlessSubscription($subscription);

            $subscription->status = SubscriptionStatus::CANCELLED();
            $subscription->save();

            return true;

        } catch (Exception $exception) {

            PaymentGatewayLog::error(
                sprintf(__('[GoCardless] Error cancelling subscription %s.', 'give-gocardless'), $subscription->id),
                [
                    'Payment Gateway' => $subscription->gatewayId,
                    'Subscription' => $subscription->id,
                    'Gateway Subscription Id' => $subscription->gatewaySubscriptionId,
                    'errorCode' => $exception->getCode(),
                    'errorMessage' => $exception->getMessage(),
                ]
            );

            throw new PaymentGatewayException(
                sprintf(
                    'Unable to cancel subscription with GoCardless. %s',
                    $exception->getMessage()
                ),
                $exception->getCode(),
                $exception
            );
        }
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function updateSubscriptionAmount(Subscription $subscription, Money $newRenewalAmount): bool
    {
        try {

            if ($subscription->amount->formatToDecimal() === $newRenewalAmount->formatToDecimal()) {
                return false;
            }

            $this->updateGoCardlessSubscriptionAmount($subscription, $newRenewalAmount);
            $subscription->amount = $newRenewalAmount;
            $subscription->save();

            return true;

        } catch (Exception $exception) {

            PaymentGatewayLog::error(
                sprintf(__('[GoCardless] Error updating subscription %s amount.', 'give-gocardless'), $subscription->id),
                [
                    'Current Amount' => $subscription->amount->formatToDecimal(),
                    'New Amount' => $newRenewalAmount->formatToDecimal(),
                    'Payment Gateway' => $subscription->gatewayId,
                    'Subscription' => $subscription->id,
                    'Gateway Subscription Id' => $subscription->gatewaySubscriptionId,
                    'errorCode' => $exception->getCode(),
                    'errorMessage' => $exception->getMessage(),
                ]
            );

            throw new PaymentGatewayException(
                sprintf(
                    'Unable to update subscription amount with GoCardless. %s',
                    $exception->getMessage()
                ),
                $exception->getCode(),
                $exception
            );
        }
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function synchronizeSubscription(Subscription $subscription): SubscriptionSynced
    {
        try {
            return give(SyncGoCardlessSubscription::class)($subscription);
        } catch (Exception $exception) {
            PaymentGatewayLog::error(
                sprintf(__('[GoCardless] Error synchronizing subscription %s.', 'give-gocardless'), $subscription->id),
                [
                    'Payment Gateway' => $subscription->gatewayId,
                    'Subscription' => $subscription->id,
                    'Gateway Subscription Id' => $subscription->gatewaySubscriptionId,
                    'errorCode' => $exception->getCode(),
                    'errorMessage' => $exception->getMessage(),
                ]
            );

            throw new PaymentGatewayException(
                sprintf(
                    'Unable to synchronize subscription with GoCardless. %s',
                    $exception->getMessage()
                ),
                $exception->getCode(),
                $exception
            );
        }
    }

    /**
     * @since 2.14.0
     */
    public function gatewayDashboardSubscriptionUrl(Subscription $subscription): string
    {
        $base_url = give_is_test_mode() ? 'https://manage-sandbox.gocardless.com/subscriptions/' : 'https://manage.gocardless.com/subscriptions/';
        return $base_url . $subscription->gatewaySubscriptionId;
    }

    /**
     * @since 2.14.0
     */
    protected function getSubscriptionDescription(Subscription $subscription): string
    {
        $description = sprintf(
            '%s - %s',
            $subscription->initialDonation()->formTitle,
            $subscription->amount->formatToDecimal()
        );

        if ($subscription->initialDonation()->levelId) {
            $description = sprintf(
                '%s - %s',
                $description,
                $subscription->initialDonation()->levelId
            );
        }

        return $description;
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    protected function createGoCardlessRedirectFlow(
        Donation $donation,
        Subscription $subscription,
        $gatewayData
    ): GoCardlessRedirectFlow {
        return give(CreateGoCardlessRedirectFlow::class)(
            $donation,
            $this->getSubscriptionDescription($subscription),
            $this->getSubscriptionReturnURL($donation, $subscription, $gatewayData)
        );
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    protected function createGoCardlessSubscription(
        Subscription $subscription, string $redirectFlowId, array $options = []
    ): GoCardlessSubscription {

        $mandateId = give(CompleteGoCardlessRedirectFlow::class)($subscription->initialDonation()->id, $redirectFlowId)->links['mandate'];
        return give(CreateGoCardlessSubscription::class)($subscription, $mandateId, $options);
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    protected function cancelGoCardlessSubscription(Subscription $subscription)
    {
        return give(CancelGoCardlessSubscription::class)($subscription);
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    protected function updateGoCardlessSubscriptionAmount(Subscription $subscription, Money $newRenewalAmount)
    {
        return give(UpdateGoCardlessSubscriptionAmount::class)($subscription, $newRenewalAmount);
    }
}
