<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Braintree;

use Exception;
use Give\Donations\Models\Donation;
use Give\Donations\Models\DonationNote;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\PaymentGateways\Commands\GatewayCommand;
use Give\Framework\PaymentGateways\Commands\SubscriptionComplete;
use Give\Framework\PaymentGateways\Commands\SubscriptionProcessing;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionDashboardLinkable;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\SubscriptionModule;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use GiveBraintree\DataTransferObjects\ApiAccessData;
use GiveBraintree\DataTransferObjects\BraintreeGatewayData;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\Actions\CancelBraintreeSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\Actions\CreateBraintreeSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\DataTransferObjects\BraintreeSubscription;

/**
 * @since 2.8.0
 */
class BraintreeGatewaySubscriptionModule extends SubscriptionModule implements SubscriptionDashboardLinkable
{
    /**
     * @since 2.8.0
     *
     * @throws Exception
     */
    public function createSubscription(
        Donation $donation,
        Subscription $subscription,
        $gatewayData
    ): GatewayCommand
    {
        try {
            $braintreeGatewayData = $gatewayData['braintreeGatewayData'];
            $braintreeSubscription = $this->createBraintreeSubscription($donation, $subscription,
                $braintreeGatewayData);

            if ($braintreeSubscription->status->isActive()) {
                return new SubscriptionComplete($braintreeSubscription->transactionId, $braintreeSubscription->id);
            }

            return new SubscriptionProcessing($braintreeSubscription->id, $braintreeSubscription->transactionId);
        } catch (Exception $e) {
            $subscription->status = SubscriptionStatus::FAILING();
            $subscription->save();

            $donation->status = DonationStatus::FAILED();
            $donation->save();

            $errorMessage = $e->getMessage();

            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(esc_html__('Donation failed. Reason: %s', 'give-braintree'), $errorMessage),
            ]);

            throw new PaymentGatewayException($errorMessage);
        }
    }

    /**
     * @since 2.8.0
     *
     * @throws Exception
     */
    public function cancelSubscription(Subscription $subscription): bool
    {
        try {
            $this->cancelBraintreeSubscription($subscription);
            $subscription->status = SubscriptionStatus::CANCELLED();
            $subscription->save();
        } catch (Exception $e) {
            throw new PaymentGatewayException($e->getMessage());
        }

        return true;
    }

    /**
     * @since 2.8.0
     */
    public function gatewayDashboardSubscriptionUrl(Subscription $subscription): string
    {
        $base = 'https://';


        if ('test' == $subscription->initialDonation()->mode->getValue()) {
            $base .= 'sandbox.';
        }

        $apiAccessData = ApiAccessData::fromOptions();

        $base .= 'braintreegateway.com/merchants/' . $apiAccessData->merchantId . '/subscriptions/';


        return esc_url($base . $subscription->gatewaySubscriptionId);
    }

    /**
     * @since 2.8.0
     *
     * @throws Exception
     */
    protected function createBraintreeSubscription(
        Donation $donation,
        Subscription $subscription,
        BraintreeGatewayData $braintreeGatewayData
    ): BraintreeSubscription {
        return give(CreateBraintreeSubscription::class)($donation, $subscription,
            $braintreeGatewayData);
    }

    /**
     * @since 2.8.0
     *
     * @throws Exception
     */
    protected function cancelBraintreeSubscription(Subscription $subscription): bool
    {
        return give(CancelBraintreeSubscription::class)($subscription);
    }
}
