<?php

/**
 * Subscription Failed Admin Email
 *
 * Subscription notification will be sent to recipient(s) when a subscription status is changed to failed.
 *
 * @package     Give_Recurring
 * @subpackage  Includes/Admin/Emails
 * @copyright   Copyright (c) 2024, GiveWP
 * @license     https://opensource.org/licenses/gpl-license GNU Public License
 * @since 4.17.0
 */

use Give\Subscriptions\Models\Subscription;

// Exit if access directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Give_Subscription_Failed_Admin_Email' ) ) :

	/**
	 * Class Give_Subscription_Failed_Admin_Email
	 *
	 * @since 4.17.0
	 */
	class Give_Subscription_Failed_Admin_Email extends Give_Email_Notification {

		/**
		 * @var Subscription|null $subscription
		 */
		private $subscription;

		/**
		 * Create a class instance.
		 *
		 * @since 4.17.0
		 */
		public function init() {

			$this->load(
				array(
					'id'                    => 'subscription-failed-admin',
					'label'                 => __( 'Subscription Failed', 'give-recurring' ),
					'description'           => __( 'Sent to designated recipient(s) when a subscription status is changed to failed.', 'give-recurring' ),
					'has_recipient_field'   => true,
					'notification_status'   => 'disabled',
					'form_metabox_setting'  => true,
					'default_email_subject' => esc_attr__( 'Subscription Failed', 'give-recurring' ),
					'default_email_message' => ( false !== give_get_option( 'subscription-failed-admin_email_message' ) ) ? give_get_option( 'subscription-failed-admin_email_message' ) : $this->get_default_email_message(),
					'default_email_header'  => __( 'Subscription Failed!', 'give-recurring' ),
				)
			);

			add_action( "give_{$this->config['id']}_email_notification", array( $this, 'setup_email_notification' ) );
			add_action( 'give_recurring_update_subscription_status', array( $this, 'maybe_send_email_notification' ), 10, 3 );
			add_action( 'givewp_subscription_updated', function($subscription){
				$old_status = $subscription->getOriginal('status')->getValue();
				$new_status = $subscription->status->getValue();
				$this->maybe_send_email_notification($subscription->id, $new_status, $old_status);
			});
		}

		/**
		 * Get default email message.
		 *
		 * @since 4.17.0
		 *
		 * @return string
		 */
		private function get_default_email_message() {
			$default_email_body  = __( 'Hi there,', 'give-recurring' ) . "\n\n";
			$default_email_body .= __( 'This email is to inform you that a subscription has failed on your website:', 'give-recurring' ) . ' {site_url}' . ".\n\n";
			$default_email_body .= '<strong>' . __( 'Donor:', 'give-recurring' ) . '</strong> {fullname}' . "\n";
			$default_email_body .= '<strong>' . __( 'Subscription:', 'give-recurring' ) . '</strong> {donation}' . "\n";
			$default_email_body .= '<strong>' . __( 'Amount:', 'give-recurring' ) . '</strong> {amount}' . "\n";
			$default_email_body .= '<strong>' . __( 'Subscription Frequency:', 'give-recurring' ) . '</strong> {subscription_frequency}' . "\n";
			$default_email_body .= '<strong>' . __( 'Completed Donations:', 'give-recurring' ) . '</strong> {subscriptions_completed}' . "\n";
			$default_email_body .= '<strong>' . __( 'Payment Method:', 'give-recurring' ) . '</strong> {payment_method}' . "\n";
			$default_email_body .= '<strong>' . __( 'Subscription ID:', 'give-recurring' ) . '</strong> {subscription_id}' . "\n\n";
			$default_email_body .= __( 'Thank you,', 'give-recurring' ) . "\n\n";
			$default_email_body .= '{sitename}' . "\n";

			/**
			 * Filter the default subscription failed notification email message.
			 *
			 * @since 2.14.1
			 *
			 * @param string $default_email_body Default email message.
			 */
			return apply_filters( 'give_default_subscription_failed_notification_email', $default_email_body );
		}

		/**
		 * Maybe send email notification when subscription status changes.
		 *
		 * @since 4.17.0
		 *
		 * @param int   $subscription_id The ID of the subscription.
		 * @param array $subscription_details Array containing new_status, old_status, and updated.
		 *
		 * @return void
		 */
		public function maybe_send_email_notification( $subscription_id, $new_status, $old_status ) {
			if ( $new_status === $old_status ) {
				return;
			}

			$subscription = Subscription::find(absint($subscription_id));
			if ( ! $subscription || ! $subscription->status->isFailing() ) {
				return;
			}

			if ( ! give_is_setting_enabled( $this->get_notification_status() ) ) {
				return;
			}

			/**
			 * Fires when a subscription status is changed to failed.
			 *
			 * @param int $subscription_id Subscription ID.
			 *
			 * @since 4.17.0
			 */
			do_action( "give_{$this->config['id']}_email_notification", $subscription_id );
		}

		/**
		 * Get email subject.
		 *
		 * @since 4.17.0
		 *
		 * @param int $form_id
		 *
		 * @return string
		 */
		public function get_email_subject( $form_id = null ) {
			$subject = wp_strip_all_tags(
				Give_Email_Notification_Util::get_value(
					$this,
					Give_Email_Setting_Field::get_prefix( $this, $form_id ) . 'email_subject',
					$form_id,
					$this->config['default_email_subject']
				)
			);

			/**
			 * Filters the subscription failed notification subject.
			 *
			 * @since 2.14.1
			 */
			$subject = apply_filters( "give_{$this->config['id']}_get_email_subject", $subject, $this, $form_id );

			return $subject;
		}

		/**
		 * Get email message.
		 *
		 * @since 2.14.1
		 *
		 * @param int $form_id
		 *
		 * @return string
		 */
		public function get_email_message( $form_id = null ) {
			$message = Give_Email_Notification_Util::get_value(
				$this,
				Give_Email_Setting_Field::get_prefix( $this, $form_id ) . 'email_message',
				$form_id,
				$this->config['default_email_message']
			);

			/**
			 * Filter the email message
			 *
			 * @since 2.14.1
			 */
			$message = apply_filters(
				"give_{$this->config['id']}_get_default_email_message",
				$message,
				$this,
				$form_id
			);

			return $message;
		}

		/**
		 * Get email attachments.
		 *
		 * @since 2.14.1
		 *
		 * @param int $form_id
		 *
		 * @return array
		 */
		public function get_email_attachments( $form_id = null ) {
			/**
			 * Filters the subscription failed notification email attachments.
			 * By default, there is no attachment but plugins can hook in to provide one or multiple.
			 *
			 * @since 2.14.1
			 */
			$attachments = apply_filters(
				"give_{$this->config['id']}_get_email_attachments",
				array(),
				$this,
				$form_id
			);

			return $attachments;
		}

		/**
		 * Setup email data.
		 *
		 * @since 4.17.0
		 */
		public function setup_email_data() {
			/**
			 * Filters the from name.
			 *
			 * @since 4.17.0
			 */
			$from_name = apply_filters(
				'give_subscription_failed_from_name',
				Give()->emails->get_from_name(),
				$this->subscription->id,
				null
			);

			/**
			 * Filters the from email.
			 *
			 * @since 4.17.0
			 */
			$from_email = apply_filters(
				'give_subscription_failed_from_address',
				Give()->emails->get_from_address(),
				$this->subscription->id,
				null
			);

			Give()->emails->__set( 'from_name', $from_name );
			Give()->emails->__set( 'from_email', $from_email );

			/**
			 * Filters the subscription failed notification email headers.
			 *
			 * @since 4.17.0
			 */
			$headers = apply_filters(
				'give_subscription_failed_notification_headers',
				Give()->emails->get_headers(),
				$this->subscription->id,
				null
			);

			Give()->emails->__set( 'headers', $headers );
		}

		/**
		 * Setup email notification.
		 *
		 * @since 4.17.0
		 *
		 * @param int $subscription_id
		 */
		public function setup_email_notification( $subscription_id ) {
			// Get the Subscription object.
			$this->subscription =  Subscription::find(absint($subscription_id));

			if ( ! $this->subscription || ! $this->subscription->id > 0 ) {
				wp_die(
					esc_html__( 'Cheatin&#8217; uh?', 'give-recurring' ),
					esc_html__( 'Error', 'give-recurring' ),
					array(
						'response' => 400,
					)
				);
			}

			// Set email data.
			$this->setup_email_data();

			// Send email.
			$this->send_email_notification(
				array(
					'subscription_id' => $this->subscription->id,
					'payment_id'      => $this->subscription->initialDonation()->id,
					'user_id'         => $this->subscription->donorId,
					'form_id'         => $this->subscription->donationFormId,
				)
			);
		}
	}

endif; // End class_exists check

return Give_Subscription_Failed_Admin_Email::get_instance();
