<?php

namespace GiveRecurring\Webhooks\Stripe\Listeners;

use Give\PaymentGateways\Gateways\Stripe\StripePaymentElementGateway\Webhooks\Listeners\StripeWebhookListenerRepository;
use Give\PaymentGateways\Gateways\Stripe\Webhooks\StripeEventListener;
use Give_Subscription;
use Stripe\Event;

/**
 * Class InvoicePaymentFailed
 * @package GiveRecurring\Webhooks\Stripe\Listeners
 *
 * @since 1.12.6
 */
class InvoicePaymentFailed extends StripeEventListener
{
    use StripeWebhookListenerRepository;

    /**
     * Processes invoice.payment_failed event.
     *
     * @since 2.18.0 Added $formId parameter and stripe_account option for Stripe Connect support
     * @since 2.15.0 Add support for Stripe API version 2025-03-31.basil and later versions
     * @since 1.12.6
     *
     * @param Event $event Stripe Event received via webhooks.
     *
     */
    public function processEvent(Event $event)
    {
        /**
         * @since 2.4.0
         */
        do_action('give_recurring_stripe_processing_invoice_payment_failed', $event);

        /* @var Invoice $invoice */
        $invoice = $event->data->object;

        $gatewaySubscriptionId = $this->getGatewaySubscriptionId($invoice);
        $subscription = give_recurring_get_subscription_by('profile', $gatewaySubscriptionId);

        if (!$subscription || !$subscription->id) {
            return;
        }

        /**
        * This checking is necessary because the invoice data returned in webhook events
        * can be incomplete and may not include the charge property, especially
        * with newer Stripe API versions like 2025-03-31.basil. By making a direct
        * API call to retrieve the invoice, we ensure we get all properties including
        * the payment_intent which is required for processing this webhook.
        */
        if (is_null($invoice->charge)) {
            $invoice = $this->getCompleteInvoiceFromStripe($event->data->object->id, $subscription->form_id);
        }

        $subscription->set_transaction_id($invoice->charge);

        /**
         * This action hook will be used to extend processing the invoice payment failed event.
         *
         * @since 1.9.4
         */
        do_action('give_recurring_stripe_process_invoice_payment_failed', $event);

        if (
            $invoice->attempted &&
            !$invoice->paid &&
            null !== $invoice->next_payment_attempt
        ) {
            $this->triggerFailedEmailNotificationEvent($subscription, $invoice);

            // Log the invoice object for debugging purpose.
            give_stripe_record_log(
                esc_html__('Subscription - Renewal Payment Failed', 'give-recurring'),
                print_r($invoice, true)
            );

            give_recurring_update_subscription_status($subscription->id, 'failing');
        }

        if (in_array(get_post_status($subscription->parent_payment_id), ['pending', 'processing'])) {
            give_update_payment_status($subscription->parent_payment_id, 'failed');
        }
    }

    /**
     * @since 2.18.0 Use getGatewaySubscriptionId to support Stripe API version 2025-03-31.basil and later versions
     * @since 2.0.0
     * @inerhitDoc
     */
    protected function getFormId(Event $event)
    {
        /* @var Invoice $invoice */
        $invoice = $event->data->object;
        $gatewaySubscriptionId = $this->getGatewaySubscriptionId($invoice);

        return (new Give_Subscription($gatewaySubscriptionId, true))->form_id;
    }

    /**
     * @since 1.12.6
     *
     * @param Give_Subscription $subscription
     * @param object $invoice
     */
    private function triggerFailedEmailNotificationEvent($subscription, $invoice)
    {
        do_action('give_donor-subscription-payment-failed_email_notification', $subscription, $invoice);
    }
}
