<?php

namespace GiveRecurring\PaymentGateways\Stripe\Traits;

use DateTime;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\PaymentGateways\Stripe\ApplicationFee;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;

/**
 * @since 2.10.0
 */
trait CanPauseStripeSubscription
{
    /**
     * @throws PaymentGatewayException
     */
    public function pauseSubscription(Subscription $subscription, array $data): void
    {
        $intervalInMonths = $data['interval_in_months'] ? (int)$data['interval_in_months'] : null;

        $this->pauseStripeSubscription($subscription, $intervalInMonths);
    }

    /**
     * @since 2.16.0 Make intervalInMonths nullable
     * @since 2.10.0
     *
     * @inerhitDoc
     * @throws PaymentGatewayException
     */
    public function pauseStripeSubscription(Subscription $subscription, ?int $intervalInMonths = null): void
    {
        try {
            $this->setupStripeApp($subscription->donationFormId);

            $resumesAt = $intervalInMonths ? $subscription->renewsAt->modify(
                "+{$intervalInMonths} months"
            ) : null;

            $stripeSubscription = \Stripe\Subscription::update(
                $subscription->gatewaySubscriptionId,
                [
                    'proration_behavior'   => 'none',
                    'billing_cycle_anchor' => 'unchanged',
                    'pause_collection'     => [
                        'behavior'   => 'void',
                        'resumes_at' => $resumesAt ? $resumesAt->format('U') : null,
                    ],
                ]
            );

            if ( ! $stripeSubscription->pause_collection) {
                throw new PaymentGatewayException(
                    'Unable to pause subscription with Stripe. No pause_collection object returned.'
                );
            }

            if ($resumesAt) {
                $subscription->renewsAt = $resumesAt;
            }

            $subscription->status = SubscriptionStatus::PAUSED();
            $subscription->save();
        } catch (\Exception $exception) {
            throw new PaymentGatewayException(
                sprintf(
                    'Unable to pause subscription with Stripe. %s',
                    $exception->getMessage()
                ),
                $exception->getCode(),
                $exception
            );
        }
    }

    /**
     * @since 2.10.0
     *
     * @inerhitDoc
     * @throws PaymentGatewayException
     */
    public function resumeSubscription(Subscription $subscription): void
    {
        try {
            $this->setupStripeApp($subscription->donationFormId);

            $stripeSubscription = \Stripe\Subscription::update(
                $subscription->gatewaySubscriptionId,
                [
                    'pause_collection' => '',
                ]
            );

            if ($stripeSubscription->pause_collection) {
                throw new PaymentGatewayException(
                    'Unable to resume subscription with Stripe. pause_collection object returned.'
                );
            }

            $renewsAt = $subscription->renewsAt;
            $currentDate = new DateTime();
            $renewsAt->setDate($currentDate->format('Y'), $currentDate->format('m'), (int)$renewsAt->format('d'));;

            while ($renewsAt < $currentDate) {
                $renewsAt->modify("+{$subscription->frequency} {$subscription->period}");
            }

            $subscription->status = SubscriptionStatus::ACTIVE();
            $subscription->save();
        } catch (\Exception $exception) {
            throw new PaymentGatewayException(
                sprintf(
                    'Unable to resume subscription with Stripe. %s',
                    $exception->getMessage()
                ),
                $exception->getCode(),
                $exception
            );
        }
    }

    /**
     * For Stripe, we can pause a subscription if the application fee is not enabled which means the Stripe Pro add-on is present, active, installed, or has a license.
     *
     * @since 2.10.0
     *
     * @inerhitDoc
     */
    public function canPauseSubscription(): bool
    {
        $isPausingSubscriptionEnabled = give_is_setting_enabled(
            give_get_option('stripe_subscription_pause', 'enabled')
        );

        return !ApplicationFee::canAddFee() && $isPausingSubscriptionEnabled;
    }
}
