<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Paystack;

use Give\Donations\Models\Donation;
use Give\Donations\Models\DonationNote;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\Exceptions\Primitives\Exception;
use Give\Framework\Http\Response\Types\RedirectResponse;
use Give\Framework\PaymentGateways\Commands\GatewayCommand;
use Give\Framework\PaymentGateways\Commands\RedirectOffsite;
use Give\Framework\PaymentGateways\Commands\SubscriptionSynced;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionAmountEditable;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionDashboardLinkable;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionTransactionsSynchronizable;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\SubscriptionModule;
use Give\Framework\Support\ValueObjects\Money;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\Models\SubscriptionNote;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use GiveRecurring\PaymentGatewayModules\Modules\Paystack\Actions\CreatePaystackPlan;
use GiveRecurring\PaymentGatewayModules\Modules\Paystack\Actions\SyncPaystackSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Paystack\DataTransferObjects\PaystackPlan;
use GiveRecurring\PaymentGatewayModules\Modules\Paystack\Traits\PaystackApiTrait;

/**
 * @since 2.16.0
 */
class PaystackGatewaySubscriptionModule extends SubscriptionModule implements
    SubscriptionDashboardLinkable,
    SubscriptionAmountEditable,
    SubscriptionTransactionsSynchronizable
{
    use PaystackApiTrait;

    /**
     * Routes for handling Paystack callbacks
     *
     * @since 2.16.0
     * @inheritDoc
     */
    public $secureRouteMethods = [
        'handleSubscriptionReturn',
    ];

    /**
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    public function synchronizeSubscription(Subscription $subscription): SubscriptionSynced
    {
        return give(SyncPaystackSubscription::class)->handle($subscription);
    }

    /**
     * @since 2.16.0
     *
     * @throws Exception
     * @throws PaymentGatewayException
     */
    public function createSubscription(
        Donation $donation,
        Subscription $subscription,
        $gatewayData
    ): GatewayCommand {
        /**
         * Create Paystack plan
         */
        $plan = $this->createPaystackPlan($subscription);

        /**
         * Initialize transaction with plan code for subscription
         */
        $reference = $donation->purchaseKey;

        $initializeResponse = $this->initializeSubscriptionTransaction(
            $donation,
            $subscription,
            $plan,
            $reference,
            $gatewayData
        );

        if (!isset($initializeResponse['authorization_url'])) {
            throw new PaymentGatewayException(
                __('Unable to initialize Paystack subscription transaction.', 'give-recurring')
            );
        }

        /**
         * Store reference for later verification
         */
        give()->subscription_meta->update_meta($subscription->id, 'give_paystack_reference', $reference);

        give()->subscription_meta->update_meta($subscription->id, 'give_paystack_plan_id', $plan->id);

        give()->subscription_meta->update_meta($subscription->id, 'give_paystack_plan_code', $plan->planCode);

        give()->payment_meta->update_meta($donation->id, 'give_paystack_access_code', $initializeResponse['access_code']);

        return new RedirectOffsite($initializeResponse['authorization_url']);
    }

    /**
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    public function cancelSubscription(Subscription $subscription)
    {
        try {
            $paystackSubscriptionEmailToken = give()->subscription_meta->get_meta($subscription->id, 'give_paystack_email_token', true);

            if (empty($paystackSubscriptionEmailToken)) {
                $paystackSubscriptionRequest = $this->makePaystackApiRequest(
                    'GET',
                    "subscription/{$subscription->gatewaySubscriptionId}"
                );

                if (!isset($paystackSubscriptionRequest['status']) || $paystackSubscriptionRequest['status'] === false) {
                    throw new PaymentGatewayException(
                        __('Unable to get Paystack subscription.', 'give-recurring')
                    );
                }

                $paystackSubscription = $paystackSubscriptionRequest['data'];
                $paystackSubscriptionEmailToken = (string)$paystackSubscription['email_token'];

                give()->subscription_meta->update_meta($subscription->id, 'give_paystack_email_token', $paystackSubscriptionEmailToken);
            }

            $response = $this->makePaystackApiRequest(
                'POST',
                "subscription/disable",
                [
                    'code' => $subscription->gatewaySubscriptionId,
                    'token' => $paystackSubscriptionEmailToken
                ]
            );

            if (!isset($response['status']) || $response['status'] === false) {
                throw new PaymentGatewayException(
                    isset($response['message']) ? $response['message'] : __('Unable to cancel subscription', 'give-recurring')
                );
            }

            $subscription->status = SubscriptionStatus::CANCELLED();
            $subscription->save();

            SubscriptionNote::create([
                'subscriptionId' => $subscription->id,
                'content' => __('Subscription cancelled in Paystack', 'give-recurring')
            ]);
        } catch (\Exception $exception) {
            throw new PaymentGatewayException(
                sprintf(
                    'Unable to cancel subscription with Paystack. %s',
                    $exception->getMessage()
                ),
                $exception->getCode(),
                $exception
            );
        }
    }

    /**
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    public function updateSubscriptionAmount(Subscription $subscription, Money $newRenewalAmount)
    {
        try {
            // Update the subscription amount locally
            $subscription->amount = $newRenewalAmount;

            $planId = give()->subscription_meta->get_meta($subscription->id, 'give_paystack_plan_id', true);

            // Update the subscription with the new plan
            $response = $this->makePaystackApiRequest(
                'PUT',
                "plan/{$planId}",
                [
                    'amount' => $newRenewalAmount->formatToMinorAmount(),
                ]
            );

            if (!isset($response['status']) || $response['status'] === false) {
                throw new PaymentGatewayException(
                    isset($response['message']) ? $response['message'] : __('Unable to update subscription amount', 'give-recurring')
                );
            }

            $subscription->save();

            SubscriptionNote::create([
                'subscriptionId' => $subscription->id,
                'content' => sprintf(
                    __('Subscription amount updated to %s in Paystack', 'give-recurring'),
                    $newRenewalAmount->formatToDecimal()
                )
            ]);
        } catch (\Exception $e) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Unable to update Paystack subscription amount: %s', 'give-recurring'),
                    $e->getMessage()
                )
            );
        }
    }

    /**
     * @since 2.16.0
     */
    public function gatewayDashboardSubscriptionUrl(Subscription $subscription): string
    {
        return esc_url("https://dashboard.paystack.com/#/subscriptions/{$subscription->gatewaySubscriptionId}");
    }

    /**
     * Create or retrieve a Paystack plan
     *
     * @param Subscription $subscription
     * @return PaystackPlan
     * @throws PaymentGatewayException
     */
    protected function createPaystackPlan(Subscription $subscription): PaystackPlan
    {
        return give(CreatePaystackPlan::class)->handle($subscription);
    }

    /**
     * Initialize a subscription transaction using Paystack's transaction/initialize endpoint
     *
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    protected function initializeSubscriptionTransaction(
        Donation $donation,
        Subscription $subscription,
        PaystackPlan $plan,
        string $reference,
        array $gatewayData
    ): array {
        $email = $donation->email;
        $currency = $donation->amount->getCurrency()->getCode();

        $redirectUrl = $this->gateway->generateSecureGatewayRouteUrl(
            'handleSubscriptionReturn',
            $donation->id,
            [
                'givewp-donation-id' => $donation->id,
                'givewp-subscription-id' => $subscription->id,
                'givewp-success-url' => $gatewayData['successUrl']
            ]
        );

        $initializeData = [
            'amount' => $subscription->amount->formatToMinorAmount(),
            'email' => $email,
            'currency' => $currency,
            'reference' => $reference,
            'plan' => $plan->planCode,
            'callback_url' => $redirectUrl,
            'metadata' => apply_filters('givewp_paystack_subscription_transaction_initialization_metadata', [
                'plugin' => 'GiveWP',
                'custom_fields' => [
                    [
                        'display_name' => 'Plugin',
                        'variable_name' => 'plugin',
                        'value' => 'GiveWP'
                    ],
                    [
                        'display_name' => 'GiveWP Subscription ID',
                        'variable_name' => 'givewp_subscription_id',
                        'value' => $subscription->id
                    ],
                    [
                        'display_name' => 'GiveWP Donation ID',
                        'variable_name' => 'givewp_donation_id',
                        'value' => $donation->id
                    ],
                    [
                        'display_name' => 'GiveWP Form ID',
                        'variable_name' => 'givewp_form_id',
                        'value' => $subscription->donationFormId
                    ],
                ],
            ]),
        ];

        $response = $this->makePaystackApiRequest('POST', 'transaction/initialize', $initializeData);

        if (!isset($response['status']) || $response['status'] === false) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Unable to initialize Paystack subscription transaction: %s', 'give-recurring'),
                    isset($response['message']) ? $response['message'] : __('Unknown error', 'give-recurring')
                )
            );
        }

        return $response['data'];
    }

    /**
     * Handle subscription verification after Paystack redirect
     * This will be called when Paystack redirects back after payment authorization
     *
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    public function handleSubscriptionReturn(array $queryParams): RedirectResponse
    {
        $donationId = (int)$queryParams['givewp-donation-id'];
        $subscriptionId = (int)$queryParams['givewp-subscription-id'];
        $successUrl = $queryParams['givewp-success-url'];

        $donation = Donation::find($donationId);
        $subscription = Subscription::find($subscriptionId);

        if (!$donation || !$subscription) {
            throw new PaymentGatewayException(__('Donation or subscription not found.', 'give-recurring'));
        }

        $reference = give()->subscription_meta->get_meta($subscription->id, 'give_paystack_reference', true);

        if (empty($reference)) {
            throw new PaymentGatewayException(__('No Paystack reference found for this payment.', 'give-recurring'));
        }

        try {
            // Verify the transaction
            $verifyTransactionResponse = $this->makePaystackApiRequest('GET', "transaction/verify/{$reference}");

            if (!isset($verifyTransactionResponse['status']) || $verifyTransactionResponse['status'] === false) {
                throw new PaymentGatewayException(
                    __('Unable to verify Paystack transaction.', 'give-recurring')
                );
            }

            $transactionId = (string)$verifyTransactionResponse['data']['id'];
            $status = (string)$verifyTransactionResponse['data']['status'];


            // Update donation status
            $donation->status = $this->getDonationStatusFromPaystackTransactionStatus($status);
            $donation->gatewayTransactionId = $transactionId;
            $donation->save();

            $planId = give()->subscription_meta->get_meta($subscription->id, 'give_paystack_plan_id', true);

            $planResponse = $this->makePaystackApiRequest('GET', "plan/{$planId}");

            if (!isset($planResponse['status']) || $planResponse['status'] === false) {
                throw new PaymentGatewayException(
                    __('Unable to get Paystack plan.', 'give-recurring')
                );
            }

            $gatewaySubscription = $planResponse['data']['subscriptions'][0];

            $subscription->gatewaySubscriptionId = (string)$gatewaySubscription['subscription_code'];

            give()->subscription_meta->update_meta($subscription->id, 'give_paystack_email_token', (string)$gatewaySubscription['email_token']);

            $subscription->status = $this->getSubscriptionStatusFromPaystackSubscriptionStatus($status);

            $subscription->save();

            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(
                    __('Paystack subscription transaction completed. Transaction ID: %s', 'give-recurring'),
                    $transactionId
                )
            ]);

            return new RedirectResponse(esc_url_raw($successUrl));
        } catch (\Exception $e) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Paystack subscription verification error: %s', 'give-recurring'),
                    $e->getMessage()
                )
            );
        }
    }

    /**
     * @since 2.16.0
     *
     * @see https://paystack.com/docs/payments/verify-payments/#transaction-statuses
     */
    protected function getDonationStatusFromPaystackTransactionStatus(string $status): DonationStatus
    {
        switch ($status) {
            case 'abandoned':
                return DonationStatus::ABANDONED();
            case 'failed':
                return DonationStatus::FAILED();
            case 'reversed':
                return DonationStatus::REFUNDED();
            case 'success':
                return DonationStatus::COMPLETE();
            default:
                return DonationStatus::PROCESSING();
        }
    }

    /**
     * @since 2.16.0
     *
     * @see https://paystack.com/docs/payments/subscriptions/#understanding-subscription-statuses
     */
    protected function getSubscriptionStatusFromPaystackSubscriptionStatus(string $status): SubscriptionStatus
    {
        switch ($status) {
            case 'completed':
                return SubscriptionStatus::COMPLETED();
            case 'attention':
                return SubscriptionStatus::FAILING();
            case 'cancelled':
                return SubscriptionStatus::CANCELLED();
            case 'non-renewing':
                return SubscriptionStatus::ACTIVE();
            case 'active':
                return SubscriptionStatus::ACTIVE();
            default:
                return SubscriptionStatus::ACTIVE();
        }
    }
}
