<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Paystack\Actions;

use DateTime;
use Give\Donations\Models\Donation;
use Give\Framework\PaymentGateways\Commands\SubscriptionSynced;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\Support\ValueObjects\Money;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\Models\SubscriptionNote;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use GiveRecurring\PaymentGatewayModules\Modules\Paystack\Traits\PaystackApiTrait;

/**
 * @since 2.16.0
 */
class SyncPaystackSubscription
{
    use PaystackApiTrait;

    /**
     * Handle synchronizing a Paystack subscription
     *
     * @since 2.16.0
     *
     * @throws PaymentGatewayException
     */
    public function handle(Subscription $subscription)
    {
        try {
            $response = $this->makePaystackApiRequest(
                'GET',
                "subscription/{$subscription->gatewaySubscriptionId}"
            );

            if (!isset($response['status']) || $response['status'] !== true) {
                throw new PaymentGatewayException(
                    sprintf(
                        __('Unable to fetch Paystack subscription: %s', 'give-recurring'),
                        isset($response['message']) ? $response['message'] : __('Unknown error', 'give-recurring')
                    )
                );
            }

            $paystackSubscription = $response['data'];
            $invoices = $paystackSubscription['invoices'];

            // Update local subscription status based on Paystack status
            $this->updateSubscriptionStatus($subscription, $paystackSubscription);

            // Create a note about the sync
            SubscriptionNote::create([
                'subscriptionId' => $subscription->id,
                'content' => sprintf(
                    __('Subscription synchronized with Paystack. Status: %s', 'give-recurring'),
                    $paystackSubscription['status']
                )
            ]);

            // Create missing renewal donations and get present donations
            [$missingDonations, $presentDonations] = $this->processMissingRenewals($subscription, $invoices);

            return new SubscriptionSynced(
                $subscription,
                $missingDonations,
                $presentDonations,
                __('Subscription synchronized successfully with Paystack', 'give-recurring')
            );
        } catch (\Exception $e) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Failed to synchronize Paystack subscription: %s', 'give-recurring'),
                    $e->getMessage()
                )
            );
        }
    }

    /**
     * Process missing renewal donations from Paystack invoices
     *
     * @since 2.16.0
     *
     * @throws \Exception
     */
    protected function processMissingRenewals(Subscription $subscription, array $invoices): array
    {
        $missingDonations = [];
        $presentDonations = $subscription->donations;
        $existingTransactionIds = [];

        // Get existing transaction IDs
        foreach ($presentDonations as $donation) {
            if ($donation->gatewayTransactionId) {
                $existingTransactionIds[] = $donation->gatewayTransactionId;
            }
        }

        // Process each invoice to create missing renewals
        foreach ($invoices as $invoice) {
            $transactionId = (string)($invoice['id'] ?? '');
            // Skip if we already have a donation for this transaction
            if (in_array($transactionId, $existingTransactionIds)) {
                continue;
            }

            // Create renewal donation
            $renewalDonation = $this->createRenewalDonationFromInvoice($subscription, $invoice);
            if ($renewalDonation) {
                $missingDonations[] = $renewalDonation;
                $existingTransactionIds[] = $renewalDonation->gatewayTransactionId;
            }
        }

        return [$missingDonations, $presentDonations];
    }

    /**
     * Create a renewal donation from a Paystack invoice
     *
     * @since 2.16.0
     *
     * @throws \Exception
     */
    protected function createRenewalDonationFromInvoice(Subscription $subscription, array $invoice): ?Donation
    {
        $transactionId = (string)($invoice['id'] ?? '');
        $amount = $invoice['amount'] ?? $subscription->amount->getAmount();
        $currency = $invoice['currency'] ?? $subscription->amount->getCurrency();
        $date = $invoice['created_at'] ?? '';

        $args = [
            'gatewayTransactionId' => $transactionId,
            'createdAt' => new DateTime($date),
        ];

        $renewalAmount = new Money($amount, $currency);

        if (!$subscription->amount->equals($renewalAmount)) {
            $args['amount'] = $renewalAmount;
        }

        return $subscription->createRenewal($args);
    }

    /**
     * Update local subscription status based on Paystack subscription status
     *
     * @since 2.16.0
     *
     * @throws \Exception
     */
    protected function updateSubscriptionStatus(Subscription $subscription, array $paystackSubscription): void
    {
        $paystackStatus = $paystackSubscription['status'];
        $originalStatus = $subscription->status;

        switch ($paystackStatus) {
            case 'active':
                $subscription->status = SubscriptionStatus::ACTIVE();
                break;
            case 'non-renewing':
                $subscription->status = SubscriptionStatus::ACTIVE();
                break;
            case 'cancelled':
                $subscription->status = SubscriptionStatus::CANCELLED();
                break;
            case 'completed':
                $subscription->status = SubscriptionStatus::COMPLETED();
                break;
            case 'attention':
                // This status indicates issues with renewal (e.g., expired cards)
                $subscription->status = SubscriptionStatus::FAILING();
                break;
            default:
                // Keep current status for unknown statuses
                break;
        }

        // Only save if status changed
        if (!$originalStatus->equals($subscription->status)) {
            $subscription->save();
        }
    }
}
