<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\GoCardless\Actions;

use Give\Subscriptions\Models\Subscription;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Subscriptions\ValueObjects\SubscriptionPeriod;
use GiveRecurring\PaymentGatewayModules\Modules\GoCardless\DataTransferObjects\GoCardlessSubscription;

/**
 * @since 2.14.0
 */
class CreateGoCardlessSubscription
{
    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function __invoke(Subscription $subscription, string $mandateId, array $options = []): GoCardlessSubscription
    {
        switch ($subscription->period->getValue()) {
            case SubscriptionPeriod::WEEK:
                $intervalUnit = 'weekly';
                break;
            case SubscriptionPeriod::MONTH:
                $intervalUnit = 'monthly';
                break;
            case SubscriptionPeriod::YEAR:
                $intervalUnit = 'yearly';
                break;
            default:
                throw new PaymentGatewayException(__('Invalid subscription period.', 'give-gocardless'));
        }

        $params = [
            'amount' => $subscription->amount->formatToMinorAmount(),
            'interval_unit' =>  $intervalUnit,
            'interval' => $subscription->frequency,
            'currency' => $subscription->amount->getCurrency()->getCode(),
            'links' => [
                'mandate' => $mandateId,
            ],
        ];

        if ($subscription->installments > 0) {
            $params['count'] = $subscription->installments;
        }

        // Merge any extra options
        $params = array_merge($params, $options);

        $response = \Give_GoCardless_API::create_subscription($params);

        if (is_wp_error($response) || isset($response['error'])) {
            throw new PaymentGatewayException(sprintf(
                __('Error to create subscription: %s - %s', 'give-gocardless'),
                isset($response['error']['code']) ? $response['error']['code'] : $response->get_error_code(),
                isset($response['error']['message']) ? $response['error']['message'] : $response->get_error_message()
            ));
        }

        if (empty($response['subscriptions']['id'])) {
            throw new PaymentGatewayException(__('Unexpected subscription response from GoCardless', 'give-gocardless'));
        }

        return GoCardlessSubscription::fromArray($response['subscriptions']);
    }
}
