<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Braintree\Actions;

use Braintree\Result\Error;
use Braintree\Result\Successful;
use Exception;
use Give\Donations\Models\Donation;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Subscriptions\Models\Subscription;
use GiveBraintree\Actions\ValidateBraintreeMerchantAccountCurrency;
use GiveBraintree\API\BraintreeApiFacade;
use GiveBraintree\DataTransferObjects\BraintreeGatewayData;
use GiveRecurring\PaymentGatewayModules\Modules\Braintree\DataTransferObjects\BraintreeSubscription;

/**
 * @since 2.8.0
 */
class CreateBraintreeSubscription
{
    /**
     * @var Donation
     */
    private $donation;

    /**
     * @var Subscription
     */
    private $subscription;

    /**
     * @var BraintreeGatewayData
     */
    private $braintreeGatewayData;

    /**
     * @var BraintreeApiFacade
     */
    private $braintreeApi;

    /**
     * @since 2.8.0
     */
    public function __construct(BraintreeApiFacade $braintreeApi)
    {
        $this->braintreeApi = $braintreeApi;
    }

    /**
     * @since 2.8.0
     *
     * @throws Exception
     */
    public function __invoke(
        Donation $donation,
        Subscription $subscription,
        BraintreeGatewayData $braintreeGatewayData
    ): BraintreeSubscription
    {
        give(ValidateBraintreeMerchantAccountCurrency::class)($donation, $this->braintreeApi);

        $this->donation = $donation;
        $this->subscription = $subscription;
        $this->braintreeGatewayData = $braintreeGatewayData;

        /**
         * @see https://developer.paypal.com/braintree/docs/reference/request/subscription/create/php
         *
         * @var Successful|Error $result
         */
        $result = $this->braintreeApi->creteSubscription([
            'paymentMethodToken' => $this->getBraintreePaymentMethodToken(),
            'planId' => $this->getBraintreePlanId(),
            'options' => [
                'startImmediately' => true,
            ],
        ]);

        if ( ! $result->success) {
            $this->braintreeApi->logResultErrors($result, $this->donation, 'Unable to create Subscription');

            throw new PaymentGatewayException(__('There was a problem creating your Braintree Subscription. Please contact the site admin.',
                'give-braintree'));
        }

        return BraintreeSubscription::fromArray([
            'transactionId' => $result->subscription->transactions[0]->id,
            'id' => $result->subscription->id,
            'status' => $result->subscription->status,
        ]);
    }

    /**
     * @since 2.8.0
     *
     * @throws Exception
     */
    private function getBraintreePaymentMethodToken(): string
    {
        $paymentMethodArgs = [
            'customerId' => $this->braintreeApi->getCustomerId($this->donation),
            'paymentMethodNonce' => $this->braintreeGatewayData->paymentMethodNonce,
            'options' => [
                'makeDefault' => true,
            ],
        ];

        if ($this->donation->billingAddress->country) {
            $paymentMethodArgs['billingAddress'] = [
                'streetAddress' => $this->donation->billingAddress->address1,
                'extendedAddress' => $this->donation->billingAddress->address2,
                'locality' => $this->donation->billingAddress->city,
                'region' => $this->donation->billingAddress->state,
                'postalCode' => $this->donation->billingAddress->zip,
                'countryCodeAlpha2' => $this->donation->billingAddress->country,
            ];
        }

        /**
         * @see https://developer.paypal.com/braintree/docs/reference/request/payment-method/create/php
         *
         * @var Successful|Error $result
         */
        $result = $this->braintreeApi->cretePaymentMethod($paymentMethodArgs);

        if ( ! $result->success) {
            $this->braintreeApi->logResultErrors($result, $this->donation, 'Unable to create Payment Method');

            throw new PaymentGatewayException(__('There was a problem creating your Braintree Payment Method. Please contact the site admin.',
                'give-braintree'));
        }

        return $result->paymentMethod->token;
    }

    /**
     * @since 2.8.0
     *
     * @throws Exception
     */
    private function getBraintreePlanId(): string
    {
        $planArgs = [
            'name' => $this->getBraintreePlanName(),
            'billingFrequency' => $this->getBraintreeBillingFrequency(),
            'currencyIsoCode' => $this->subscription->amount->getCurrency()->getCode(),
            'price' => $this->subscription->amount->formatToDecimal(),
        ];

        if ($this->subscription->installments > 0) {
            $planArgs['numberOfBillingCycles'] = $this->subscription->installments;
        }

        /**
         * @see https://developer.paypal.com/braintree/docs/reference/request/plan/create/php
         *
         * @var Successful|Error $result
         */
        $result = $this->braintreeApi->cretePlan($planArgs);

        if ( ! $result->success) {
            $this->braintreeApi->logResultErrors($result, $this->donation, 'Unable to create Plan');

            throw new PaymentGatewayException(__('There was a problem creating your Braintree Plan. Please contact the site admin.',
                'give-braintree'));
        }

        return $result->plan->id;
    }

    /**
     * @since 2.8.0
     */
    private function getBraintreePlanName(): string
    {
        return $this->donation->formTitle . " (Subscription ID: " . $this->subscription->id . ")";
    }

    /**
     * @since 2.8.0
     *
     * @throws PaymentGatewayException
     */
    private function getBraintreeBillingFrequency(): int
    {
        $billingFrequency = 0;

        if ($this->subscription->period->isDay()) {
            throw new PaymentGatewayException(__('The Braintree gateway does not support daily subscriptions. Please contact the site admin.',
                'give-braintree'));
        }

        if ($this->subscription->period->isWeek()) {
            throw new PaymentGatewayException(__('The Braintree gateway does not support weekly subscriptions. Please contact the site admin.',
                'give-braintree'));
        }

        if ($this->subscription->period->isMonth()) {
            $billingFrequency = 1;
        }

        if ($this->subscription->period->isQuarter()) {
            $billingFrequency = 3;
        }

        if ($this->subscription->period->isYear()) {
            $billingFrequency = 12;
        }

        return $billingFrequency;
    }
}
