<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Blink;

use Exception;
use Give\Donations\Models\Donation;
use Give\Donations\Models\DonationNote;
use Give\Donations\ValueObjects\DonationStatus;
use Give\Framework\Http\Response\Types\RedirectResponse;
use Give\Framework\PaymentGateways\Commands\GatewayCommand;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionDashboardLinkable;
use Give\Framework\PaymentGateways\Contracts\Subscription\SubscriptionPaymentMethodEditable;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give\Framework\PaymentGateways\SubscriptionModule;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use GiveBlink\Actions\CreateBlinkIntent;
use GiveBlink\Actions\HandleCreditCardResponse;
use GiveBlink\Actions\ValidatePaymentMethod;
use GiveBlink\API\BlinkApi;
use GiveBlink\DataTransferObjects\GatewayDataDTO;
use GiveBlink\Gateway\BlinkGateway;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\Actions\CancelBlinkSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\Actions\CreateBlinkSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\Actions\HandleBlinkSubscriptionWebhookRequest;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\Actions\UpdateBlinkSubscription;
use GiveRecurring\PaymentGatewayModules\Modules\Blink\DataTransferObjects\BlinkSubscriptionWebhookRequestDTO;

/**
 * @since 2.11.0
 */
class BlinkGatewaySubscriptionModule extends SubscriptionModule implements
    SubscriptionDashboardLinkable,
    SubscriptionPaymentMethodEditable
{
    /**
     * @since 2.11.0
     */
    public $secureRouteMethods = [
        'handleSubscriptionReturn',
    ];

    /**
     * @since 2.11.0
     */
    public $routeMethods = [
        'handleSubscriptionNotification',
    ];

    /**
     * @since 2.11.0
     *
     * @throws Exception
     */
    public function createSubscription(Donation $donation, Subscription $subscription, $gatewayData): GatewayCommand
    {
        try {
            $data = GatewayDataDTO::fromArray($gatewayData);
            $paymentMethod = $data->paymentMethod;

            (new ValidatePaymentMethod())($paymentMethod, true);

            $data->successUrl = $this->getSubscriptionReturnUrl($subscription, $data);
            $data->notificationUrl = $this->getSubscriptionNotificationUrl($subscription);

            $intent = (new CreateBlinkIntent())($donation, $data);

            $response = (new CreateBlinkSubscription())($subscription, $data, $intent);

            return (new HandleCreditCardResponse())($response);
        } catch (Exception $e) {
            $subscription->status = SubscriptionStatus::FAILING();
            $subscription->save();

            $donation->status = DonationStatus::FAILED();
            $donation->save();

            $errorMessage = $e->getMessage();

            DonationNote::create([
                'donationId' => $donation->id,
                'content' => sprintf(esc_html__('Donation failed. Reason: %s', 'give-blink'), $errorMessage),
            ]);

            throw new PaymentGatewayException($errorMessage);
        }
    }

    /**
     * @since 2.11.0
     *
     * @throws PaymentGatewayException
     */
    public function cancelSubscription(Subscription $subscription): bool
    {
        try {
            (new CancelBlinkSubscription())($subscription);
            $subscription->status = SubscriptionStatus::CANCELLED();
            $subscription->save();
        } catch (Exception $e) {
            throw new PaymentGatewayException($e->getMessage());
        }

        return true;
    }

    /**
     * @since 2.11.0
     *
     * @throws Exception
     */
    protected function handleSubscriptionReturn(array $queryParams): RedirectResponse
    {
        $subscriptionId = isset($queryParams['givewp-subscription-id']) ? sanitize_text_field(
            (int)$queryParams['givewp-subscription-id']
        ) : null;
        if (! $subscriptionId) {
            throw new Exception('Missing subscription ID.');
        }

        /** @var Subscription $subscription */
        $subscription = Subscription::find($subscriptionId);

        if (! $subscription) {
            throw new Exception('Subscription not found.');
        }

        $subscription->gatewaySubscriptionId = sanitize_text_field($queryParams['id']);
        $subscription->status = $this->getSubscriptionStatus(
            sanitize_text_field($queryParams['status'])
        );
        $subscription->save();

        $donation = $subscription->initialDonation();

        $blinkSubscription = BlinkApi::client()->getRepeatPayment($subscription->gatewaySubscriptionId);

        /**
         * TODO: Implement updating the gatewayTransactionId prop with transaction_id
         * Expected to available in Blink's API response soon.
         *
         * $donation->gatewayTransactionId = $blinkSubscription['transaction_id'] ?? '';
         */

        $initialPayment = $blinkSubscription['recurring_data'][0] ?? '';
        $donation->gatewayTransactionId = (string)($initialPayment['recurring_id'] ?? '');
        $donationStatus = give(BlinkGateway::class)->getDonationStatus($initialPayment['status'] ?? '', '');

        $donation->status = $donationStatus;
        $donation->save();

        if ($subscription->status->isActive()) {
            PaymentGatewayLog::success(
                sprintf('Subscription active. Subscription ID: %s', $subscription->id),
                [
                    'Payment Gateway' => $subscription->gateway()->getId(),
                    'Gateway Subscription Id' => $subscription->gatewaySubscriptionId,
                    'Gateway Transaction Id' => $subscription->initialDonation()->gatewayTransactionId,
                    'Subscription Id' => $subscription->id,
                    'Donation Id' => $subscription->initialDonation()->id,
                ]
            );
        }

        $redirectUrl = $subscription->status->isFailing()
            ? $queryParams['givewp-cancel-url']
            : $queryParams['givewp-return-url'];

        return new RedirectResponse($redirectUrl);
    }

    /**
     * @since 2.11.0
     *
     * @throws Exception
     */
    public function handleSubscriptionNotification(): void
    {
        try {
            $blinkSubscriptionWebhookRequest = BlinkSubscriptionWebhookRequestDTO::fromArray($_REQUEST);
            (new HandleBlinkSubscriptionWebhookRequest())($blinkSubscriptionWebhookRequest);
        } catch (\Exception $e) {
            PaymentGatewayLog::error(
                'Blink Webhook Failed. Error: ' . $e->getMessage()
            );
            esc_html_e('Blink Webhook Failed.', 'give-blink');
        }

        exit();
    }

    /**
     * @since 2.11.0
     */
    protected function getSubscriptionReturnUrl(
        Subscription $subscription,
        GatewayDataDTO $data
    ): string {
        $returnUrl = $this->gateway->generateSecureGatewayRouteUrl(
            'handleSubscriptionReturn',
            $subscription->initialDonation()->id,
            [
                'givewp-donation-id' => $subscription->initialDonation()->id,
                'givewp-subscription-id' => $subscription->id,
                'givewp-return-url' => $data->successUrl,
                'givewp-cancel-url' => $data->cancelUrl,
            ]
        );

        return apply_filters('givewp_blink_subscription_return_url', $returnUrl);
    }

    /**
     * @since 2.11.0
     */
    protected function getSubscriptionNotificationUrl(Subscription $subscription): string
    {
        $notificationUrl = $this->gateway->generateGatewayRouteUrl(
            'handleSubscriptionNotification',
            [
                'hook_type' => 'subscription',
                'payment_id' => $subscription->initialDonation()->id,
                'subscription_id' => $subscription->id,
            ]
        );

        return apply_filters('givewp_blink_subscription_notification_url', $notificationUrl);
    }

    /**
     * @since 2.11.0
     */
    protected function getSubscriptionStatus(string $status): SubscriptionStatus
    {
        $status = strtolower($status);

        switch ($status) {
            case 'active':
                return SubscriptionStatus::ACTIVE();
            case 'pending':
                return SubscriptionStatus::PENDING();
        }

        return SubscriptionStatus::FAILING();
    }


    /**
     * @since 2.11.0
     */
    public function gatewayDashboardSubscriptionUrl(Subscription $subscription): string
    {
        return esc_url('https://secure.blinkpayment.co.uk/admin/customer-centre/manage-repeat-payments');
    }

    /**
     * @since 2.11.0
     *
     * @throws PaymentGatewayException
     */
    public function updateSubscriptionPaymentMethod(Subscription $subscription, $gatewayData): void
    {
        (new UpdateBlinkSubscription())($subscription, GatewayDataDTO::fromArray($gatewayData));
    }
}
