<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\Paystack\Actions;

use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use GivePaystack\Paystack\Gateway\PaystackGateway;

/**
 * @since 2.16.0
 */
class ProcessWebhookNotifications
{
    /**
     * @since 2.16.0
     * @see https://paystack.com/docs/payments/webhooks/#types-of-events
     * @param object $request
     * @param PaystackGateway $gateway
     */
    public function __invoke($request, PaystackGateway $gateway): void
    {
        if (!isset($request->event)) {
            return;
        }

        try {
            switch ($request->event) {
                /**
                 * invoice.create: An invoice has been created for a subscription on your account. This usually happens 3 days before the subscription is due or whenever we send the customer their first pending invoice notification
                 */
                case 'invoice.create':
                /**
                 * invoice.update: An invoice has been updated. This usually means we were able to charge the customer successfully. You should inspect the invoice object returned and take necessary action.
                 */
                case 'invoice.update':
                    $gatewaySubscriptionId = $request->data->subscription->subscription_code;

                    $gatewayTransactionId = $request->data->transaction->reference;

                    $gateway->webhook->events->subscriptionRenewalDonation($gatewaySubscriptionId, $gatewayTransactionId);

                    break;
                /**
                 * invoice.payment_failed: A payment for an invoice failed
                 */
                case 'invoice.payment_failed':
                    $gatewaySubscriptionId = $request->data->subscription->subscription_code;

                    $gateway->webhook->events->subscriptionFailing($gatewaySubscriptionId);

                    break;
                /**
                 * subscription.create: A subscription has been created.  This event is sent to indicate that a subscription was created for the customer who was charged.  If you created the subscription by adding a plan code to a transaction, a charge.success event is also sent to indicate that the transaction was successful. (handled in paystack-gateway plugin)
                 */
                case 'subscription.create':
                    $gatewaySubscriptionId = $request->data->subscription_code;

                    $gateway->webhook->events->subscriptionActive($gatewaySubscriptionId);

                    break;
                /**
                 * subscription.disable: A subscription on your account has been disabled
                 *
                 * On completion of all billing cycles for a subscription, a final subscription.disable event will be sent, with status set to complete.
                 *
                 * Cancelling a subscription: On the next payment date, a subscription.disable event will be sent to indicate that the subscription has been cancelled.
                 */
                case 'subscription.disable':
                    $gatewaySubscriptionId = $request->data->subscription_code;

                    $gateway->webhook->events->subscriptionCancelled($gatewaySubscriptionId);

                    break;
                /**
                 * subscription.not_renew: A subscription on your account's status has changed to non-renewing. This means the subscription will not be charged on the next payment date.
                 *
                 * Cancelling a subscription: A subscription.not_renew event will be sent to indicate that the subscription will not renew on the next payment date.
                 */
                case 'subscription.not_renew':
                    $gatewaySubscriptionId = $request->data->subscription_code;

                    $gateway->webhook->events->subscriptionSuspended($gatewaySubscriptionId);

                    break;
            }
        } catch (\Exception $e) {
            PaymentGatewayLog::error('Paystack webhook error', ['error' => $e->getMessage()]);
        }
    }
}
