<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\GoCardless\Actions;

use DateTime;
use Exception;
use Give\Donations\Models\Donation;
use Give\Framework\PaymentGateways\Commands\SubscriptionSynced;
use Give\Framework\Support\ValueObjects\Money;
use Give\Subscriptions\Models\Subscription;
use Give\Subscriptions\ValueObjects\SubscriptionPeriod;
use Give\Subscriptions\ValueObjects\SubscriptionStatus;
use GiveRecurring\PaymentGatewayModules\Modules\GoCardless\DataTransferObjects\GoCardlessSubscription;

/**
 * @since 2.14.0
 */
class SyncGoCardlessSubscription
{
    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function __invoke(Subscription $subscription): SubscriptionSynced
    {
        /**
         * Step #1 - DETAILS: check if the subscription is up-to-date with the gateway
         */
        $gatewaySubscription = $this->getGoCardlessSubscriptionDetails($subscription);
        $this->updateSubscriptionStatusFromGateway($subscription, $gatewaySubscription);
        $this->updateSubscriptionPeriodFromGateway($subscription, $gatewaySubscription);

        /**
         * Step #2 - TRANSACTIONS: check the transaction list for the subscription on the gateway side and create the
         * missing transactions (as renewal donations) on our side, then store them in an array and also the already
         * present transactions on our side in another array
         */
        $gatewayTransactions = $this->getGoCardlessSubscriptionTransactions($subscription);
        $missingDonations = $this->createMissingDonations($subscription, $gatewayTransactions);
        $presentDonations = $subscription->donations;

        /**
         * Step #3 - When this command gets handled by our API, it will return a JSON response to be used on the UI
         * with the subscription details and the missing/present transactions created in the previous step.
         */
        return new SubscriptionSynced(
            $subscription, // do not save the subscription, so our API can see what's dirty
            $missingDonations, // array<Donation> of the added missing donations
            $presentDonations, // array<Donation> of the already present donations
            __('GoCardless subscriptions can be synchronized as far back as available invoice history allows.', 'give-gocardless')
        );
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    protected function getGoCardlessSubscriptionDetails(Subscription $subscription): GoCardlessSubscription
    {
        return give(GetGoCardlessSubscriptionDetails::class)($subscription);
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    protected function getGoCardlessSubscriptionTransactions(Subscription $subscription): array
    {
        return give(GetGoCardlessSubscriptionTransactions::class)($subscription);
    }

    /**
     * @since 2.14.0
     */
    private function updateSubscriptionStatusFromGateway(Subscription $subscription, GoCardlessSubscription $gatewaySubscription): void
    {
        switch ($gatewaySubscription->status) {
            case 'active':
                $subscription->status = SubscriptionStatus::ACTIVE();
                break;
            case 'cancelled':
                $subscription->status = SubscriptionStatus::CANCELLED();
                break;
            case 'finished':
                $subscription->status = SubscriptionStatus::COMPLETED();
                break;
            case 'paused':
                $subscription->status = SubscriptionStatus::SUSPENDED();
                break;
            default:
                // Keep current status if unknown
                break;
        }
    }

    /**
     * @since 2.14.0
     */
    private function updateSubscriptionPeriodFromGateway(Subscription $subscription, GoCardlessSubscription $gatewaySubscription): void
    {
        switch ($gatewaySubscription->intervalUnit) {
            case 'weekly':
                $subscription->period = SubscriptionPeriod::WEEK();
                break;
            case 'monthly':
                $subscription->period = SubscriptionPeriod::MONTH();
                break;
            case 'yearly':
                $subscription->period = SubscriptionPeriod::YEAR();
                break;
            default:
                // Keep current period if unknown
                break;
        }

        $subscription->frequency = $gatewaySubscription->interval;
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    private function createMissingDonations(Subscription $subscription, array $gatewayTransactions): array
    {
        $missingDonations = [];
        $existingTransactionIds = [];

        // Get existing transaction IDs from current donations
        foreach ($subscription->donations as $donation) {
            if (!empty($donation->gatewayTransactionId)) {
                $existingTransactionIds[] = $donation->gatewayTransactionId;
            }
        }

        // Create missing donations for transactions that don't exist locally
        foreach ($gatewayTransactions as $transactionId => $transaction) {
            if (!in_array($transactionId, $existingTransactionIds)) {
                $missingDonations[] = $this->createRenewalDonation(
                    $subscription,
                    $transactionId,
                    $transaction['amount'],
                    $transaction['date']
                );
            }
        }

        return $missingDonations;
    }

    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    private function createRenewalDonation(Subscription $subscription, string $transactionId, float $amount, int $date): Donation
    {
        $renewalDonation = $subscription->createRenewal([
            'amount' => Money::fromDecimal($amount, $subscription->amount->getCurrency()),
            'gatewayTransactionId' => $transactionId,
            'createdAt' => new DateTime('@' . $date),
        ]);

        return $renewalDonation;
    }
}
