<?php

namespace GiveRecurring\PaymentGatewayModules\Modules\GoCardless\Actions;

use Exception;
use Give\Framework\PaymentGateways\Exceptions\PaymentGatewayException;
use Give\Framework\PaymentGateways\Log\PaymentGatewayLog;
use Give\Subscriptions\Models\Subscription;

/**
 * @since 2.14.0
 */
class CancelGoCardlessSubscription
{
    /**
     * @since 2.14.0
     *
     * @throws Exception
     */
    public function __invoke(Subscription $subscription): bool
    {
        if (empty($subscription->gatewaySubscriptionId)) {
            throw new PaymentGatewayException(
                sprintf(
                    __('Unable to cancel subscription #%s. Subscription does not have gatewaySubscriptionId.', 'give-gocardless'),
                    $subscription->id
                )
            );
        }

        $response = \Give_GoCardless_API::cancel_subscription($subscription->gatewaySubscriptionId);

        if (is_wp_error($response) || isset($response['error'])) {
            throw new PaymentGatewayException(sprintf(
                __('Error to cancel subscription #%s: %s - %s', 'give-gocardless'),
                $subscription->gatewaySubscriptionId,
                isset($response['error']['code']) ? $response['error']['code'] : $response->get_error_code(),
                isset($response['error']['message']) ? $response['error']['message'] : $response->get_error_message()
            ));
        }

        $payments = $this->getPaymentsFromSubscription($subscription->gatewaySubscriptionId);

        foreach ($payments as $payment) {

            if ('pending_submission' === $payment['status']) {

                $cancelPayment = \Give_GoCardless_API::cancel_payment($payment['id']);

                if (is_wp_error($cancelPayment)) {
                    PaymentGatewayLog::error(
                        sprintf(__('[GoCardless] Failed to cancel pending payment %s on GoCardless side for subscription %s.', 'give-gocardless'), $payment['id'], $subscription->id),
                        [
                            'Payment Gateway' => $subscription->gatewayId,
                            'Subscription' => $subscription->id,
                            'Gateway Subscription Id' => $subscription->gatewaySubscriptionId,
                            'GoCardless Payment Id' => $payment['id'],
                            'errorCode' => $cancelPayment->get_error_code(),
                            'errorMessage' => $cancelPayment->get_error_message(),
                        ]
                    );
                }
            }
        }

        return true;
    }

    /**
     * @since 2.14.0
     */
    private function getPaymentsFromSubscription(string $subscriptionId): array
    {
        $subscriptionArgs = [
            'subscription' => $subscriptionId,
        ];

        $payments = \Give_GoCardless_API::get_payments($subscriptionArgs);

        if (is_wp_error($payments) || empty($payments['payments'])) {
            return [];
        }

        return $payments['payments'];
    }
}
