<?php

namespace GiveRecurring\DonorDashboard\Routes;

use Exception;
use Give\DonorDashboards\Tabs\Contracts\Route as RouteAbstract;
use Give\Framework\PaymentGateways\PaymentGatewayRegister;
use Give\Subscriptions\Models\Subscription;
use Give_Subscription as LegacySubscription;
use GiveRecurring\Infrastructure\Log;
use WP_REST_Request;

/**
 * @since 2.10.0
 */
class ManagePausingSubscriptionRoute extends RouteAbstract
{

    /**
     * @since 2.10.0
     */
    public function endpoint(): string
    {
        return 'recurring-donations/subscription/manage-pausing';
    }

    /**
     * @since 2.10.0
     */
    public function args(): array
    {
        return [
            'id' => [
                'type' => 'int',
                'required' => true,
            ],
            'action' => [
                'type' => 'string',
                'required' => true,
                'enum' => ['pause', 'resume'],
            ],
            'interval_in_months' => [
                'type' => 'int',
                'required' => false,
            ],
        ];
    }

    /**
     * Should handle subscription pausing request.
     *
     * @since 2.10.0
     * @throws Exception
     */
    public function handleRequest(WP_REST_Request $request)
    {
        $legacySubscription = new LegacySubscription($request->get_param('id'));

        if (! $legacySubscription->can_pause()) {
            return;
        }

        try {
            $subscriptionId = (int)$request->get_param('id');

            /** @var Subscription $subscription */
            $subscription = Subscription::find($subscriptionId);

            $gatewayId = $subscription->gatewayId;

            /** @var PaymentGatewayRegister $paymentGatewayRegister */
            $paymentGatewayRegister = give(PaymentGatewayRegister::class);

            $gateway = $subscription->gateway();

            if (
                $paymentGatewayRegister->hasPaymentGateway($gatewayId) &&
                $gateway->supportsSubscriptions() &&
                $gateway->canPauseSubscription()
            ) {
                if ($request->get_param('action') === 'pause') {
                    $gateway->pauseSubscription($subscription, $request->get_params());
                } elseif ($request->get_param('action') === 'resume') {
                    $gateway->resumeSubscription($subscription);
                }
            }
        } catch (Exception $exception) {
            Log::error(
                sprintf(
                    'Failed to %1$s subscription %2$s with gateway %3$s',
                    $request->get_param('action'),
                    $legacySubscription->id,
                    $legacySubscription->gateway
                ),
                ['Exception' => $exception->getMessage()]
            );

            throw $exception;
        }
    }
}
