import React from 'react';

import {Button, Modal, TabPanel} from '@wordpress/components';
import {useState} from '@wordpress/element';
import {__} from '@wordpress/i18n';

import GatewaySettings from './gateway-settings';
import useGatewayDefaultSettingsEffect from './use-gateway-default-settings-effect';

type Gateway = {
    id: string;
    label: string;
    enabled: boolean;
}

declare const window: {
    feeRecoveryBlockSettings: {
        gateways: Gateway[];
    };
} & Window;

export default function PerGatewaySettings({attributes: {perGatewaySettings}, setAttributes}) {
    const {gateways} = window.feeRecoveryBlockSettings;
    const [isOpen, setOpen] = useState<boolean>(false);

    useGatewayDefaultSettingsEffect(gateways, perGatewaySettings, setAttributes);

    const gatewaysAsTabs = gateways.map((gateway) => {
        const disabled = perGatewaySettings.hasOwnProperty(gateway.id) && !perGatewaySettings[gateway.id].enabled;

        return {
            name: gateway.id,
            title: gateway.label,
            className: disabled ? 'is-disabled' : '',
        };
    });

    const openModal = () => setOpen(true);
    const closeModal = () => setOpen(false);

    const CloseButton = ({label, onClick}) => {
        return (
            <div className={'closeButton'}>
                <Button variant={'primary'} onClick={onClick}>
                    {label}
                </Button>
            </div>
        );
    };

    const makeUpdateSetting = (tab, settings) => (key, value) => {
        setAttributes({
            perGatewaySettings: {
                ...perGatewaySettings,
                [tab.name]: {
                    ...settings,
                    [key]: value,
                },
            },
        });
    };

    const tabContent = (tab) => {
        const settings = perGatewaySettings[tab.name];

        return (
            <GatewaySettings
                {...{
                    settings,
                    updateSetting: makeUpdateSetting(tab, settings),
                }}
            />
        );
    };

    return (
        <>
            <Button variant={'secondary'} onClick={openModal} style={{width: '100%', justifyContent: 'center'}}>
                {__('Customize fee per gateway', 'give-fee-recovery')}
            </Button>
            {isOpen && (
                <Modal
                    className={'feeRecovery-modal'}
                    title={__('Customize Fee Per Gateway', 'give-fee-recovery')}
                    onRequestClose={closeModal}
                    isDismissible={true}
                    shouldCloseOnClickOutside={false}
                >
                    <CloseButton label={__('Set and close', 'givewp')} onClick={closeModal} />

                    <div
                        style={
                            {'--disabled-tab-badge': `'${__('Disabled', 'give-fee-recovery')}'`} as React.CSSProperties
                        }
                    >
                        <TabPanel className={'tabPanel'} orientation={'vertical'} tabs={gatewaysAsTabs}>
                            {tabContent}
                        </TabPanel>
                    </div>
                </Modal>
            )}
        </>
    );
}
