import {InspectorControls} from '@wordpress/block-editor';
import {BaseControl, PanelBody, SelectControl, TextareaControl, ToggleControl} from '@wordpress/components';
import {__} from '@wordpress/i18n';

import PerGatewaySettings from './per-gateway-settings';
import FeeInputs from './fee-inputs';
import type {BlockEditProps} from '@wordpress/blocks';

type BlockInspectorControlsAttributes = {
    useGlobalSettings: boolean;
    feeSupportForAllGateways: boolean;
    feePercentage: number;
    feeBaseAmount: number;
    maxFeeAmount: number;
    includeInDonationSummary: boolean;
    donorOptIn: boolean;
    feeCheckboxLabel: string;
    feeMessage: string;
    perGatewaySettings: object;
    defaultDonorOptIn: boolean;
};

export default function BlockInspectorControls({
                                                   attributes,
                                                   setAttributes
                                               }: BlockEditProps<BlockInspectorControlsAttributes>) {
    const {
        useGlobalSettings,
        feeSupportForAllGateways,
        feePercentage,
        feeBaseAmount,
        maxFeeAmount,
        includeInDonationSummary,
        donorOptIn,
        feeCheckboxLabel,
        feeMessage,
        defaultDonorOptIn
    } = attributes;

    const updateSetting = (key, value) => {
        setAttributes({ [key]: value });
    };

    return (
        <InspectorControls>
            <PanelBody title={__('Field Settings', 'give-fee-recovery')} initialOpen={true}>
                <SelectControl
                    label={__('Fee Recovery Options', 'give-fee-recovery')}
                    options={[
                        { label: __('Global', 'give-fee-recovery'), value: 'true' },
                        { label: __('Customize', 'give-fee-recovery'), value: 'false' }
                    ]}
                    value={useGlobalSettings ? 'true' : 'false'}
                    onChange={() => updateSetting('useGlobalSettings', !useGlobalSettings)}
                />
                {!useGlobalSettings && (
                    <>
                        <ToggleControl
                            label={__('Fee support for all gateways', 'give-fee-recovery')}
                            checked={feeSupportForAllGateways}
                            onChange={() => setAttributes({ feeSupportForAllGateways: !feeSupportForAllGateways })}
                            help={__(
                                'Disable this option if you want to configure the fees per gateway',
                                'give-fee-recovery'
                            )}
                        />
                        {feeSupportForAllGateways ? (
                            <FeeInputs
                                feePercentage={feePercentage}
                                feeBaseAmount={feeBaseAmount}
                                maxFeeAmount={maxFeeAmount}
                                updateSetting={updateSetting}
                            />
                        ) : (
                            <BaseControl id="per-gateway-settings">
                                <PerGatewaySettings attributes={attributes} setAttributes={setAttributes} />
                            </BaseControl>
                        )}
                        <ToggleControl
                            label={__('Include in Donation Summary', 'give-fee-recovery')}
                            checked={includeInDonationSummary}
                            onChange={() => setAttributes({ includeInDonationSummary: !includeInDonationSummary })}
                            help={__(
                                'If enabled, and the Donation Summary block is used, then the fee will be included in the summary list.',
                                'give-fee-recovery'
                            )}
                        />
                        <ToggleControl
                            label={__('Donor opt-in', 'give-fee-recovery')}
                            checked={donorOptIn}
                            onChange={() => setAttributes({ donorOptIn: !donorOptIn })}
                            help={__(
                                'This allows donors to opt-in to cover the fees. If disabled, covering the fees is set by default.',
                                'give-fee-recovery'
                            )}
                        />
                        {donorOptIn ? (
                            <>
                                <ToggleControl
                                    label={__('Enable donor opt-in by default', 'give-fee-recovery')}
                                    checked={defaultDonorOptIn}
                                    onChange={() => setAttributes({ defaultDonorOptIn: !defaultDonorOptIn })}
                                    help={__(
                                        'If enabled, donors will be opted-in to recover fees by default. They can still opt-out if they choose.',
                                        'give-fee-recovery'
                                    )}
                                />
                                <TextareaControl
                                    label={__('Fee Message', 'give-fee-recovery')}
                                    value={feeCheckboxLabel}
                                    onChange={(feeCheckboxLabel) => setAttributes({ feeCheckboxLabel })}
                                    help={__(
                                        'This is the message the donor sees next to a checkbox indicating that they choose to donate the credit card fees.',
                                        'give-fee-recovery'
                                    )}
                                />
                            </>
                        ) : (
                            <TextareaControl
                                label={__('Fee Message', 'give-fee-recovery')}
                                value={feeMessage}
                                onChange={(feeMessage) => setAttributes({ feeMessage })}
                                help={__(
                                    'This is the message the donor sees next to a checkbox indicating that they choose to donate the credit card fees.',
                                    'give-fee-recovery'
                                )}
                            />
                        )}
                    </>
                )}
            </PanelBody>
        </InspectorControls>
    );
}
