import {useEffect, useMemo, useState} from '@wordpress/element';
import classNames from 'classnames';
import {FC} from 'react';
import './styles.scss';
import {__} from '@wordpress/i18n';
import type {GiveWP} from './types/window';

declare const window: GiveWP & Window;

const {checkbox: Checkbox} = window.givewp.form.templates.fields;
const {paragraph: Paragraph} = window.givewp.form.templates.elements;

export type Attributes = {
    donorOptIn: boolean;
    feeBaseAmount: number;
    feeCheckboxLabel: string;
    feeMessage: string;
    feePercentage: number;
    feeSupportForAllGateways: boolean;
    includeInDonationSummary: boolean;
    maxFeeAmount: number;
    perGatewaySettings: Record<string, any>;
    defaultDonorOptIn: boolean;
};

export interface PropTypes extends Attributes {
    inputProps: Record<string, any>;
    ErrorMessage: FC<{}>;
    fieldError: string | null;
}

/**
 * @since 2.0
 */
const addFeeToMessage = (label, amount) => label.replace('{fee_amount}', amount);

/**
 * @since 2.3.1 updated strings for translation support
 * @since 2.3.0 updated to use defaultDonorOptIn
 * @since 2.1.2 fix calculation of fee amount
 * @since 2.1.0 falsely amount values now calculate as 0
 * @since 2.0
 */
export default function FeeRecoveryField({
    ErrorMessage,
    fieldError,
    inputProps,
    donorOptIn,
    feeCheckboxLabel,
    feeMessage,
    feeSupportForAllGateways,
    perGatewaySettings,
    includeInDonationSummary,
    feePercentage,
    feeBaseAmount,
    maxFeeAmount,
    defaultDonorOptIn,
}: PropTypes) {
    let feeRecoveryEnabled: boolean = true;

    const {useWatch, useFormContext, useCurrencyFormatter, useDonationSummary} = window.givewp.form.hooks;
    const {setValue} = useFormContext();
    const [feeAmountOptIn, setFeeAmountOptIn] = useState<boolean>(defaultDonorOptIn);
    const donationSummary = useDonationSummary();

    const gatewayId: string = useWatch({name: 'gatewayId'});
    const amount: number = Number(useWatch({name: 'amount'}));
    const currency: string = useWatch({name: 'currency'});
    const currencyFormatter: Intl.NumberFormat = useCurrencyFormatter(currency);
    const feeRecovery = useWatch({name: 'feeRecovery'});

    // if feeSupportForAllGateways is false, then check per gateway settings based on the current gateway
    if (!feeSupportForAllGateways) {
        if (!perGatewaySettings[gatewayId]?.enabled) {
            feeRecoveryEnabled = false;
        } else {
            feePercentage = perGatewaySettings[gatewayId].feePercentage;
            feeBaseAmount = perGatewaySettings[gatewayId].feeBaseAmount;
            maxFeeAmount = perGatewaySettings[gatewayId].maxFeeAmount;
        }
    }

    const feeAmountRecovered: number = useMemo<number>(() => {
        if (!amount) {
            return 0;
        }

        const recoveredAmount = (amount + feeBaseAmount) / (1 - feePercentage / 100) - amount;
        const maxAmount = maxFeeAmount;

        if (maxAmount > 0 && recoveredAmount > maxAmount) {
            return maxAmount;
        }

        return recoveredAmount;
    }, [amount, feePercentage, feeBaseAmount, maxFeeAmount]);

    const label = useMemo<string>(
        () =>
            addFeeToMessage(!donorOptIn ? feeMessage : feeCheckboxLabel, currencyFormatter.format(feeAmountRecovered)),
        [feeAmountRecovered]
    );

    useEffect(() => {
        if ((feeAmountOptIn || !donorOptIn) && feeRecoveryEnabled) {
            setValue('feeRecovery', feeAmountRecovered);

            if (includeInDonationSummary) {
                donationSummary.addItem({
                    id: 'feeRecovery',
                    label: __('Cover Donation Fees', 'give-fee-recovery'),
                    value: currencyFormatter.format(feeAmountRecovered),
                    description: (
                        <div style={{display: 'flex', gap: '0.5rem', alignItems: 'center'}}>
                            <span>{__('Ensures 100% of your donation reaches our cause', 'give-fee-recovery')}</span>
                        </div>
                    ),
                });
            }

            donationSummary.addToTotal('feeRecovery', feeAmountRecovered);
        } else {
            setValue('feeRecovery', 0);

            donationSummary.removeItem('feeRecovery');

            donationSummary.removeFromTotal('feeRecovery');
        }
    }, [amount, feeAmountOptIn, donorOptIn, gatewayId, feeRecoveryEnabled, feeAmountRecovered, feeRecovery]);

    return (
        feeRecoveryEnabled && (
            <div className={classNames('feeRecovery-container', {invalid: fieldError})}>
                <input type="hidden" {...inputProps} />

                {donorOptIn ? (
                    <Checkbox
                        Label={() => label}
                        ErrorMessage={ErrorMessage}
                        inputProps={{
                            checked: feeAmountOptIn,
                            onChange: () => setFeeAmountOptIn(!feeAmountOptIn),
                        }}
                    />
                ) : (
                    <Paragraph content={label} />
                )}
            </div>
        )
    );
}
