<?php
namespace GiveFeeRecovery\FormExtension\DonationForm\Rules;

use Closure;
use Give\Framework\FieldsAPI\Exceptions\EmptyNameException;
use Give\Framework\Support\ValueObjects\Money;
use Give\Vendors\StellarWP\Validation\Contracts\ValidatesOnFrontEnd;
use Give\Vendors\StellarWP\Validation\Contracts\ValidationRule;
use GiveFeeRecovery\FormExtension\DonationForm\Fields\FeeRecovery;

class FeeRecoveryRule implements ValidationRule, ValidatesOnFrontEnd
{
    protected $field;

    /**
     * @inheritDoc
     *
     * @since 2.0
     */
    public static function id(): string
    {
        return 'feeRecovery';
    }

    /**
     * @inheritDoc
     *
     * @since 2.0
     */
    public static function fromString(string $options = null): ValidationRule
    {
        return new self(null);
    }

    /**
     * @inheritDoc
     *
     * @since 2.0
     */
    public function serializeOption()
    {
        return null;
    }

    public function __construct(?FeeRecovery $field = null)
    {
        $this->field = $field;
    }

    /**
     * @inheritDoc
     *
     * @since 2.3.6 allow donation to proceed while respecting max fee limit
     * @since 2.0
     * @throws EmptyNameException
     */
    public function __invoke($value, Closure $fail, string $key, array $values)
    {
        // If no field is provided, skip validation (this can happen when created via fromString)
        if (!$this->field) {
            return;
        }

        $currency = $values['currency'];
        $gatewayId = $values['gatewayId'];
        $amount = $values['amount'];

        $donorOptIn = (bool)$this->field->getDonorOptIn();

        if ($donorOptIn && !$value) {
            return;
        }

        $value = Money::fromDecimal($value, $currency);

        $feePercentage = (float)$this->field->getFeePercentage();
        $feeBaseAmount = Money::fromDecimal($this->field->getFeeBaseAmount(), $currency);
        $maxFeeAmount = Money::fromDecimal($this->field->getMaxFeeAmount() ?? 0, $currency);

        if (!$this->field->getFeeSupportForAllGateways()) {
            $perGatewaySettings = $this->field->getPerGatewaySettings()[$gatewayId];

            if (!$perGatewaySettings || !$perGatewaySettings['enabled']) {
                $fail(__('Fee recovery is not set up to work with the selected payment method.', 'give-fee-recovery'));
            }

            $feePercentage = (float)$perGatewaySettings['feePercentage'];
            $feeBaseAmount = Money::fromDecimal($perGatewaySettings['feeBaseAmount'], $currency);
            $maxFeeAmount = Money::fromDecimal($perGatewaySettings['maxFeeAmount'], $currency);
        }

        $donationAmount = Money::fromDecimal($amount, $currency);
        $feeRecoveredAmount = $donationAmount
            ->add($feeBaseAmount)
            ->divide(1 - ($feePercentage / 100))
            ->subtract($donationAmount);

        // If there's a maximum fee amount set and the calculated fee exceeds it,
        // use the maximum fee amount as the expected amount to compare against the received value later
        if ($maxFeeAmount->getAmount() > 0 && $feeRecoveredAmount->greaterThan($maxFeeAmount)) {
            $feeRecoveredAmount = $maxFeeAmount;
        }

        if (!$value->equals($feeRecoveredAmount)) {
            $fail(sprintf(
                __('The fee recovery amount is not correct. Received: %s. Calculated: %s', 'give-fee-recovery'),
                $value->formatToLocale(),
                $feeRecoveredAmount->formatToLocale()
            ));
        }

    }
}
