<?php

namespace GiveFeeRecovery\FormExtension\DonationForm\Actions;

use Give\DonationForms\Repositories\DonationFormRepository;
use Give\Donations\Models\Donation;
use Give\Framework\Blocks\BlockModel;
use Give\Framework\FieldsAPI\Exceptions\EmptyNameException;
use Give\Subscriptions\Models\Subscription;
use GiveFeeRecovery\FormExtension\DonationForm\Fields\FeeRecovery;
use GiveFeeRecovery\FormExtension\DonationForm\Rules\FeeRecoveryRule;
use GiveFeeRecovery\FormExtension\Hooks\UpdateDonationWithFeeAmountRecovered;
use GiveFeeRecovery\FormExtension\Hooks\UpdateSubscriptionWithFeeAmountRecovered;
use GiveFeeRecovery\FormExtension\Repositories\FeeRecoverySettings;

class ConvertFeeRecoveryBlockToFieldsApi
{
    /**
     * @since 2.2.2 remove union return type for PHP 7.4 compatibility
     * @since 2.0
     *
     * @throws EmptyNameException
     */
    public function __invoke(BlockModel $block, int $formId): FeeRecovery
    {
        return FeeRecovery::make('feeRecovery')
            ->tap(function (FeeRecovery $feeRecoveryField) use ($block, $formId) {
                if ($block->getAttribute('useGlobalSettings')) {
                    $this->setGlobalAttributes($feeRecoveryField);
                } else {
                    $this->setPerFormAttributes($feeRecoveryField, $block);
                }

                if (!$feeRecoveryField->getFeeSupportForAllGateways()) {
                    $this->updatePerGatewaySettings($feeRecoveryField, $block, $formId);
                }

                $feeRecoveryField->rules('numeric', new FeeRecoveryRule($feeRecoveryField));

                $feeRecoveryField->scope(function (FeeRecovery $field, $value, Donation $donation, ?Subscription $subscription) {
                    if (empty($value)) {
                        return;
                    }

                    (new UpdateDonationWithFeeAmountRecovered())($value, $donation);

                    if ($subscription){
                        (new UpdateSubscriptionWithFeeAmountRecovered())($value, $subscription);
                    }
                });

                return $feeRecoveryField;
            });
    }

    /**
     * @since 2.2.1 update global settings to use settings repository with default values
     * @since 2.0
     */
    private function setGlobalAttributes(FeeRecovery $field): void
    {
        /** @var FeeRecoverySettings $settings */
        $settings = give(FeeRecoverySettings::class);

        $field
            ->useGlobalSettings(true)
            ->feeSupportForAllGateways($settings->getFeeSupportForAllGateways())
            ->feePercentage($settings->getFeePercentage())
            ->feeBaseAmount($settings->getFeeBaseAmount())
            ->maxFeeAmount($settings->getFeeMaxAmount())
            ->includeInDonationSummary($settings->getIncludeFeeBreakdown())
            ->donorOptIn($settings->getDonorOptIn())
            ->feeCheckboxLabel($settings->getFeeCheckboxLabel())
            ->feeMessage($settings->getFeeMessage());
    }

    /**
     * @since 2.3.0 updated to use defaultDonorOptIn
     * @since 2.0
     *
     * @return void
     */
    private function setPerFormAttributes(FeeRecovery $field, BlockModel $block)
    {
        $field
            ->useGlobalSettings(false)
            ->feeSupportForAllGateways((bool)$block->getAttribute('feeSupportForAllGateways'))
            ->feePercentage((float)$block->getAttribute('feePercentage'))
            ->feeBaseAmount((float)$block->getAttribute('feeBaseAmount'))
            ->maxFeeAmount((float)$block->getAttribute('maxFeeAmount'))
            ->includeInDonationSummary((bool)$block->getAttribute('includeInDonationSummary'))
            ->donorOptIn((bool)$block->getAttribute('donorOptIn'))
            ->feeCheckboxLabel($block->getAttribute('feeCheckboxLabel'))
            ->feeMessage($block->getAttribute('feeMessage'))
            ->defaultDonorOptIn((bool)$block->getAttribute('defaultDonorOptIn'));
    }

    /**
     * @since 2.1.2 update per gateway settings individually to enforce types
     * @since 2.0
     *
     * @return void
     */
    private function updatePerGatewaySettings(FeeRecovery $field, BlockModel $block, int $formId)
    {
        /** @var DonationFormRepository $repository */
        $repository = give(DonationFormRepository::class);

        $supportedGateways = $repository->getEnabledPaymentGateways($formId);
        $settings = $block->getAttribute('perGatewaySettings');

        foreach ($supportedGateways as $gatewayId => $gateway) {
            if ($field->getUseGlobalSettings()) {
                $this->updateGatewayWithGlobalSettings($field, $gatewayId);
            } elseif ($settings[$gatewayId]) {
                $this->updateGatewayWithBlockSettings($field, $gatewayId, $settings[$gatewayId]);
            } else {
                $this->updateGatewayWithDefaultSettings($field, $gatewayId);
            }

            $this->validateGatewaySettings($field, $gatewayId);
        }
    }

    /**
     * @since 2.1.2
     *
     * @return void
     */
    private function updateGatewayWithBlockSettings(FeeRecovery $field, string $gatewayId, array $gatewaySettings)
    {
        $field->perGatewaySettings($gatewayId, [
            'enabled' => $gatewaySettings['enabled'],
            'feePercentage' => (float)$gatewaySettings['feePercentage'],
            'feeBaseAmount' => (float)$gatewaySettings['feeBaseAmount'],
            'maxFeeAmount' => (float)$gatewaySettings['maxFeeAmount'],
        ]);
    }

    /**
     * @since 2.0
     *
     * @return void
     */
    private function updateGatewayWithGlobalSettings(FeeRecovery $field, string $gatewayId)
    {
        $field->perGatewaySettings($gatewayId, [
            'enabled' => give_get_option('give_fee_gateway_fee_enable_option_' . $gatewayId) === 'enabled',
            'feePercentage' => (float)give_get_option('give_fee_gateway_fee_percentage_' . $gatewayId),
            'feeBaseAmount' => (float)give_get_option('give_fee_gateway_fee_base_amount_' . $gatewayId),
            'maxFeeAmount' => (float)give_get_option('give_fee_gateway_fee_maximum_fee_amount_' . $gatewayId),
        ]);
    }

    /**
     * @since 2.0
     *
     * @return void
     */
    private function updateGatewayWithDefaultSettings(FeeRecovery $field, string $gatewayId)
    {
        $field->perGatewaySettings($gatewayId, [
            'enabled' => true,
            'feePercentage' => 2.9,
            'feeBaseAmount' => 0.3,
            'maxFeeAmount' => null
        ]);
    }

    /**
     * @since 2.0
     *
     * @return void
     */
    private function validateGatewaySettings(FeeRecovery $field, string $gatewayId): void
    {
        $gatewaySettings = $field->getPerGatewaySettings()[$gatewayId];

        if (!$gatewaySettings['enabled']) {
            return;
        }

        if ((!$gatewaySettings['feePercentage']) > 0) {
            $field->perGatewaySettings($gatewayId, [
                'enabled' => false,
                'feePercentage' => 0,
                'feeBaseAmount' => 0,
                'maxFeeAmount' => null
            ]);
        }
    }
}
