<?php

namespace GiveFeeRecovery\FormExtension\DonationForm\Actions;

use Exception;
use Give\DonationForms\Models\DonationForm as DonationFormModel;
use Give\Framework\FieldsAPI\DonationForm;
use Give\Framework\FieldsAPI\PaymentGateways;
use Give\Framework\FormDesigns\Registrars\FormDesignRegistrar;
use Give\Log\Log;
use GiveFeeRecovery\FormExtension\DonationForm\Fields\FeeRecovery;


/**
 * This adds the fee recovery confirmation field to the form schema in very specific situations
 * 1. The form is using a multiStep layout.
 * 2. Fee recovery is enabled and has per-gateway settings.
 * 3. The fee recovery field is not on the same step/section of the gateways field.
 *
 * This is necessary in multiStep enabled form layouts and per-gateway settings where the gateway settings alter the original fee amount.
 *
 * @since 2.3.1
 */
class AddFeeRecoveryConfirmation
{
    /**
     * @since 2.3.2 Check if the designId setting is set on the form before proceeding.
     * @since 2.3.1
     */
    public function __invoke(DonationForm $form, int $formId)
    {
        /** @var FeeRecovery $feeRecovery */
        $feeRecovery = $form->getNodeByName('feeRecovery');

        if (!$feeRecovery || empty($feeRecovery->getPerGatewaySettings())) {
            return;
        }

        $donationForm = DonationFormModel::find($formId);

        if (!$donationForm) {
            return;
        }

        /** @var FormDesignRegistrar $formDesignRegistrar */
        $formDesignRegistrar = give(FormDesignRegistrar::class);

        if (!$donationForm->settings->designId || !$formDesignRegistrar->hasDesign($donationForm->settings->designId)) {
            return;
        }

        $formDesign = $formDesignRegistrar->getDesign($donationForm->settings->designId);

        if (!$formDesign->isMultiStep()) {
            return;
        }

        // loop through all sections and fields to find the PaymentGateways field,
        // then check if the fee recovery field is in the same section
        // if not, insert the fee recovery field before the PaymentGateways field
        foreach ($form->all() as $section) {
            foreach ($section->all() as $field) {
                if (($field instanceof PaymentGateways) && !$section->getNodeByName($feeRecovery->getName())) {
                    try {
                        $section->insertBefore(
                            $field->getName(),
                            $feeRecovery->toFeeRecoveryConfirmation()
                        );
                    } catch (Exception $e) {
                        Log::error($e->getMessage());

                        return;
                    }
                }
            }
        }
    }
}
