/*
 * printThis v2.0.0 - Vanilla JavaScript Version
 * @desc Printing utility for vanilla JavaScript (converted from jQuery plugin)
 * @author Jason Day (original jQuery version)
 * @author Samuel Rouse (original jQuery version)
 * @author Converted to vanilla JS
 *
 * Resources (based on):
 * - jPrintArea: http://plugins.jquery.com/project/jPrintArea
 * - jqPrint: https://github.com/permanenttourist/jquery.jqprint
 * - Ben Nadel: http://www.bennadel.com/blog/1591-Ask-Ben-Print-Part-Of-A-Web-Page-With-jQuery.htm
 *
 * Licensed under the MIT licence:
 *              http://www.opensource.org/licenses/mit-license.php
 *
 * (c) Jason Day 2015-2022
 *
 * Usage:
 *
 *  printThis(document.querySelector("#mySelector"), {
 *      debug: false,                   // show the iframe for debugging
 *      importCSS: true,                // import parent page css
 *      importStyle: true,              // import style tags
 *      printContainer: true,           // grab outer container as well as the contents of the selector
 *      loadCSS: "path/to/my.css",      // path to additional css file - use an array [] for multiple
 *      pageTitle: "",                  // add title to print page
 *      removeInline: false,            // remove all inline styles from print elements
 *      removeInlineSelector: "body *", // custom selectors to filter inline styles. removeInline must be true
 *      printDelay: 1000,               // variable print delay
 *      header: null,                   // prefix to html
 *      footer: null,                   // postfix to html
 *      base: false,                    // preserve the BASE tag, or accept a string for the URL
 *      formValues: true,               // preserve input/form values
 *      canvas: true,                   // copy canvas elements
 *      doctypeString: '<!DOCTYPE html>', // enter a different doctype for older markup
 *      removeScripts: false,           // remove script tags from print content
 *      copyTagClasses: true,           // copy classes from the html & body tag
 *      copyTagStyles: true,            // copy styles from html & body tag (for CSS Variables)
 *      beforePrintEvent: null,         // callback function for printEvent in iframe
 *      beforePrint: null,              // function called before iframe is filled
 *      afterPrint: null                // function called before iframe is removed
 *  });
 *
 * Notes:
 *  - the loadCSS will load additional CSS (with or without @media print) into the iframe, adjusting layout
 */

(function(window) {
    'use strict';

    // Default options
    const defaults = {
        debug: false,
        importCSS: true,
        importStyle: true,
        printContainer: true,
        loadCSS: "",
        pageTitle: "",
        removeInline: false,
        removeInlineSelector: "*",
        printDelay: 1000,
        header: null,
        footer: null,
        base: false,
        formValues: true,
        canvas: true,
        doctypeString: '<!DOCTYPE html>',
        removeScripts: false,
        copyTagClasses: true,
        copyTagStyles: true,
        beforePrintEvent: null,
        beforePrint: null,
        afterPrint: null
    };

    // Utility functions
    function extend(target, ...sources) {
        sources.forEach(source => {
            if (source) {
                Object.keys(source).forEach(key => {
                    target[key] = source[key];
                });
            }
        });
        return target;
    }

    function isArray(obj) {
        return Array.isArray(obj);
    }

    function appendContent(container, content) {
        if (!content) return;

        if (typeof content === 'string') {
            container.insertAdjacentHTML('beforeend', content);
        } else if (content.nodeType) {
            container.appendChild(content.cloneNode(true));
        }
    }

    function cloneElement(element, deep = true) {
        return element.cloneNode(deep);
    }

    function copyValues(origin, clone, elementSelector) {
        const originalElements = origin.querySelectorAll(elementSelector);
        const clonedElements = clone.querySelectorAll(elementSelector);

        clonedElements.forEach((item, index) => {
            if (originalElements[index]) {
                item.value = originalElements[index].value;
            }
        });
    }

    function appendBody(body, element, opt) {
        // Clone the element
        let content = cloneElement(element, true);

        if (opt.formValues) {
            // Copy original select and textarea values to their cloned counterpart
            copyValues(element, content, 'select, textarea');
        }

        if (opt.removeScripts) {
            const scripts = content.querySelectorAll('script');
            scripts.forEach(script => script.remove());
        }

        if (opt.printContainer) {
            // Grab element as container
            body.appendChild(content);
        } else {
            // Otherwise just print interior elements of container
            while (content.firstChild) {
                body.appendChild(content.firstChild);
            }
        }
    }

    function setDocType(iframe, doctype) {
        const win = iframe.contentWindow || iframe.contentDocument || iframe;
        const doc = win.document || win.contentDocument || win;
        doc.open();
        doc.write(doctype);
        doc.close();
    }

    function attachOnBeforePrintEvent(iframe, beforePrintHandler) {
        const win = iframe.contentWindow || iframe.contentDocument || iframe;

        if (typeof beforePrintHandler === "function") {
            if ('matchMedia' in win) {
                win.matchMedia('print').addListener(function(mql) {
                    if (mql.matches) beforePrintHandler();
                });
            } else {
                win.onbeforeprint = beforePrintHandler;
            }
        }
    }

    // Main printThis function
    function printThis(element, options = {}) {
        const opt = extend({}, defaults, options);

        // Handle multiple elements or single element
        const elements = element.nodeType ? [element] : Array.from(element);

        elements.forEach(el => {
            processPrint(el, opt);
        });
    }

    function processPrint(element, opt) {
        const strFrameName = "printThis-" + (new Date()).getTime();
        let iframe;

        if (window.location.hostname !== document.domain && navigator.userAgent.match(/msie/i)) {
            // Ugly IE hacks due to IE not inheriting document.domain from parent
            const iframeSrc = "javascript:document.write(\"<head><script>document.domain=\\\"" + document.domain + "\\\";</s" + "cript></head><body></body>\")";
            iframe = document.createElement('iframe');
            iframe.name = "printIframe";
            iframe.id = strFrameName;
            iframe.className = "MSIE";
            document.body.appendChild(iframe);
            iframe.src = iframeSrc;
        } else {
            // Other browsers inherit document.domain
            iframe = document.createElement('iframe');
            iframe.id = strFrameName;
            iframe.name = 'printIframe';
            document.body.appendChild(iframe);
        }

        // Show frame if in debug mode
        if (!opt.debug) {
            iframe.style.position = "absolute";
            iframe.style.width = "0px";
            iframe.style.height = "0px";
            iframe.style.left = "-600px";
            iframe.style.top = "-600px";
        }

        // Before print callback
        if (typeof opt.beforePrint === "function") {
            opt.beforePrint();
        }

        setTimeout(() => {
            // Add doctype to fix the style difference between printing and render
            if (opt.doctypeString) {
                setDocType(iframe, opt.doctypeString);
            }

            const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
            const head = iframeDoc.querySelector("head") || iframeDoc.createElement("head");
            const body = iframeDoc.querySelector("body") || iframeDoc.createElement("body");

            if (!iframeDoc.querySelector("head")) {
                iframeDoc.documentElement.appendChild(head);
            }
            if (!iframeDoc.querySelector("body")) {
                iframeDoc.documentElement.appendChild(body);
            }

            const baseElements = document.querySelectorAll('base');
            let baseURL;

            // Add base tag to ensure elements use the parent domain
            if (opt.base === true && baseElements.length > 0) {
                baseURL = baseElements[0].getAttribute('href');
            } else if (typeof opt.base === 'string') {
                baseURL = opt.base;
            } else {
                baseURL = document.location.protocol + '//' + document.location.host;
            }

            head.insertAdjacentHTML('beforeend', '<base href="' + baseURL + '">');

            // Import page stylesheets
            if (opt.importCSS) {
                const links = document.querySelectorAll("link[rel=stylesheet]");
                links.forEach(link => {
                    const href = link.getAttribute("href");
                    if (href) {
                        const media = link.getAttribute("media") || "all";
                        head.insertAdjacentHTML('beforeend',
                            "<link type='text/css' rel='stylesheet' href='" + href + "' media='" + media + "'>");
                    }
                });
            }

            // Import style tags
            if (opt.importStyle) {
                const styles = document.querySelectorAll("style");
                styles.forEach(style => {
                    head.appendChild(style.cloneNode(true));
                });
            }

            // Add title of the page
            if (opt.pageTitle) {
                head.insertAdjacentHTML('beforeend', "<title>" + opt.pageTitle + "</title>");
            }

            // Import additional stylesheet(s)
            if (opt.loadCSS) {
                if (isArray(opt.loadCSS)) {
                    opt.loadCSS.forEach(css => {
                        head.insertAdjacentHTML('beforeend', "<link type='text/css' rel='stylesheet' href='" + css + "'>");
                    });
                } else {
                    head.insertAdjacentHTML('beforeend', "<link type='text/css' rel='stylesheet' href='" + opt.loadCSS + "'>");
                }
            }

            const pageHtml = document.documentElement;
            const htmlElement = iframeDoc.documentElement;

            // CSS VAR in html tag when dynamic apply
            htmlElement.style.cssText = pageHtml.style.cssText;

            // Copy 'root' tag classes
            let tag = opt.copyTagClasses;
            if (tag) {
                tag = tag === true ? 'bh' : tag;
                if (tag.indexOf('b') !== -1) {
                    body.className = document.body.className;
                }
                if (tag.indexOf('h') !== -1) {
                    htmlElement.className = pageHtml.className;
                }
            }

            // Copy ':root' tag styles
            tag = opt.copyTagStyles;
            if (tag) {
                tag = tag === true ? 'bh' : tag;
                if (tag.indexOf('b') !== -1) {
                    body.style.cssText = document.body.style.cssText;
                }
                if (tag.indexOf('h') !== -1) {
                    htmlElement.style.cssText = pageHtml.style.cssText;
                }
            }

            // Print header
            appendContent(body, opt.header);

            if (opt.canvas) {
                // Add canvas data-ids for easy access after cloning
                let canvasId = 0;
                const canvases = element.querySelectorAll('canvas');

                // Check if the element itself is a canvas
                if (element.tagName === 'CANVAS') {
                    element.setAttribute('data-printthis', canvasId++);
                }

                canvases.forEach(canvas => {
                    canvas.setAttribute('data-printthis', canvasId++);
                });
            }

            appendBody(body, element, opt);

            if (opt.canvas) {
                // Re-draw new canvases by referencing the originals
                const iframeCanvases = body.querySelectorAll('canvas');
                iframeCanvases.forEach(canvas => {
                    const cid = canvas.getAttribute('data-printthis');
                    const srcCanvas = document.querySelector('[data-printthis="' + cid + '"]');

                    if (srcCanvas) {
                        const ctx = canvas.getContext('2d');
                        ctx.drawImage(srcCanvas, 0, 0);

                        // Remove the markup from the original
                        srcCanvas.removeAttribute('data-printthis');
                    }
                });
            }

            // Remove inline styles
            if (opt.removeInline) {
                const selector = opt.removeInlineSelector || '*';
                const elements = body.querySelectorAll(selector);
                elements.forEach(el => {
                    el.removeAttribute("style");
                });
            }

            // Print footer
            appendContent(body, opt.footer);

            // Attach event handler function to beforePrint event
            attachOnBeforePrintEvent(iframe, opt.beforePrintEvent);

            setTimeout(() => {
                if (iframe.className === "MSIE") {
                    // IE hack
                    window.frames["printIframe"].focus();
                    head.insertAdjacentHTML('beforeend', "<script>window.print();</script>");
                } else {
                    // Proper method
                    if (document.queryCommandSupported && document.queryCommandSupported("print")) {
                        iframe.contentWindow.document.execCommand("print", false, null);
                    } else {
                        iframe.contentWindow.focus();
                        iframe.contentWindow.print();
                    }
                }

                // Remove iframe after print
                if (!opt.debug) {
                    setTimeout(() => {
                        iframe.remove();
                    }, 1000);
                }

                // After print callback
                if (typeof opt.afterPrint === "function") {
                    opt.afterPrint();
                }

            }, opt.printDelay);

        }, 333);
    }

    // Expose printThis to global scope
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = printThis;
    } else {
        window.printThis = printThis;
    }

})(typeof window !== 'undefined' ? window : this);