<?php
/**
 * Grid Fixed Title layout class.
 *
 * @since ??
 *
 * @package Envira_Gallery
 * @author  Envira Gallery Team <support@enviragallery.com>
 */

namespace Envira\Frontend\Gallery_Markup\Layouts;

use Envira\Frontend\Gallery_Markup\Base;
use Envira\Frontend\Gallery_Markup\Layouts\Grid_Fixed_Title_Item;
use Envira\Utils\Shortcode_Utils;

/**
 * Grid Fixed Title layout class.
 */
class Grid_Fixed_Title extends Base {
	/**
	 * Generate wrapper classes
	 *
	 * @since 1.9.0
	 */
	protected function get_wrapper_classes() {
		$classes   = parent::get_wrapper_classes();
		$classes[] = 'envira-layout-grid-fixed-title';

		return $classes;
	}

	/**
	 * Get item config.
	 *
	 * @param array $config Item config.
	 *
	 * @return Grid_Fixed_Title_Item
	 */
	protected function get_item_object( $config ) {
		return new Grid_Fixed_Title_Item( $config );
	}

	/**
	 * Override markup to force description position to 'above' for Grid Fixed Title layout
	 *
	 * @since ??
	 * @return string
	 */
	public function markup() {
		// Store original description position
		$original_position = envira_get_config( 'description_position', $this->data );

		// Force description position to 'above' so it renders inside wrapper for CSS Grid
		$this->data['config']['description_position'] = 'above';

		// Filter to inject title
		add_filter( 'envira_gallery_output_before_description', [ $this, 'inject_title' ], 10, 2 );

		// Check if description exists before adding wrappers
		$description           = Shortcode_Utils::get_description( $this->data );
		$added_wrapper_filters = false;

		if ( ! empty( $description ) ) {
			// Filter to wrap description
			add_filter( 'envira_gallery_output_before_description', [ $this, 'open_description_wrapper' ], 11, 2 );
			add_filter( 'envira_gallery_output_after_description', [ $this, 'close_description_wrapper' ], 10, 2 );
			$added_wrapper_filters = true;
		}

		// Call parent markup
		$markup = parent::markup();

		// Remove filter
		remove_filter( 'envira_gallery_output_before_description', [ $this, 'inject_title' ] );

		if ( $added_wrapper_filters ) {
			remove_filter( 'envira_gallery_output_before_description', [ $this, 'open_description_wrapper' ] );
			remove_filter( 'envira_gallery_output_after_description', [ $this, 'close_description_wrapper' ] );
		}

		// Restore original position
		$this->data['config']['description_position'] = $original_position;

		return $markup;
	}

	/**
	 * Inject title into description area.
	 *
	 * @param string $output Output HTML.
	 * @param array  $data   Gallery data.
	 *
	 * @return string
	 */
	public function inject_title( $output, $data ) {
		// Attempt to get title from data (e.g. Gutenberg attribute) or get_the_title
		$title = '';

		// If passed from Gutenberg attributes (if saved in config)
		if ( ! empty( $data['config']['title'] ) ) {
			$title = $data['config']['title'];
		} elseif ( ! empty( $data['config']['gallery_title'] ) ) {
			$title = $data['config']['gallery_title'];
		} else {
			// Fallback to post title if gallery ID is available
			$gallery_id = isset( $data['id'] ) ? $data['id'] : ( isset( $this->gallery_id ) ? $this->gallery_id : false );
			if ( $gallery_id ) {
				// Remove fragment from ID if present
				$gallery_id = preg_replace( '/-[0-9]+$/', '', $gallery_id );
				$title      = get_the_title( $gallery_id );
			}
		}

		if ( $title ) {
			$output .= '<h1 class="envira-gallery-title">' . esc_html( $title ) . '</h1>';
		}

		return $output;
	}

	/**
	 * Open description content wrapper.
	 *
	 * @param string $output Output HTML.
	 * @param array  $data   Gallery data.
	 *
	 * @return string
	 */
	public function open_description_wrapper( $output, $data ) {
		return $output . '<div class="envira-gallery-description-content">';
	}

	/**
	 * Close description content wrapper.
	 *
	 * @param string $output Output HTML.
	 * @param array  $data   Gallery data.
	 *
	 * @return string
	 */
	public function close_description_wrapper( $output, $data ) {
		return $output . '</div>';
	}
}
