<?php
/**
 * Dynamic Gallery Shortcode class.
 *
 * @since 1.0.0
 *
 * @package Envira_Dynamic
 * @author  Envira Team
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Dynamic Gallery Shortcode class.
 *
 * @since 1.0.0
 *
 * @package Envira_Dynamic
 * @author  Envira Team
 */
class Envira_Dynamic_Gallery_Shortcode {

	/**
	 * Holds the class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public static $instance;

	/**
	 * Path to the file.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $file = __FILE__;

	/**
	 * Holds the base class object.
	 *
	 * @since 1.0.0
	 *
	 * @var object
	 */
	public $base;

	/**
	 * Primary class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// Register Gallery Dynamic Shortcode.
		add_shortcode( 'envira-gallery-dynamic', [ $this, 'shortcode' ] );
		add_shortcode( 'envira-gallery_dynamic', [ $this, 'shortcode' ] );
		add_filter( 'post_gallery', [ $this, 'override_gallery' ], 9999, 2 );
		add_filter( 'envira_gallery_custom_gallery_data', [ $this, 'envira_gallery_custom_gallery_data' ], 10, 3 );

		// Misc.
		add_filter( 'envira_gallery_output_width', [ $this, 'get_width_of_folder_image' ], 10, 6 );
		add_filter( 'envira_gallery_output_height', [ $this, 'get_height_of_folder_image' ], 10, 6 );
		add_filter( 'envira_gallery_should_cache', [ $this, 'prevent_cache' ], 10, 2 );
	}

	/**
	 * Attempt to generate a width for a folder based image (needed for things like automatic layout)
	 *
	 * @since 1.0.0
	 *
	 * @param int    $output_width Output Width.
	 * @param int    $id ID.
	 * @param array  $item Item.
	 * @param array  $data Gallery data.
	 * @param array  $i Index.
	 * @param string $output_src Output Src.
	 * @param string $size Size.
	 * @return int output_width
	 */
	public function get_width_of_folder_image( $output_width, $id, $item, $data, $i, $output_src, $size = 'medium' ) {

		if ( strpos( $data['id'], 'folder_' ) === false ) {
			return $output_width;
		}

		$file = basename( $output_src );
		$re   = '/\d+[Xx]\d+\./';
		preg_match( $re, $file, $matches );
		if ( $matches ) {
			$size         = str_replace( '.', '', $matches[0] );
			$sizes        = explode( 'x', $size );
			$output_width = $sizes[0];
		} else {
			$output_width = get_option( "{$size}_size_w" );
		}

		return $output_width;
	}

	/**
	 * Attempt to generate a height for a folder based image (needed for things like automatic layout)
	 *
	 * @since 1.0.0
	 *
	 * @param int    $output_height Output Height.
	 * @param int    $id ID.
	 * @param array  $item Item.
	 * @param array  $data Gallery data.
	 * @param array  $i Index.
	 * @param string $output_src Output Src.
	 * @param string $size Size.
	 * @return interger output_height
	 */
	public function get_height_of_folder_image( $output_height, $id, $item, $data, $i, $output_src, $size = 'medium' ) {

		if ( strpos( $data['id'], 'folder_' ) === false ) {
			return $output_height;
		}

		$file = basename( $output_src );
		$re   = '/\d+[Xx]\d+\./';
		preg_match( $re, $file, $matches );
		if ( $matches ) {
			$size          = str_replace( '.', '', $matches[0] );
			$sizes         = explode( 'x', $size );
			$output_height = $sizes[1];
		} else {
			$output_height = get_option( "{$size}_size_h" );
		}

		return $output_height;
	}

	/**
	 * Parses the Dynamic Gallery attributes and filters them into the data.
	 *
	 * @since 1.0.0
	 *
	 * @param bool   $data_found Boolean (false) since no data is found yet.
	 * @param array  $atts  Array of shortcode attributes to parse.
	 * @param object $post The current post object.
	 *
	 * @return array $data Array of dynamic gallery data.
	 */
	public function envira_gallery_custom_gallery_data( $data_found, $atts, $post ) {

		// If the dynamic attribute is not set to true, do nothing.
		if ( empty( $atts['dynamic'] ) ) {
			return $data_found;
		}

		$dynamic_id = false;

		// Now that we have a dynamic slider, prepare atts to be parsed with defaults.
		if ( isset( $atts['default'] ) ) {

			$defaults = get_post_meta( $atts['default'], '_eg_gallery_data', true );

			// reset and remove some of the stuff from the original gallery.
			$defaults['config']['type'] = 'dynamic';
			$defaults['id']             = '';
			$defaults['gallery']        = [];

		} else {

			$dynamic_id = Envira_Dynamic_Common::get_instance()->get_gallery_dynamic_id();
			$defaults   = get_post_meta( $dynamic_id, '_eg_gallery_data', true );

		}

		$data = [];
		foreach ( (array) $atts as $key => $value ) {
			// Cast any 'true' or 'false' atts to a boolean value.
			if ( 'true' === $value ) {
				$atts[ $key ] = 1;
				$value        = 1;
			}

			if ( 'false' === $value ) {
				$atts[ $key ] = 0;
				$value        = 0;
			}

			// Store data.
			$data[ $key ] = $value;
		}

		// If the data is empty, return false.
		if ( empty( $data ) || empty( $defaults ) ) {
			return false;
		}

		// Merge in the defaults into the data.
		$config = $defaults;

		if ( isset( $atts['custom_id'] ) ) {

			$config['id'] = $atts['custom_id'];

		} else {

			$config['id'] = str_replace( [ '-', ',', '/' ], '_', $atts['dynamic'] ); // Replace dashes and commas with underscores.

		}

		// Remove special characters from potential images args.
		if ( isset( $data['images'] ) ) {
			$data['images'] = urldecode( $data['images'] );
			$data['images'] = str_replace( ', ', ',', $data['images'] );
		}

		$config_array     = $defaults['config'];
		$parsed_array     = wp_parse_args( $data, $defaults['config'] );
		$config['config'] = $parsed_array;

		// Store the dynamic ID within the config
		// This allows Addons which support both Galleries and Albums grab the Dynamic ID to figure out
		// whether the Addon is looking at a Gallery or Album.
		$config['config']['id'] = $config['id'];

		// make sure we pass the settings proper.
		$config['dynamic_id'] = isset( $atts['default'] ) ? $atts['default'] : get_option( 'envira_dynamic_gallery' );
		// Inject images.
		$data = $this->inject_images( $config, false );

		// Parse the args and return the data.
		return apply_filters( 'envira_dynamic_gallery_parsed_data', $data, $defaults, $atts, $post );
	}

	/**
	 * Injects gallery images into the given $data array, using the $data settings (i.e. the dynamic gallery settings)
	 *
	 * @since 1.0.0
	 *
	 * @param array $data  Gallery Config.
	 * @param int   $id      The gallery ID.
	 * @return array $data Amended array of gallery config, with images.
	 */
	public function inject_images( $data, $id ) {

		// Return early if not a Dynamic gallery.
		if ( ! envira_get_config( 'dynamic', $data ) ) {
			return $data;
		}

		// $id should be false, so we need to set it now.
		if ( ! $id ) {
			$id = envira_get_config( 'dynamic', $data );
			if ( is_array( $id ) ) {
				$id = $id[0];
			}
		}

		/**
		* Get images based on supplied Dynamic settings
		* Checks for:
		* - Media Library Image IDs: [envira-gallery-dynamic id="custom-xxx" images="id,id,id"]
		* - NextGen Gallery ID: [envira-gallery-dynamic id="nextgen-id"]
		* - Folder: [envira-gallery-dynamic id="folder-foldername"]
		*/
		$dynamic_data = [];
		$rule_matched = false;

		$id_type = 0 === strpos( $id, 'custom-' ) ? 'custom' : null;
		$id_type = ! $id_type && 0 === strpos( $id, 'nextgen-' ) ? 'nextgen' : null;
		$id_type = ! $id_type && 0 === strpos( $id, 'folder-' ) ? 'folder' : null;

		switch ( $id_type ) {
			case 'custom':
				$dynamic_data = $this->get_custom_images( $dynamic_data, $id, $data );
				$rule_matched = true;
				break;
			case 'nextgen':
				$dynamic_data = $this->get_nextgen_images( $dynamic_data, $id, $data );
				$rule_matched = true;
				break;
			case 'folder':
				$dynamic_data = $this->get_folder_images( $dynamic_data, $id, $data );
				$rule_matched = true;
				break;
			default:
				// Backwards compatibility for custom preg_match filters.
				$extra_types = apply_filters( 'envira_dynamic_get_dynamic_gallery_types', [] );

				if ( is_array( $extra_types ) ) {
					foreach ( $extra_types as $filter_to_execute => $preg_match ) {
						if ( ! is_array( $id ) && preg_match( $preg_match, $id ) ) {
							// Run action for this preg_match.
							$rule_matched = true;
							$dynamic_data = apply_filters( $filter_to_execute, $dynamic_data, $id, $data );
							break;
						}
					}
				}
				break;
		}

		/**
		* Get images based on supplied Dynamic settings
		* Checks for:
		* - Post/Page ID: [envira-gallery-dynamic id="id" exclude="id,id,id"]
		*/
		if ( ! $rule_matched ) {
			$exclude      = ! empty( $data['config']['exclude'] ) ? $data['config']['exclude'] : false;
			$images       = $this->get_attached_images( $id, $exclude );
			$dynamic_data = $this->get_custom_images( $dynamic_data, $id, $data, implode( ',', (array) $images ) );
		}

		// Filter.
		$dynamic_data = apply_filters( 'envira_gallery_dynamic_queried_data', $dynamic_data, $id, $data );

		// Check image(s) were found.
		if ( ! is_array( $dynamic_data ) || count( $dynamic_data ) === 0 ) {
			// No images found, nothing to inject - just return data.
			return $data;
		}

		// Generate thumbnails.
		$dynamic_data = $this->maybe_generate_thumbnails( $data, $dynamic_data );
		$sort_order   = envira_get_config( 'sort_order', $data );

		// Insert images into gallery data.
		$data['gallery'] = $dynamic_data;

		if ( '0' !== $sort_order ) {

			$data = envira_sort_gallery( $data, $sort_order, envira_get_config( 'sorting_direction', $data ) );
		}

		// Return the modified data.
		return apply_filters( 'envira_gallery_dynamic_data', $data, $id );
	}

	/**
	 * Generates thumbnails for the Dynamic Gallery if they are enabled on the Dynamic Gallery Settings
	 *
	 * @since 1.0.2
	 *
	 * @param array $data Dynamic Gallery Data.
	 * @param array $dynamic_data Gallery Images.
	 * @return array Gallery Images w/ thumbnail attribute
	 */
	public function maybe_generate_thumbnails( $data, $dynamic_data ) {

		if ( ! $dynamic_data ) {
			return $dynamic_data;
		}

		// If the thumbnails option is checked for the Dynamic Gallery, generate thumbnails accordingly.
		if ( ! isset( $data['config']['thumbnails'] ) || ! $data['config']['thumbnails'] ) {
			return $dynamic_data;
		}

		$dimensions = envira_get_thumbnail_dimensions( $data );

		// Build args for image resizing.
		$args = [
			'position' => 'c',
			'width'    => $dimensions['width'],
			'height'   => $dimensions['height'],
			'quality'  => 100,
			'retina'   => false,
			'data'     => $data,
		];
		$args = apply_filters( 'envira_gallery_crop_image_args', $args );

		// Iterate through dynamically obtained images, creating thumbnails.
		foreach ( $dynamic_data as $id => $item ) {
			// Generate the cropped image.
			$cropped_image = envira_resize_image( $item['src'], $args['width'], $args['height'], true, $args['position'], $args['quality'], $args['retina'], $args['data'] );

			// If there is an error, possibly output error message, otherwise woot!
			if ( is_wp_error( $cropped_image ) ) {
				// If debugging is defined, print out the error.
				if ( defined( 'ENVIRA_GALLERY_CROP_DEBUG' ) && filter_var( ENVIRA_GALLERY_CROP_DEBUG, FILTER_VALIDATE_BOOLEAN ) ) {
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_var_export, WordPress.Security.EscapeOutput.OutputNotEscaped
					echo '<pre>' . var_export( $cropped_image->get_error_message(), true ) . '</pre>';
				}
			} else {
				$dynamic_data[ $id ]['thumb'] = $cropped_image;
			}
		}

		// Return.
		return $dynamic_data;
	}

	/**
	 * Retrieves the image data for custom image sets
	 *
	 * @param array      $dynamic_data    Existing Dynamic Data Array.
	 * @param int        $id                ID (either custom-ID or Page/Post ID).
	 * @param array      $data            Gallery Configuration.
	 * @param bool|array $images     Array of image IDs to use (optional).
	 * @return bool|array            Array of data on success, false on failure
	 */
	public function get_custom_images( $dynamic_data, $id, $data, $images = false ) {

		// Image IDs will be set in either:
		// - 1. $data['config']['images'] (by parse_shortcode_attributes())
		// - 2. $images array (passed to this function).
		$data_images = envira_get_config( 'images', $data );
		if ( ! $data_images ) {
			if ( ! $images ) {
				// No images specified matching (1) or (2) above - bail.
				return false;
			}
		} else {
			$images = $data_images;
		}

		// $images now reflects the exact images we want to include in the Gallery
		$images = explode( ',', (string) $images );
		foreach ( (array) $images as $i => $image_id ) {
			$image_id = trim( $image_id, '”' );
			// Get image attachment and check it exists.
			$attachment = get_post( $image_id );
			if ( ! $attachment ) {
				continue;
			}

			// Get image details.
			$src = wp_get_attachment_image_src( $image_id, 'full' );

			// Build image attributes to match Envira Gallery.
			$dynamic_data[ $image_id ] = [
				'status'          => 'published',
				'src'             => ( isset( $src[0] ) ? esc_url( $src[0] ) : '' ),
				'title'           => $attachment->post_title,
				'link'            => ( isset( $src[0] ) ? esc_url( $src[0] ) : '' ),
				'alt'             => get_post_meta( $image_id, '_wp_attachment_image_alt', true ),
				'caption'         => $attachment->post_excerpt,
				'thumb'           => '',
				'link_new_window' => 0,
			];

			if ( isset( $attachment->meta_data['full']['width'] ) ) {
				$dynamic_data[ $image_id ]['width'] = $attachment->meta_data['full']['width'];
			}
			if ( isset( $attachment->meta_data['full']['height'] ) ) {
				$dynamic_data[ $image_id ]['height'] = $attachment->meta_data['full']['height'];
			}
		}

		return apply_filters( 'envira_gallery_dynamic_custom_image_data', $dynamic_data, $id, $data );
	}

	/**
	 * Retrieves the image data for a given NextGen Gallery ID
	 *
	 * @param array $dynamic_data    Existing Dynamic Data Array.
	 * @param int   $id            NextGen Gallery ID.
	 * @param array $data        Gallery Configuration.
	 * @return bool|array        Array of data on success, false on failure
	 */
	public function get_nextgen_images( $dynamic_data, $id, $data ) {

		// Return false if the NextGen database class is not available.
		if ( ! class_exists( 'nggdb' ) ) {
			return false;
		}

		// Get NextGen Gallery ID.
		$nextgen_id = explode( '-', $id );
		$id         = $nextgen_id[1];

		// Get NextGen Gallery Objects.
		$nggdb   = new nggdb();
		$objects = apply_filters( 'envira_gallery_dynamic_get_nextgen_image_data', $nggdb->get_gallery( $id ), $id );

		// Return if no objects found.
		if ( ! $objects ) {
			return false;
		}

		// Build gallery.
		foreach ( (array) $objects as $key => $object ) {
			// Depending on the NextGEN version, the structure of the object will vary.
			if ( ! isset( $object->_ngiw ) ) {
				// Get path for gallery.
				if ( ! isset( $nextgen_gallery_path ) ) {
					global $wpdb;
					// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
					$nextgen_gallery_path = $wpdb->get_row( $wpdb->prepare( "SELECT path FROM $wpdb->nggallery WHERE gid = %d", $id ) );
				}

				$image     = $object->_orig_image;
				$image_url = get_bloginfo( 'url' ) . '/' . $nextgen_gallery_path->path . '/' . str_replace( ' ', '%20', $image->filename );
			} else {
				$image     = $object->_ngiw->_orig_image;
				$image_url = get_bloginfo( 'url' ) . '/' . $image->path . '/' . str_replace( ' ', '%20', $image->filename );
			}

			// Build image attributes to match Envira Gallery.
			$dynamic_data[ $image->pid ] = [
				'status'          => 'published',
				'src'             => $image_url,
				'title'           => ( isset( $image->alttext ) ? wp_strip_all_tags( esc_attr( $image->alttext ) ) : '' ),
				'link'            => $image_url,
				'alt'             => ( isset( $image->alttext ) ? wp_strip_all_tags( esc_attr( $image->alttext ) ) : '' ),
				'caption'         => ( isset( $image->description ) ? $image->description : '' ),
				'thumb'           => '',
				'link_new_window' => 0,
			];

		}

		return apply_filters( 'envira_gallery_dynamic_nextgen_images', $dynamic_data, $objects, $id, $data );
	}

	/**
	 * Retrieves the image data for a given folder inside the wp-content folder
	 *
	 * @param array  $dynamic_data    Existing Dynamic Data Array.
	 * @param string $folder         Directory Name.
	 * @param array  $data            Gallery Configuration.
	 * @return bool|array            Array of data on success, false on failure
	 */
	public function get_folder_images( $dynamic_data, $folder, $data ) {
		// Get folder.
		$folder_parts = explode( '-', $folder );

		// Remove the first element, which should be 'folder', then put the array back together.
		unset( $folder_parts[0] );
		if ( empty( $folder_parts ) ) {
			return false;
		}
		$folder = implode( '-', $folder_parts );

		// Check directory exists.
		$upload_dir  = wp_upload_dir();
		$content_dir = dirname( $upload_dir['basedir'] );
		$folder_path = trailingslashit( $content_dir ) . $folder;
		$folder_url  = trailingslashit( dirname( $upload_dir['baseurl'] ) ) . $folder;

		if ( ! file_exists( $folder_path ) ) {
			// TO-DO -> Return A Message.
			return false;
		}

		$files = [];
		// Open a directory, and read its contents.
		if ( is_dir( $folder_path ) ) {
			$dh = opendir( $folder_path );
			if ( $dh ) {
				$file = readdir( $dh );
				while ( false !== $file ) {
					$file_path = $folder_path . '/' . $file;
					if ( ! is_dir( $file_path ) ) {
						$files[ $file ] = [
							'size'  => filesize( $file_path ),
							'mtime' => filemtime( $file_path ),
						];
					}
					$file = readdir( $dh );
				}
				closedir( $dh );
			}
		}

		ksort( $files );

		if ( count( $files ) === 0 ) {
			return false;
		}

		// Get all images from $files.
		$images = [];
		foreach ( $files as $file_name => $file_info ) {
			$file_parts = pathinfo( $file_name );
			if ( isset( $file_parts['extension'] ) ) {
				$is_lightbox_thumb = substr( $file_parts['filename'], -2 ) === '_c';
				$correct_extension = in_array( $file_parts['extension'], [ 'jpg', 'jpeg', 'png', 'gif', 'JPG', 'JPEG', 'PNG', 'GIF' ], true );

				if ( ! $correct_extension || $is_lightbox_thumb ) {
					continue;
				}

				$images[ $file_name ] = $file_info;
			}
		}

		// Check we have at least one image.
		if ( count( $images ) === 0 ) {
			return false;
		}

		if ( ( isset( $data['config']['random'] ) && 'date' === $data['config']['random'] ) || ( isset( $data['config']['sort'] ) && 'date' === $data['config']['sort'] ) ) {
			/** A ' sort="date ' was passed in so we attempt to sort by filedate. */

			/**
			 * Sort Order.
			 *
			 * @param array  $a First value.
			 * @param string $b Second value.
			 */
			function images_sort_by_order_desc( $a, $b ) {
				return $a['mtime'] - $b['mtime'];
			}
			/**
			 * Sort Order.
			 *
			 * @param array  $a First value.
			 * @param string $b Second value.
			 */
			function images_sort_by_order_asc( $a, $b ) {
				return $b['mtime'] - $a['mtime'];
			}
			empty( $data['config']['sorting_direction'] ) || 'ASC' === $data['config']['sorting_direction'] ? uasort( $images, 'images_sort_by_order_asc' ) : uasort( $images, 'images_sort_by_order_desc' );
		}

		if ( ! empty( $data['config']['limit'] ) && intval( $data['config']['limit'] ) > 0 ) {
			$images = array_slice( $images, 0, intval( $data['config']['limit'] ) );
		}

		$counter = 0;

		// Build gallery.
		foreach ( (array) $images as $image_filename => $image_info ) {

			// Get file path and URL.
			$file_path = $folder_path . '/' . $image_filename;
			$file_url  = $folder_url . '/' . $image_filename;

			// Get file info.
			$info = pathinfo( $folder_path . '/' . $image_filename );
			if ( ! isset( $file_parts['extension'] ) ) {
				continue;
			}
			$ext  = $info['extension'];
			$name = wp_basename( $file_path, ".$ext" );

			// If the current file we are on is a resized file, don't include it in the gallery
			// Check for different retina image sizes, which are x2
			// Gallery.
			$suffix        = '-' . envira_get_config( 'crop_width', $data ) . 'x' . envira_get_config( 'crop_height', $data ) . ( envira_get_config( 'crop', $data ) ? '_c ' : '' ) . '.' . $ext;
			$suffix_retina = '-' . envira_get_config( 'crop_width', $data ) * 2 . 'x' . envira_get_config( 'crop_height', $data ) * 2 . ( envira_get_config( 'crop', $data ) ? '_c ' : '' ) . '.' . $ext;
			if ( strpos( $image_filename, $suffix ) !== false || strpos( $image_filename, $suffix_retina ) !== false ) {
				continue;
			}

			// Mobile.
			$suffix        = '-' . envira_get_config( 'mobile_width', $data ) . 'x' . envira_get_config( 'mobile_height', $data ) . ( envira_get_config( 'crop', $data ) ? '_c ' : '' ) . '.' . $ext;
			$suffix_retina = '-' . envira_get_config( 'mobile_width', $data ) * 2 . 'x' . envira_get_config( 'mobile_height', $data ) * 2 . ( envira_get_config( 'crop', $data ) ? '_c ' : '' ) . '.' . $ext;
			if ( strpos( $image_filename, $suffix ) !== false || strpos( $image_filename, $suffix_retina ) !== false ) {
				continue;
			}

			// Lightbox Thumbnails.
			$thumbnails_width  = envira_get_config( 'thumbnails_width', $data );
			$thumbnails_height = envira_get_config( 'thumbnails_height', $data );
			$suffix            = '-' . $thumbnails_width . 'x' . envira_get_config( 'thumbnails_height', $data ) . '_c.' . $ext;
			$suffix_retina     = '-' . ( 'auto' === $thumbnails_width ? 'auto' : $thumbnails_width * 2 ) . 'x' . ( 'auto' === $thumbnails_height ? 'auto' : $thumbnails_height * 2 ) . '_c.' . $ext;
			if ( strpos( $image_filename, $suffix ) !== false || strpos( $image_filename, $suffix_retina ) !== false ) {
				continue;
			}

			// Generate width and height
			// $size = getimagesize( $file_path );.
			$dynamic_data[ $counter . '_folder' ] = [
				'status'          => 'published',
				'src'             => $file_url,
				'title'           => $info['filename'],
				'link'            => $file_url,
				'alt'             => $info['filename'],
				'caption'         => '',
				'thumb'           => '',
				'link_new_window' => 0,
			];

			++$counter;
		}

		return apply_filters( 'envira_gallery_dynamic_folder_images', $dynamic_data, $files, $folder, $data );
	}

	/**
	 * Retrieves the image data for images attached to the given Post/Page/CPT ID
	 *
	 * @param int    $id            Post/Page/CPT ID.
	 * @param array  $exclude        What to exclude.
	 * @param string $fields     Fields to return.
	 * @return bool|array        Array of data on success, false on failure
	 */
	private function get_attached_images( $id, $exclude, $fields = 'ids' ) {

		// Prepare query args.
		$args = [
			'orderby'        => 'menu_order',
			'order'          => 'ASC',
			'post_type'      => 'attachment',
			'post_parent'    => $id,
			'post_mime_type' => 'image',
			'post_status'    => null,
			'posts_per_page' => -1,
			'fields'         => $fields,
		];

		// Add images to exclude if necessary.
		if ( $exclude ) {
			$args['post__not_in'] = (array) explode( ',', $exclude );
		}

		// Allow args to be filtered and then query the images.
		$args   = apply_filters( 'envira_gallery_dynamic_attached_image_args', $args, $id, $fields, $exclude );
		$images = get_posts( $args );

		// If no images are found, return false.
		if ( ! $images ) {
			return false;
		}

		return apply_filters( 'envira_gallery_dynamic_attached_images', $images, $id, $exclude, $fields );
	}

	/**
	 * Overrides the default WordPress Gallery with an Envira Gallery
	 *
	 * @since 1.0.0
	 *
	 * @param string $html HTML.
	 * @param array  $atts Attributes.
	 * @return string HTML
	 */
	public function override_gallery( $html, $atts ) {

		// Read the native gallery override config value from the Dynamic Gallery, assuming it's disabled.
		$dynamic_gallery_override = false;
		$dynamic_gallery_id       = Envira_Dynamic_Common::get_instance()->get_gallery_dynamic_id();
		$data                     = Envira_Gallery::get_instance()->get_gallery( $dynamic_gallery_id );
		$dynamic_gallery_override = Envira_Gallery_Shortcode::get_instance()->get_config( 'native_gallery_override', $data );

		// If there is no envira="true" argument in the shortcode, and the Dynamic Gallery's override setting is disabled,
		// just return the default output.
		if ( ! isset( $atts['envira'] ) && ! $dynamic_gallery_override && ! apply_filters( 'envira_gallery_dynamic_pre_gallery', false ) ) {
			return $html;
		}

		// If here, we need to replace [gallery] with Envira Gallery.
		// Declare a static incremental to ensure unique IDs when multiple galleries are called.
		global $post;
		static $dynamic_i = 0;

		$parsed_args = []; // was not declared before code cleanup.

		// Either grab custom images or images attached to the post.
		$images = false;
		if ( ! empty( $atts['ids'] ) ) {
			$images = $atts['ids'];
		} else {
			if ( empty( $post->ID ) ) {
				return $html;
			}

			$exclude = ! empty( $atts['exclude'] ) ? $atts['exclude'] : false;
			$images  = $this->get_attached_images( $post->ID, $exclude );
		}

		// If no images have been found, return the default HTML.
		if ( ! $images ) {
			return $html;
		}

		// Set the shortcode atts to be passed into shortcode regardless.
		$args           = [];
		$args['images'] = implode( ',', (array) $images );
		// $args['link']   = ! empty( $atts['link'] ) ? $atts['link'] : 'none';
		// Check if the envira_args attribute is set and parse the query string provided.
		if ( ! empty( $atts['envira_gallery_args'] ) ) {
			wp_parse_str( html_entity_decode( $atts['envira_gallery_args'] ), $parsed_args );
			$args = array_merge( $parsed_args, $args );
			$args = apply_filters( 'envira_gallery_dynamic_gallery_args', $args, $atts, $dynamic_i );
		}

		// Prepare the args to be output into query string shortcode format for the shortcode.
		$output_args = '';
		foreach ( $args as $k => $v ) {
			$output_args .= $k . '=' . $v . ' ';
		}

		// Increment the static counter.
		++$dynamic_i;

		// Map to the new Envira shortcode with the proper data structure.
		$result = do_shortcode( '[envira-gallery-dynamic id="custom-gallery-' . $dynamic_i . $post->ID . '" ' . trim( $output_args ) . ']' );

		return $result;
	}

	/**
	 * Overrides the default WordPress Gallery with an Envira Gallery
	 *
	 * @since 1.0.0
	 *
	 * @param boolean $should_cache HTML.
	 * @param array   $data Attributes.
	 * @return string HTML
	 */
	public function prevent_cache( $should_cache, $data ) {

		if ( ! empty( $data['gallery_id'] ) && 'folder_dynamic' === $data['gallery_id'] ) {
			return false;
		}

		return $should_cache;
	}

	/**
	 * Create the gallery shortcode
	 *
	 * @since 1.0.0
	 *
	 * @param array $atts Array of shortcode attributes.
	 */
	public function shortcode( $atts ) {

		// If no ID, return false.
		if ( empty( $atts['id'] ) ) {
			return false;
		}

		// Pull out the ID and remove from atts.
		$id = $atts['id'];
		unset( $atts['id'] );

		// Prepare the args to be output into query string shortcode format for the shortcode.
		$output_args = '';
		foreach ( $atts as $k => $v ) {
			$output_args .= $k . '=' . $v . ' ';
		}

		// Map to the Envira Gallery shortcode with the proper data structure.
		return do_shortcode( '[envira-gallery dynamic="' . $id . '" ' . trim( $output_args ) . ']' );
	}

	/**
	 * Returns the singleton instance of the class.
	 *
	 * @since 1.0.0
	 *
	 * @return object The Envira_Dynamic_Gallery_Shortcode object.
	 */
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Envira_Dynamic_Gallery_Shortcode ) ) {
			self::$instance = new Envira_Dynamic_Gallery_Shortcode();
		}

		return self::$instance;
	}
}

// Load the shortcode class.
$envira_dynamic_gallery_shortcode = Envira_Dynamic_Gallery_Shortcode::get_instance();
