<?php
/**
 * Capabilities class.
 *
 * @since 1.3.0
 *
 * @package Envira_Albums
 * @author  Envira Team
 */

namespace Envira\Albums\Utils;

use Envira\Admin\Permissions;

/**
 * Album Capabilities
 */
class Capabilities {

	/**
	 * Primary class constructor.
	 *
	 * @since 1.3.0
	 */
	public function __construct() {

		// Actions.
		add_action( 'admin_init', [ $this, 'add_capabilities' ] );

		add_filter( 'envira_permissions_fields_mapping', [ $this, 'envira_permissions_fields_mapping' ], 10, 1 );
		add_filter( 'envira_permissions_labels', [ $this, 'envira_permissions_labels' ], 10, 1 );

		add_action( 'envira_permissions_mapper', [ $this, 'map_permissions' ] );
		add_filter( 'envira_permissions_map', [ $this, 'envira_albums_permissions' ], 10, 1 );

		// Set default permissions.
		$this->set_default_permissions();
	}

	/**
	 * Localize the permissions.
	 *
	 * @since 1.9.11
	 *
	 * @param array $permissions Permissions.
	 */
	public function envira_permissions_fields_mapping( array $permissions ): array {
		$map = [];
		$map['envira_permissions_edit_others_albums'] = [ 'envira_permissions_edit_others_albums' ];
		$map['envira_permissions_edit_albums']        = [ 'envira_permissions_create_albums', 'envira_permissions_edit_albums' ];
		$map['envira_permissions_delete_albums']      = [ 'envira_permissions_create_albums', 'envira_permissions_edit_albums', 'envira_permissions_delete_albums' ];
		$map['envira_permissions_create_albums']      = [ 'envira_permissions_create_albums' ];

		return array_merge( $permissions, $map );
	}

	/**
	 * Localize the permissions labels.
	 *
	 * @since 1.9.11
	 *
	 * @param array $labels Labels.
	 */
	public function envira_permissions_labels( array $labels ): array {

		$albums_labels = [
			'envira_permissions_create_albums'      => esc_html__( 'Create Albums' ),
			'envira_permissions_edit_albums'        => esc_html__( 'Edit Albums' ),
			'envira_permissions_delete_albums'      => esc_html__( 'Delete Albums' ),
			'envira_permissions_edit_others_albums' => esc_html__( 'Edit Others Albums' ),
		];

		return array_merge( $labels, $albums_labels );
	}

	/**
	 * Add the Envira Album capabilities to the roles.
	 *
	 * @param array $permissions Permissions.
	 * @since 1.9.11
	 */
	public function envira_albums_permissions( array $permissions ): array {

		$albums_permissions = [
			'envira_permissions_create_albums'      => [ 'create_envira_albums', 'publish_envira_albums', 'read_envira_album' ],
			'envira_permissions_edit_albums'        => [ 'edit_envira_album', 'edit_envira_albums', 'edit_published_envira_albums' ],
			'envira_permissions_delete_albums'      => [ 'delete_envira_album', 'delete_envira_albums', 'delete_published_envira_albums' ],
			'envira_permissions_edit_others_albums' => [
				'edit_other_envira_albums',
				'edit_others_envira_albums',
				'edit_private_envira_albums',
				'read_private_envira_albums',
			],
		];

		return array_merge( $permissions, $albums_permissions );
	}

	/**
	 * Map the permissions and capabilities for user roles.
	 *
	 * @since 1.9.11
	 */
	public function map_permissions() {
		// Map the permissions and capabilities and user roles.

		$permissions = new Permissions();

		$permissions_map = $permissions->map_permissions();

		$albums_permissions = $this->envira_albums_permissions( $permissions_map );

		$permissions->envira_permissions_update( $albums_permissions );
	}

	/**
	 * Set the default album permissions for user roles.
	 *
	 * @since 1.9.11
	 */
	public function set_default_permissions() {

		$has_default_permissions = get_option( 'envira_permissions_albums_default', false );

		if ( $has_default_permissions ) {
			return;
		}

		if ( ! class_exists( 'Envira\Admin\Permissions' ) ) {
			return;
		}

		$permissions = new Permissions();

		$all    = [ 'administrator', 'editor', 'author' ];
		$editor = [ 'administrator', 'editor' ];

		$capabilities = [
			'envira_permissions_create_albums'      => $all,
			'envira_permissions_edit_albums'        => $all,
			'envira_permissions_delete_albums'      => $all,
			'envira_permissions_edit_others_albums' => $editor,
		];

		foreach ( $capabilities as $capability => $roles ) {
			update_option( $capability, maybe_serialize( $roles ) );
			$permissions->add_capabilities_by_permissions( $capability, $roles );
		}

		update_option( 'envira_permissions_albums_default', 'yes' );
	}

	/**
	 * Registers Envira Album capabilities for each Role, if they don't already exist.
	 *
	 * If capabilities don't exist, they're copied from Posts. This ensures users prior to 1.3.0
	 * get like-for-like behaviour in Envira and don't notice the new capabilities.
	 *
	 * @since 1.3.0
	 */
	public function add_capabilities() {

		// Bail out if it is settings page update action. - Ignore nonce check as we don't use post data here.
		// phpcs:ignore WordPress.Security.NonceVerification
		if ( isset( $_POST['envira-permissions-nonce'] ) && sanitize_text_field( wp_unslash( $_POST['envira-permissions-nonce'] ) ) !== null ) {
			return;
		}

		// Grab the administrator role, and if it already has an Envira capability key defined, bail
		// as we only need to register our capabilities once.
		$administrator = get_role( 'administrator' );
		if ( $administrator->has_cap( 'edit_other_envira_albums' ) ) {
			return;
		}

		// If here, we need to assign capabilities
		// Define the roles we want to assign capabilities to.
		$roles = [
			'administrator',
			'editor',
			'author',
			'contributor',
			'subscriber',
		];

		// Iterate through roles.
		foreach ( $roles as $role_name ) {
			// Properly get the role as WP_Role object.
			$role = get_role( $role_name );
			if ( ! is_object( $role ) ) {
				continue;
			}

			// Map this Role's Post capabilities to our Envira Album capabilities.
			$caps = [
				'edit_envira_album'              => $role->has_cap( 'edit_posts' ),
				'read_envira_album'              => $role->has_cap( 'read' ),
				'delete_envira_album'            => $role->has_cap( 'delete_posts' ),

				'edit_envira_albums'             => $role->has_cap( 'edit_posts' ),
				'edit_other_envira_albums'       => $role->has_cap( 'edit_others_posts' ),
				'edit_others_envira_albums'      => $role->has_cap( 'edit_others_posts' ),
				'publish_envira_albums'          => $role->has_cap( 'publish_posts' ),
				'read_private_envira_albums'     => $role->has_cap( 'read_private_posts' ),

				'read'                           => $role->has_cap( 'read' ),
				'delete_envira_albums'           => $role->has_cap( 'delete_posts' ),
				'delete_private_envira_albums'   => $role->has_cap( 'delete_private_posts' ),
				'delete_published_envira_albums' => $role->has_cap( 'delete_published_posts' ),
				'delete_others_envira_albums'    => $role->has_cap( 'delete_others_posts' ),
				'edit_private_envira_albums'     => $role->has_cap( 'edit_private_posts' ),
				'edit_published_envira_albums'   => $role->has_cap( 'edit_published_posts' ),
				'create_envira_albums'           => $role->has_cap( 'edit_posts' ),
			];

			// Add the above Envira capabilities to this Role.
			foreach ( $caps as $envira_cap => $value ) {
				// Don't add if value is false.
				if ( ! $value ) {
					continue;
				}

				$role->add_cap( $envira_cap );
			}
		}
	}
}
