<?php
/**
 * Settings admin class.
 *
 * @since 1.3.0
 *
 * @package Envira_Albums
 * @author  Envira Gallery Team <support@enviragallery.com>
 */

namespace Envira\Albums\Admin;

use Envira\Admin\Permissions;

/**
 * Settings admin class.
 *
 * @since 1.3.0
 *
 * @package Envira_Albums
 * @author  Envira Gallery Team <support@enviragallery.com>
 */
class Settings {

	/**
	 * Primary class constructor.
	 *
	 * @since 1.3.0
	 */
	public function __construct() {

		// NextGEN Importer Addon Support.
		add_filter( 'envira_nextgen_importer_settings_tab_nav', [ $this, 'nextgen_settings_register_tabs' ] );
		add_action( 'envira_nextgen_importer_tab_settings_albums', [ $this, 'nextgen_settings_tab' ] );
		add_filter( 'envira_permissions_fields', [ $this, 'envira_permissions_albums_fields' ], 10, 1 );
		add_filter( 'envira_permissions_default_roles', [ $this, 'envira_permissions_albums_default_roles' ], 10, 1 );
		add_action( 'envira_permissions_addons_tab', [ $this, 'envira_albums_permissions_tab' ] );
	}

	/**
	 * Adds an Albums Tab to the NextGEN Importer Settings Screen
	 *
	 * @since 1.3.0
	 *
	 * @param   array $tabs   Tabs.
	 * @return  array           Tabs
	 */
	public function nextgen_settings_register_tabs( $tabs ) {

		$tabs['albums'] = __( 'Albums', 'envira-nextgen-importer' );
		return $tabs;
	}

	/**
	 * Callback for displaying the UI for the Albums Settings tab in the NextGEN Importer.
	 *
	 * @since 1.3.0
	 */
	public function nextgen_settings_tab() {

		// Check and see if NextGEN is installed... if not, do not attempt to display settings and instead report an error.
		if ( ! is_plugin_active( 'nextgen-gallery/nggallery.php' ) ) { ?>
			<div id="envira-nextgen-importer-settings-galleries">
				<p>Please install and activate the <a href="https://wordpress.org/plugins/nextgen-gallery/" target="_blank" rel="noopener">NextGEN Gallery plugin</a> before using this addon.</p>
			</div>
			<?php
			return;
		}

		// Get NextGEN Albums.
		$albums = \Envira_Nextgen_Wrapper::get_instance()->get_albums();

		// Get settings (contains imported albums).
		$settings = get_option( 'envira_nextgen_importer' );
		?>

		<!-- Progress Bar -->
		<div id="album-progress"><div id="album-progress-label"></div></div>

		<div id="envira-nextgen-importer-settings-albums">
			<form id="envira-nextgen-importer-albums" method="post">
				<table class="form-table">
					<tbody>
						<tr id="envira-settings-albums-key-box">
							<th scope="row">
								<label for="envira-settings-albums-key"><?php esc_html_e( 'Albums to Import', 'envira-nextgen-importer' ); ?></label>
							</th>
							<td>
								<?php
								if ( false !== $albums ) {
									foreach ( $albums as $album ) {
										// Check if album imported from NextGEN previously.
										$imported = ( ( isset( $settings['albums'] ) && isset( $settings['albums'][ $album->id ] ) ) ? true : false );
										?>
										<label for="albums-<?php echo esc_attr( $album->id ); ?>" data-id="<?php echo esc_attr( $album->id ); ?>"<?php echo ( $imported ? ' class="imported"' : '' ); ?>>
											<input type="checkbox" name="albums" id="albums-<?php echo esc_attr( $album->id ); ?>" value="<?php echo esc_attr( $album->id ); ?>" />
											<?php echo esc_html( $album->name ); ?>
											<span>
												<?php
												if ( $imported ) {
													// Already imported.
													esc_html_e( 'Imported', 'envira-nextgen-importer' );
												}
												?>
											</span>
										</label>
										<?php
									}
								}
								?>
							</td>
						</tr>
						<tr>
							<th scope="row">
								&nbsp;
							</th>
							<td>
								<?php
								submit_button( __( 'Import Albums', 'envira-nextgen-importer' ), 'primary', 'envira-gallery-verify-submit', false );
								?>
							</td>
						</tr>
						<?php do_action( 'envira_nextgen_importer_settings_albums_box' ); ?>
					</tbody>
				</table>
			</form>
		</div>
		<?php
	}

	/**
	 * Add the default roles for albums.
	 *
	 * @since 1.7.11
	 *
	 * @param array $roles The roles.
	 *
	 * @return array
	 */
	public function envira_permissions_albums_default_roles( array $roles ): array {

		$all    = [ 'administrator', 'editor', 'author' ];
		$editor = [ 'administrator', 'editor' ];

		$permissions = [
			'envira_permissions_create_albums'      => $all,
			'envira_permissions_edit_albums'        => $all,
			'envira_permissions_delete_albums'      => $all,
			'envira_permissions_edit_others_albums' => $editor,
		];

		return array_merge( $roles, $permissions );
	}

	/**
	 * Add the albums fields to the permissions fields.
	 *
	 * @since 1.7.11
	 *
	 * @param array $fields The fields.
	 *
	 * @return array
	 */
	public function envira_permissions_albums_fields( array $fields ): array {
		$albums_fields = [ 'envira_permissions_create_albums', 'envira_permissions_edit_albums', 'envira_permissions_delete_albums', 'envira_permissions_edit_others_albums' ];
		return array_merge( $fields, $albums_fields );
	}

	/**
	 * Add the albums fields to the permissions fields.
	 *
	 * @since 1.7.11
	 *
	 * @return void
	 */
	public function envira_albums_permissions_tab() {
		$permissions = new \Envira\Admin\Permissions();
		?>
		<tr>
			<td colspan="2">
				<h2><?php esc_html_e( 'Albums Permissions', 'envira-gallery' ); ?></h2>
			</td>
		</tr>
		<tr id="envira-settings-albums-permissions-create" class="envira-settings-roles-select">
						<th scope="row">
							<label for="envira_permissions_create"><?php esc_html_e( 'Create Albums', 'envira-gallery' ); ?></label>
</th>
<td>
		<?php $permissions->render_permissions_field( 'envira_permissions_create_albums' ); ?>
	<p class="description"><?php esc_html_e( 'Users that have at least one of these roles will be able to create albums.', 'envira-gallery' ); ?></p>
</td>
</tr>
<tr id="envira-settings-albums-permissions-edit" class="envira-settings-roles-select">
	<th scope="row">
		<label for="envira_permissions_edit"><?php esc_html_e( 'Edit Albums', 'envira-gallery' ); ?></label>
	</th>
	<td>
		<?php $permissions->render_permissions_field( 'envira_permissions_edit_albums' ); ?>
		<p class="description"><?php esc_html_e( 'Users that have at least one of these roles will be able to edit/update albums.', 'envira-gallery' ); ?></p>
	</td>
</tr>
<tr id="envira-settings-albums-permissions-delete" class="envira-settings-roles-select">
	<th scope="row">
		<label for="envira-permissions-delete"><?php esc_html_e( 'Delete Albums', 'envira-gallery' ); ?></label>
	</th>
	<td>
		<?php $permissions->render_permissions_field( 'envira_permissions_delete_albums' ); ?>
		<p class="description"><?php esc_html_e( 'Users that have at least one of these roles will be able to delete albums.', 'envira-gallery' ); ?></p>
	</td>
</tr>
<tr id="envira-settings-albums-permissions-edit-others" class="envira-settings-roles-select">
	<th scope="row">
		<label for="envira_permissions_edit_others"><?php esc_html_e( 'Edit Other\'s Albums', 'envira-gallery' ); ?></label>
	</th>
	<td>
		<?php $permissions->render_permissions_field( 'envira_permissions_edit_others_albums' ); ?>
		<p class="description"><?php esc_html_e( 'Users that have at least one of these roles will be able to edit other\'s albums.', 'envira-gallery' ); ?></p>
	</td>
</tr>
		<?php
	}
}
