<?php

defined( 'ABSPATH' ) || exit;

/**
 * Compatibility with Custom Start Date for WCS.
 *
 * @class ASP_SSWS_Compat_CSD_For_WCS
 * @package Class
 */
class ASP_SSWS_Compat_CSD_For_WCS {

	/**
	 * Init ASP_SSWS_Compat_CSD_For_WCS.
	 */
	public static function init() {
		add_filter( 'asp_erws_get_reminders', __CLASS__ . '::add_reminder' );
		add_filter( 'asp_erws_get_reminder_intervals', __CLASS__ . '::add_reminder_interval' );
		add_filter( 'asp_erws_get_reminder_email_classes', __CLASS__ . '::add_reminder_email_class' );
		add_filter( 'asp_erws_get_reminder_hooks', __CLASS__ . '::add_reminder_hook' );
		add_filter( 'asp_erws_get_email_classes', __CLASS__ . '::add_email_class' );
		add_filter( 'asp_erws_before_reminders_settings_array', __CLASS__ . '::add_reminder_settings' );
		add_action( 'wscsd_start_date_changed', __CLASS__ . '::maybe_schedule_when_start_date_changed', 15, 2 );
		add_action( 'asp_erws_after_reminders_maybe_scheduled_for_status', __CLASS__ . '::maybe_schedule_for_status', 10, 2 );
		add_action( 'asp_erws_after_reminders_maybe_scheduled_for_date_type', __CLASS__ . '::maybe_schedule_for_date_type', 10, 2 );
		add_action( 'asp_erws_woocommerce_scheduled_subscription_csd_scheduled_activation_reminder', __CLASS__ . '::remind_scheduled_activation', 10, 2 );
	}

	/**
	 * Register new reminder.
	 * 
	 * @param array $reminders
	 * @return array
	 */
	public static function add_reminder( $reminders ) {
		$reminders[ 'csd_scheduled_activation' ] = __( 'Subscription Scheduled Activation Reminder', 'email-reminders-for-woocommerce-subscriptions' );
		return $reminders;
	}

	/**
	 * Gets the new reminder interval.
	 * 
	 * @param array $reminder_intervals
	 * @return array
	 */
	public static function add_reminder_interval( $reminder_intervals ) {
		$reminder_intervals[ 'csd_scheduled_activation' ] = get_option( 'erws_csd_scheduled_activation_reminder_days' );
		return $reminder_intervals;
	}

	/**
	 * Register new reminder email class.
	 * 
	 * @param array $reminder_classes
	 * @return array
	 */
	public static function add_reminder_email_class( $reminder_classes ) {
		$reminder_classes[ 'csd_scheduled_activation' ] = 'ASP_ERWS_Email_Customer_Subscription_CSD_Scheduled_Activation_Reminder';
		return $reminder_classes;
	}

	/**
	 * Register new reminder hook.
	 * 
	 * @param array $reminder_hooks
	 * @return array
	 */
	public static function add_reminder_hook( $reminder_hooks ) {
		$reminder_hooks[ 'asp_erws_woocommerce_scheduled_subscription_csd_scheduled_activation_reminder' ] = 'start';
		return $reminder_hooks;
	}

	/**
	 * Load new email class.
	 * 
	 * @param array $emails
	 */
	public static function add_email_class( $emails ) {
		$emails[] = 'ASP_ERWS_Email_Customer_Subscription_CSD_Scheduled_Activation_Reminder';
		return $emails;
	}

	/**
	 * Register new reminder settings.
	 * 
	 * @param array $settings
	 * @return array
	 */
	public static function add_reminder_settings( $settings ) {
		$settings = array(
			'name'        => __( 'Remind Scheduled Activation', 'email-reminders-for-woocommerce-subscriptions' ),
			'id'          => 'erws_csd_scheduled_activation_reminder_days',
			'type'        => 'text',
			'default'     => '',
			'placeholder' => __( 'e.g. 3,2,1', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc'        => __( 'day(s) before activation date.', 'email-reminders-for-woocommerce-subscriptions' ),
			'desc_tip'    => __( 'Specify the day(s) (e.g., 3, 2, 1) before the subscription activation date to notify customers, in descending order.', 'email-reminders-for-woocommerce-subscriptions' ),
		);

		return $settings;
	}

	/**
	 * Maybe schedule.
	 * 
	 * @param int $subscription_id
	 * @param int $start_date
	 */
	public static function maybe_schedule_when_start_date_changed( $subscription_id, $start_date ) {
		$subscription = wcs_get_subscription( $subscription_id );
		self::maybe_schedule_for_date_type( 'start', $subscription );
	}

	/**
	 * Maybe schedule.
	 * 
	 * @param string $new_status
	 * @param WC_Subscription $subscription
	 */
	public static function maybe_schedule_for_status( $new_status, $subscription ) {
		if ( 'scheduled' === $new_status ) {
			self::maybe_schedule_for_date_type( 'start', $subscription );
		}
	}

	/**
	 * Maybe schedule.
	 * 
	 * @param string $date_type
	 * @param WC_Subscription $subscription
	 */
	public static function maybe_schedule_for_date_type( $date_type, $subscription ) {
		if ( is_object( $subscription ) && $subscription->has_status( 'scheduled' ) && 'start' === $date_type ) {
			$timestamp   = $subscription->get_time( $date_type );
			$action_args = ASP_ERWS()->get_action_args( $date_type, $subscription );

			if ( ! $timestamp || ! as_next_scheduled_action( 'asp_erws_woocommerce_scheduled_subscription_csd_scheduled_activation_reminder', $action_args ) || as_next_scheduled_action( 'wscsd_scheduled_start', $action_args ) !== $timestamp ) {
				// Clear the respective schedules first.
				ASP_ERWS()->unschedule_all_actions_by_date_type( $date_type, $action_args );

				if ( $timestamp && ASP_ERWS()->can_schedule_reminder( $subscription, $date_type ) ) {
					ASP_ERWS()->schedule_reminders( $timestamp, 'asp_erws_woocommerce_scheduled_subscription_csd_scheduled_activation_reminder', get_option( 'erws_csd_scheduled_activation_reminder_days' ), $action_args );
				}
			}
		}
	}

	/**
	 * Remind the subscriber before the subscription is active.
	 * 
	 * @param int $subscription_id The ID of a 'shop_subscription' post
	 * @param int $remind_day
	 */
	public static function remind_scheduled_activation( $subscription_id, $remind_day ) {
		$subscription = wcs_get_subscription( $subscription_id );

		if ( $subscription && $subscription->has_status( 'scheduled' ) && ! empty( $subscription->get_time( 'start' ) ) ) {
			/**
			 * Remind scheduled activation.
			 * 
			 * @since version 2.3.1
			 */
			do_action( 'asp_erws_woocommerce_subscriptions_remind_csd_scheduled_activation', $subscription, $remind_day );
		}
	}
}
