<?php

defined( 'ABSPATH' ) || exit;

/**
 * Email Reminders for WooCommerce Subscriptions Email Preview Class
 */
class ASP_ERWS_Email_Preview {

	/**
	 * The email being previewed
	 *
	 * @var string
	 */
	private static $email_type;

	/**
	 * Constructor.
	 */
	public static function init() {
		add_filter( 'woocommerce_prepare_email_for_preview', __CLASS__ . '::prepare_email_for_preview' );
	}

	/**
	 * Prepare subscription email dummy data for preview.
	 *
	 * @param WC_Email $email The email object.
	 * @return WC_Email
	 */
	public static function prepare_email_for_preview( $email ) {
		self::$email_type = get_class( $email );

		if ( ! self::is_subscription_email() ) {
			return $email;
		}

		$dummy_subscription = self::get_dummy_subscription();

		switch ( self::$email_type ) {
			case 'ASP_ERWS_Email_Customer_Subscription_Trial_Ending_Reminder':
			case 'ASP_ERWS_Email_Customer_Subscription_Manual_Renewal_Reminder':
			case 'ASP_ERWS_Email_Customer_Subscription_Auto_Renewal_Reminder':
			case 'ASP_ERWS_Email_Customer_Subscription_Expiration_Reminder':
			case 'ASP_ERWS_Email_Customer_Subscription_After_Expiry_Reminder':
			case 'ASP_ERWS_Email_Customer_Subscription_CSD_Scheduled_Activation_Reminder':
				$email->set_object( $dummy_subscription );
				$email->remind_day         = 1;
				$email->custom_template_id = asp_erws_get_custom_template_for_subscription( $dummy_subscription, $email->get_reminder_slug(), $email->remind_day );
				break;
		}

		return $email;
	}

	/**
	 * Get a dummy subscription for use in preview emails.
	 *
	 * @return WC_Subscription
	 */
	private static function get_dummy_subscription() {
		$subscription = new WC_Subscription();
		$product      = self::get_dummy_product();

		$subscription->add_product( $product, 2 );
		$subscription->set_id( 12346 );
		$subscription->set_customer_id( 1 );
		$subscription->set_date_created( gmdate( 'Y-m-d H:i:s', strtotime( '-1 month' ) ) );
		$subscription->set_currency( 'USD' );
		$subscription->set_total( 100 );
		$subscription->set_billing_period( 'month' );
		$subscription->set_billing_interval( 1 );
		$subscription->set_start_date( gmdate( 'Y-m-d H:i:s', strtotime( '-1 month' ) ) );
		$subscription->set_trial_end_date( gmdate( 'Y-m-d H:i:s', strtotime( '+1 week' ) ) );
		$subscription->set_next_payment_date( gmdate( 'Y-m-d H:i:s', strtotime( '+1 week' ) ) );
		$subscription->set_end_date( gmdate( 'Y-m-d H:i:s', strtotime( '+1 month' ) ) );

		$address = self::get_dummy_address();

		$subscription->set_billing_address( $address );
		$subscription->set_shipping_address( $address );

		/**
		 * Filter the dummy subscription object used in email previews.
		 *
		 * @param WC_Subscription $subscription The dummy subscription object.
		 * @param string $email_type The email type being previewed.
		 * @since 2.4.0
		 */
		return apply_filters( 'asp_erws_email_preview_dummy_subscription', $subscription, self::$email_type );
	}

	/**
	 * Get a dummy product for use when previewing subscription emails.
	 *
	 * @return WC_Product
	 */
	private static function get_dummy_product() {
		$product = new WC_Product();
		$product->set_name( 'Dummy Product' );
		$product->set_price( 50 );

		/**
		 * Filter the dummy subscription product object used in email previews.
		 *
		 * @param WC_Product $product The dummy product object.
		 * @param string     $email_type The email type being previewed.
		 * @since 2.4.0
		 */
		return apply_filters( 'asp_erws_email_preview_dummy_product', $product, self::$email_type );
	}

	/**
	 * Get a dummy address used when previewing subscription emails.
	 *
	 * @return array
	 */
	private static function get_dummy_address() {
		$address = array(
			'first_name' => 'John',
			'last_name'  => 'Doe',
			'company'    => 'Company',
			'email'      => 'john@company.com',
			'phone'      => '555-555-5555',
			'address_1'  => '123 Fake Street',
			'city'       => 'Faketown',
			'postcode'   => '12345',
			'country'    => 'US',
			'state'      => 'CA',
		);

		/**
		 * Filter the dummy address used in email previews.
		 *
		 * @param array $address    The dummy address.
		 * @param string $email_type The email type being previewed.
		 * @since 2.4.0
		 */
		return apply_filters( 'asp_erws_email_preview_dummy_address', $address, self::$email_type );
	}

	/**
	 * Check if the email being previewed is a subscription email.
	 *
	 * Subscription emails include:
	 * - ASP_ERWS_Emails::get_email_classes() - core subscription emails.
	 *
	 * @return bool Whether the email being previewed is a subscription email.
	 */
	private static function is_subscription_email() {
		return in_array( self::$email_type, ASP_ERWS_Emails::get_email_classes(), true );
	}
}
